# backend/app/services/cimavet_sync_service.py
"""
Servicio de sincronización con API CIMAVet (medicamentos veterinarios).
Issue #354: Integración CIMAVet para completar cobertura farmacéutica.

Patrón similar a catalog_maintenance_service.py pero simplificado.
Aplica lecciones aprendidas de Issue #114 (CIMA resiliency).
"""

import os
from typing import Dict, Optional
from datetime import datetime, timezone

import httpx
import structlog
from sqlalchemy.orm import Session

from app.models.product_catalog_vet import ProductCatalogVet
from app.models.catalog_sync_history import CatalogSyncHistory, SyncStatus, SyncType, TriggerType
from app.utils.datetime_utils import utc_now
from app.services.cima_circuit_breaker import CimaSyncCircuitBreaker  # Reusar circuit breaker CIMA

logger = structlog.get_logger(__name__)

# Constantes de configuración
CIMAVET_API_TIMEOUT_SECONDS = 30.0  # Timeout conservador - CIMAVet tiene ~3,500 productos (más ligero que CIMA)
                                     # Tiempo esperado de respuesta: <10s para catálogo completo
                                     # Margen para variabilidad de red

BATCH_COMMIT_SIZE = 100  # Commits cada 100 productos
                         # Balancea memoria (~50KB/producto = 5MB/batch) con overhead transaccional
                         # Seguro para Render plan Starter (512MB RAM)

# Keywords para detectar productos que requieren receta veterinaria
PRESCRIPTION_KEYWORDS = ["sujeto a prescripción"]  # Case-insensitive matching
                                                    # Agregar variantes si API usa formulaciones alternativas

# Confidence score para productos CIMAVet (fuente oficial AEMPS)
CIMAVET_CONFIDENCE_SCORE = 90  # Alto (fuente oficial), pero menor que Nomenclator (95)
                               # CIMAVet puede tener actualizaciones retrasadas vs catálogo humano


class CIMAVetSyncService:
    """
    Servicio de sincronización con API CIMAVet de AEMPS.

    Características:
    - Sincronización cada 10 días (productos veterinarios cambian menos frecuentemente que humanos)
    - Circuit breaker reutilizado de CIMA (lecciones Issue #114)
    - ~3,500 productos veterinarios oficiales
    - Prefijo "VET-" obligatorio en national_code
    - Resiliente a fallos de API (retry con backoff exponencial)

    API CIMAVet:
    - Documentación: https://listadomedicamentos.aemps.gob.es/CIMAVet_REST_API.pdf
    - Similar a CIMA pero con campos específicos veterinarios
    """

    def __init__(self):
        self.circuit_breaker = CimaSyncCircuitBreaker()

        # API URL configurable vía environment variable
        # URL verificada mediante testing: https://cimavet.aemps.es/cimavet/rest
        # API solo soporta búsquedas individuales por CN, NO descarga masiva
        self.api_base_url = os.getenv(
            "CIMAVET_API_BASE_URL",
            "https://cimavet.aemps.es/cimavet/rest"
        )

        self.timeout = httpx.Timeout(CIMAVET_API_TIMEOUT_SECONDS)
        self.sync_interval_days = 10  # Sync cada 10 días

    async def enrich_from_cimavet(self, codigo_nacional: str) -> Optional[dict]:
        """
        Busca producto veterinario individual en API CIMAVet por código nacional.

        ESTRATEGIA: Sync Incremental (Sprint 3 - Issue #354)
        - La API CIMAVet NO soporta descarga masiva del catálogo
        - Solo permite búsquedas individuales por CN
        - Este método se usa para enriquecer productos vet on-demand

        Args:
            codigo_nacional: Código nacional del producto (sin prefijo VET-)

        Returns:
            Diccionario con datos parseados del producto o None si no se encuentra

        Raises:
            httpx.HTTPError: Si hay error de conectividad con la API
        """
        logger.info(f"[CIMAVET ENRICH] Buscando producto CN: {codigo_nacional}")

        # Verificar circuit breaker antes de intentar
        if not self.circuit_breaker.can_attempt():
            circuit_status = self.circuit_breaker.get_status()
            logger.warning(
                f"[CIMAVET ENRICH] Circuit breaker bloqueando intentos - "
                f"Estado: {circuit_status['state']}, "
                f"Fallos: {circuit_status.get('failure_count', 0)}"
            )
            return None

        async with httpx.AsyncClient(timeout=self.timeout) as client:
            try:
                # Endpoint presentaciones retorna datos más completos que medicamento
                url = f"{self.api_base_url}/presentaciones"
                params = {"cn": codigo_nacional}

                response = await client.get(url, params=params)

                # 204 No Content = Producto no encontrado (válido, no error)
                if response.status_code == 204:
                    logger.debug(f"[CIMAVET ENRICH] Producto CN {codigo_nacional} no encontrado (204)")
                    self.circuit_breaker.record_success()  # 204 es respuesta válida
                    return None

                response.raise_for_status()
                data = response.json()

                # Éxito - registrar en circuit breaker
                self.circuit_breaker.record_success()

                # API retorna formato paginado: {totalFilas, pagina, tamanioPagina, resultados: [...]}
                if isinstance(data, dict) and 'resultados' in data:
                    resultados = data.get('resultados', [])

                    if not resultados:
                        logger.debug(f"[CIMAVET ENRICH] CN {codigo_nacional} sin resultados")
                        return None

                    # Tomar primera presentación (puede haber múltiples por medicamento)
                    pres = resultados[0]

                    # Parsear presentación a formato interno
                    producto_parseado = self._parse_presentacion(pres, codigo_nacional)

                    logger.info(
                        f"[CIMAVET ENRICH] ✅ Encontrado CN {codigo_nacional}: "
                        f"{producto_parseado.get('nombre_comercial', 'N/A')[:60]}"
                    )

                    return producto_parseado

                # Formato no esperado
                logger.warning(f"[CIMAVET ENRICH] Formato inesperado para CN {codigo_nacional}: {list(data.keys())}")
                return None

            except httpx.HTTPStatusError as e:
                if e.response.status_code == 404:
                    logger.debug(f"[CIMAVET ENRICH] CN {codigo_nacional} no encontrado (404)")
                    self.circuit_breaker.record_success()  # 404 es respuesta válida
                    return None

                # Otros errores HTTP son fallos reales
                self.circuit_breaker.record_failure()
                logger.error(f"[CIMAVET ENRICH] Error HTTP {e.response.status_code} para CN {codigo_nacional}: {e}")
                raise
            except httpx.HTTPError as e:
                # Errores de conectividad son fallos
                self.circuit_breaker.record_failure()
                logger.error(f"[CIMAVET ENRICH] Error conectividad para CN {codigo_nacional}: {e}")
                raise
            except Exception as e:
                # Errores inesperados son fallos
                self.circuit_breaker.record_failure()
                logger.error(f"[CIMAVET ENRICH] Error inesperado para CN {codigo_nacional}: {type(e).__name__}: {e}")
                raise

    async def sync_cimavet_catalog(self, db: Session, force: bool = False, national_codes: Optional[list] = None) -> Dict:
        """
        Sincroniza catálogo CIMAVet desde API AEMPS.

        ESTRATEGIA: Sync Incremental (Sprint 3 - Issue #354)
        - La API CIMAVet NO permite descarga masiva del catálogo
        - Solo se pueden consultar productos individuales por CN
        - Este método requiere una lista de national_codes para sincronizar

        Args:
            db: Sesión de base de datos
            force: Forzar sincronización aunque no hayan pasado 10 días
            national_codes: Lista de códigos nacionales a sincronizar (REQUERIDO)
                           Si es None, retorna error explicando limitación de API

        Returns:
            Diccionario con estadísticas del proceso

        Ejemplo:
            # Sincronizar productos vet específicos
            codes = ['572240', '572239', '589610']  # SYNULOX, APOQUEL, etc
            result = await cimavet_sync_service.sync_cimavet_catalog(db, national_codes=codes)
        """
        if national_codes is None:
            logger.warning(
                "[CIMAVET SYNC] API CIMAVet no soporta descarga masiva. "
                "Proporcione lista de national_codes para sync incremental."
            )
            return {
                "status": "skipped",
                "reason": "api_limitation",
                "message": "API CIMAVet solo permite búsquedas individuales por CN. Use parámetro national_codes.",
            }

        logger.info(
            f"[CIMAVET SYNC] Iniciando sincronización incremental de {len(national_codes)} productos..."
        )

        # 1. Verificar última sincronización
        if not force:
            last_sync = (
                db.query(CatalogSyncHistory)
                .filter(CatalogSyncHistory.sync_type == SyncType.CIMAVET)
                .order_by(CatalogSyncHistory.started_at.desc())
                .first()
            )

            if last_sync and last_sync.completed_at:
                days_since_sync = (utc_now() - last_sync.completed_at).days
                if days_since_sync < self.sync_interval_days:
                    logger.info(
                        f"[CIMAVET SYNC] Última sync hace {days_since_sync} días. "
                        f"Saltando (intervalo: {self.sync_interval_days} días)..."
                    )
                    return {
                        "status": "skipped",
                        "reason": "recent_sync",
                        "days_since_last_sync": days_since_sync,
                    }

        # 2. Crear registro de sincronización
        from datetime import datetime
        start_time = datetime.now()

        sync_record = CatalogSyncHistory(
            sync_type=SyncType.CIMAVET,
            status=SyncStatus.IN_PROGRESS,
            sync_date=utc_now(),
            records_updated=0,
            duration_seconds=0.0,
            triggered_by=TriggerType.MANUAL if force else TriggerType.AUTOMATIC
        )
        db.add(sync_record)
        db.commit()

        stats = {"processed": 0, "created": 0, "updated": 0, "errors": 0, "not_found": 0, "products": []}

        try:
            # 3. Sync incremental: Consultar cada CN individualmente
            logger.info(f"[CIMAVET SYNC] Procesando {len(national_codes)} códigos nacionales...")

            for codigo_nacional in national_codes:
                try:
                    stats["processed"] += 1

                    # Consultar API CIMAVet para este CN específico
                    vet_data = await self.enrich_from_cimavet(codigo_nacional)

                    # Producto no encontrado en API (204 o sin resultados)
                    if not vet_data:
                        stats["not_found"] += 1
                        logger.debug(f"[CIMAVET SYNC] CN {codigo_nacional} no encontrado en API")
                        continue

                    # Prefijo VET- para evitar colisión con códigos uso humano
                    code_original = vet_data.get("codigo_nacional")
                    if not code_original:
                        logger.warning(f"[CIMAVET SYNC] Producto sin código nacional, saltando: {vet_data}")
                        stats["errors"] += 1
                        continue

                    vet_code = f"VET-{code_original}"

                    # Verificar si ya existe
                    existing = (
                        db.query(ProductCatalogVet).filter(ProductCatalogVet.national_code == vet_code).first()
                    )

                    if existing:
                        # Actualizar existente
                        self._update_vet_product(existing, vet_data)
                        stats["updated"] += 1
                        logger.debug(f"[CIMAVET SYNC] Actualizado: {vet_code}")
                    else:
                        # Crear nuevo
                        new_vet_product = self._create_vet_product(vet_code, vet_data)
                        db.add(new_vet_product)
                        stats["created"] += 1
                        logger.debug(f"[CIMAVET SYNC] Creado: {vet_code}")

                    # Batch commit para balance memoria/performance
                    if stats["processed"] % BATCH_COMMIT_SIZE == 0:
                        db.commit()
                        logger.info(
                            f"[CIMAVET SYNC] Progreso: {stats['processed']}/{len(national_codes)} - "
                            f"Creados: {stats['created']}, Actualizados: {stats['updated']}, "
                            f"Errores: {stats['errors']}, No encontrados: {stats['not_found']}"
                        )

                except Exception as e:
                    logger.error(f"[CIMAVET SYNC] Error procesando CN {codigo_nacional}: {str(e)}")
                    stats["errors"] += 1
                    continue

            # 5. Commit final
            db.commit()

            # 6. Actualizar registro de sincronización
            end_time = datetime.now()
            duration = (end_time - start_time).total_seconds()

            sync_record.status = SyncStatus.COMPLETED
            sync_record.records_updated = stats["created"] + stats["updated"]
            sync_record.duration_seconds = duration
            db.commit()

            logger.info(f"[CIMAVET SYNC] Completado exitosamente - {stats} - Duración: {duration:.2f}s")
            return {"status": "completed", "stats": stats}

        except Exception as e:
            # Error fatal - marcar sincronización como fallida
            logger.error(f"[CIMAVET SYNC] Error fatal durante sincronización: {str(e)}")

            end_time = datetime.now()
            duration = (end_time - start_time).total_seconds()

            sync_record.status = SyncStatus.FAILED
            sync_record.error_message = str(e)
            sync_record.duration_seconds = duration
            db.commit()

            return {"status": "failed", "reason": str(e), "stats": stats}

    async def _fetch_cimavet_api(self) -> list:
        """
        Consulta API CIMAVet y retorna lista de productos veterinarios.

        Filtros aplicados:
        - estado == "aut" (autorizados)
        - comerc == true (comercializados)

        Campos importantes:
        - presentacion.cn: Código nacional (múltiples presentaciones por producto)
        - dispensacion: Indica si requiere prescripción

        Returns:
            Lista de diccionarios con datos de presentaciones veterinarias (1 por CN)
        """
        logger.info("[CIMAVET API] Consultando API CIMAVet...")

        async with httpx.AsyncClient(timeout=self.timeout) as client:
            try:
                # Endpoint para obtener medicamentos veterinarios
                response = await client.get(f"{self.api_base_url}/medicamentos")
                response.raise_for_status()

                products_raw = response.json()
                logger.info(f"[CIMAVET API] Recibidos {len(products_raw)} productos de API")

                # Parsear y filtrar productos
                presentaciones = self._parse_cimavet_response(products_raw)
                logger.info(f"[CIMAVET API] Procesadas {len(presentaciones)} presentaciones válidas")

                return presentaciones

            except httpx.HTTPError as e:
                logger.error(f"[CIMAVET API] Error HTTP: {e}")
                raise
            except Exception as e:
                logger.error(f"[CIMAVET API] Error inesperado: {e}")
                raise

    def _parse_cimavet_response(self, products_raw: list) -> list:
        """
        Parsea respuesta de API CIMAVet y filtra productos válidos.

        Filtros:
        - estado == "aut" (autorizados)
        - comerc == true (comercializados)

        Un producto puede tener múltiples presentaciones, cada una con su CN.
        Retornamos 1 registro por presentación válida.

        Args:
            products_raw: Lista de productos de API CIMAVet

        Returns:
            Lista de presentaciones válidas (1 dict por CN)
        """
        presentaciones_validas = []

        for producto in products_raw:
            # Filtro 1: Solo productos autorizados
            if producto.get("estado") != "aut":
                continue

            # Filtro 2: Solo productos comercializados
            if not producto.get("comerc"):
                continue

            # Procesar presentaciones (múltiples CN por producto)
            presentaciones = producto.get("presentacion", [])
            if not isinstance(presentaciones, list):
                presentaciones = [presentaciones] if presentaciones else []

            for presentacion in presentaciones:
                cn = presentacion.get("cn")
                if not cn:
                    logger.warning(f"[CIMAVET PARSER] Presentación sin CN, saltando: {presentacion}")
                    continue

                # Determinar si requiere receta basado en dispensacion
                dispensacion = presentacion.get("dispensacion", "")
                requiere_receta = (
                    any(kw in dispensacion.lower() for kw in PRESCRIPTION_KEYWORDS)
                    if dispensacion else False
                )

                presentaciones_validas.append({
                    "codigo_nacional": cn,
                    "nombre_comercial": producto.get("nombre", ""),
                    "requiere_receta": requiere_receta,
                    "laboratorio_titular": producto.get("labtitular", ""),
                    "principios_activos": producto.get("principiosActivos", []),
                    "especies_destino": producto.get("especieDestino", []),
                    "forma_farmaceutica": presentacion.get("formaFarmaceutica", ""),
                    "via_administracion": presentacion.get("viaAdministracion", ""),
                    "pvp": presentacion.get("pvp"),
                    "estado_registro": producto.get("estado", ""),
                    "dispensacion": dispensacion,
                })

        return presentaciones_validas

    def _parse_presentacion(self, pres: dict, codigo_nacional: str) -> dict:
        """
        Parsea una presentación individual de la API CIMAVet (endpoint /presentaciones).

        Formato API /presentaciones?cn=XXX retorna:
        {
            "totalFilas": 1,
            "resultados": [{
                "nregistro": "848 ESP",
                "cn": "572240",
                "nombre": "SYNULOX 200/50 mg COMPRIMIDOS...",
                "principiosActivos": [{"id": 419, "nombre": "CLAVULANATO POTASIO", ...}],
                "labtitular": "Zoetis Spain S.L.",
                "dispensacion": {"id": 1, "nombre": "Sujeto a prescripción veterinaria"},
                "forma": {"id": 40, "nombre": "COMPRIMIDO"},
                "receta": false,
                ...
            }]
        }

        Args:
            pres: Diccionario con datos de presentación de API
            codigo_nacional: CN para validación

        Returns:
            Diccionario con campos normalizados para ProductCatalogVet
        """
        # Extraer dispensación (es un objeto con {id, nombre})
        dispensacion_obj = pres.get("dispensacion", {})
        dispensacion_nombre = ""
        dispensacion_id = None

        if isinstance(dispensacion_obj, dict):
            dispensacion_nombre = dispensacion_obj.get("nombre", "")
            dispensacion_id = dispensacion_obj.get("id")
        elif isinstance(dispensacion_obj, str):
            # Fallback si API retorna string en lugar de objeto
            dispensacion_nombre = dispensacion_obj

        # Clasificación basada en dispensacion.id (más confiable)
        # API CIMAVet retorna:
        #   id: 1 → "Sujeto a prescripción veterinaria" → RECETA
        #   id: 2 → "No sujeto a prescripción veterinaria" → LIBRE
        #
        # BUG FIX (Sprint 3 - Issue #354):
        # El campo 'receta' siempre retorna False
        # Las keywords "prescripción"/"veterinaria" aparecen en AMBOS textos
        # La única forma confiable es usar el ID de dispensación
        if dispensacion_id is not None:
            requiere_receta = (dispensacion_id == 1)
        else:
            # Fallback a keywords solo si no hay id (API inconsistente)
            # Buscar NEGACIÓN ("no sujeto", "no requiere") → LIBRE
            # Buscar AFIRMACIÓN ("sujeto", "requiere") sin "no" → RECETA
            requiere_receta = False  # Default seguro: libre
            if dispensacion_nombre:
                nombre_lower = dispensacion_nombre.lower()
                if "no sujeto" in nombre_lower or "no requiere" in nombre_lower:
                    requiere_receta = False
                elif "sujeto" in nombre_lower or "requiere" in nombre_lower:
                    requiere_receta = True

        # Parsear principios activos (array de objetos en endpoint /presentaciones)
        principios_activos = []
        pactivos_api = pres.get("principiosActivos", [])
        if pactivos_api:
            for pa in pactivos_api:
                principios_activos.append({
                    "nombre": pa.get("nombre", ""),
                    "cantidad": pa.get("cantidad"),
                    "unidad": pa.get("unidad", ""),
                })

        # Extraer forma farmacéutica
        forma_obj = pres.get("forma", {})
        forma_nombre = ""
        if isinstance(forma_obj, dict):
            forma_nombre = forma_obj.get("nombre", "")

        return {
            "codigo_nacional": codigo_nacional,
            "nombre_comercial": pres.get("nombre", ""),
            "requiere_receta": requiere_receta,
            "laboratorio_titular": pres.get("labtitular", ""),
            "principios_activos": principios_activos,
            "especies_destino": [],  # Endpoint presentaciones no incluye este campo
            "forma_farmaceutica": forma_nombre,
            "via_administracion": "",  # Endpoint presentaciones no incluye este campo directamente
            "pvp": pres.get("pvp"),
            "estado_registro": "aut",  # Asumimos autorizados (endpoint ya filtra)
            "dispensacion": dispensacion_nombre,
        }

    def _create_vet_product(self, vet_code: str, vet_data: dict) -> ProductCatalogVet:
        """
        Crea instancia de ProductCatalogVet desde datos de API.

        Args:
            vet_code: Código con prefijo VET-
            vet_data: Diccionario con datos parseados de API CIMAVet

        Returns:
            Instancia de ProductCatalogVet lista para insertar en BD
        """
        # Convertir principios activos de lista de dicts a lista de strings (solo nombres)
        principios_activos_list = vet_data.get("principios_activos", [])
        principios_activos_nombres = [
            pa.get("nombre") for pa in principios_activos_list if pa.get("nombre")
        ] if isinstance(principios_activos_list, list) else []

        return ProductCatalogVet(
            national_code=vet_code,
            vet_nombre_comercial=vet_data.get("nombre_comercial", "")[:500],
            vet_principios_activos=principios_activos_nombres,
            vet_laboratorio_titular=(
                vet_data.get("laboratorio_titular", "")[:200] if vet_data.get("laboratorio_titular") else None
            ),
            vet_forma_farmaceutica=(
                vet_data.get("forma_farmaceutica", "")[:200] if vet_data.get("forma_farmaceutica") else None
            ),
            vet_via_administracion=(
                vet_data.get("via_administracion", "")[:100] if vet_data.get("via_administracion") else None
            ),
            vet_especies_destino=vet_data.get("especies_destino", []),
            vet_condiciones_prescripcion=(
                vet_data.get("dispensacion", "")[:50] if vet_data.get("dispensacion") else None
            ),
            # CRÍTICO: Campo que determina prescription vs venta_libre para VETERINARIA
            # Basado en "sujeto a prescripción" en campo dispensacion
            vet_requiere_receta=vet_data.get("requiere_receta"),
            vet_pvp=vet_data.get("pvp"),
            vet_estado_registro=vet_data.get("estado_registro", "")[:50] if vet_data.get("estado_registro") else None,
            vet_numero_registro=None,  # No nos interesa según especificación
            data_source="CIMAVet",
            enrichment_confidence=CIMAVET_CONFIDENCE_SCORE,
            created_at=utc_now(),
            updated_at=utc_now(),
        )

    def _update_vet_product(self, existing: ProductCatalogVet, vet_data: dict):
        """
        Actualiza producto veterinario existente con nuevos datos de API.

        Args:
            existing: Producto existente en BD
            vet_data: Nuevos datos parseados de API CIMAVet
        """
        # Convertir principios activos de lista de dicts a lista de strings (solo nombres)
        principios_activos_list = vet_data.get("principios_activos", [])
        principios_activos_nombres = [
            pa.get("nombre") for pa in principios_activos_list if pa.get("nombre")
        ] if isinstance(principios_activos_list, list) else []

        existing.vet_nombre_comercial = vet_data.get("nombre_comercial", "")[:500]
        existing.vet_principios_activos = principios_activos_nombres
        existing.vet_laboratorio_titular = (
            vet_data.get("laboratorio_titular", "")[:200] if vet_data.get("laboratorio_titular") else None
        )
        existing.vet_forma_farmaceutica = (
            vet_data.get("forma_farmaceutica", "")[:200] if vet_data.get("forma_farmaceutica") else None
        )
        existing.vet_via_administracion = (
            vet_data.get("via_administracion", "")[:100] if vet_data.get("via_administracion") else None
        )
        existing.vet_especies_destino = vet_data.get("especies_destino", [])
        existing.vet_condiciones_prescripcion = (
            vet_data.get("dispensacion", "")[:50] if vet_data.get("dispensacion") else None
        )
        # CRÍTICO: Campo que determina prescription vs venta_libre
        existing.vet_requiere_receta = vet_data.get("requiere_receta")
        existing.vet_pvp = vet_data.get("pvp")
        existing.vet_estado_registro = vet_data.get("estado_registro", "")[:50] if vet_data.get("estado_registro") else None
        # vet_numero_registro no actualizado (no nos interesa)
        existing.updated_at = utc_now()


# Instancia global del servicio
cimavet_sync_service = CIMAVetSyncService()
