﻿"""
Estrategia de reintentos para sincronización CIMA.

Este módulo implementa backoff exponencial con jitter para evitar
el "thundering herd problem" cuando múltiples workers intentan
recuperarse simultáneamente.
"""

import random
from datetime import datetime, timedelta
from typing import Optional, Tuple

import structlog

from app.utils.datetime_utils import utc_now

logger = structlog.get_logger(__name__)

# Configuración de reintentos
MAX_RECOVERY_ATTEMPTS = 10  # Aumentado de 5 a 10
BASE_DELAY_SECONDS = 60
MAX_DELAY_SECONDS = 900  # 15 minutos
JITTER_PERCENTAGE = 0.2  # ±20%


def calculate_backoff(attempt: int) -> int:
    """
    Calcular delay con backoff exponencial y jitter.

    El jitter ayuda a evitar que múltiples procesos intenten
    recuperarse exactamente al mismo tiempo, distribuyendo
    la carga de manera más uniforme.

    Args:
        attempt: Número de intento (1-based)

    Returns:
        Delay en segundos con jitter aplicado
    """
    if attempt < 1:
        attempt = 1

    # Backoff exponencial: delay = base * 2^(attempt-1)
    base_delay = min(BASE_DELAY_SECONDS * (2 ** (attempt - 1)), MAX_DELAY_SECONDS)

    # Aplicar jitter: ±20% del delay base
    jitter_range = base_delay * JITTER_PERCENTAGE
    jitter = jitter_range * (random.random() - 0.5)  # Random entre -10% y +10%

    final_delay = int(base_delay + jitter)

    logger.debug(
        "cima.retry.backoff_calculated",
        event_type="retry_strategy",
        attempt=attempt,
        base_delay_seconds=base_delay,
        jitter_seconds=int(jitter),
        final_delay_seconds=final_delay,
    )

    return final_delay


def get_next_retry_time(attempt: int) -> datetime:
    """
    Calcular el próximo tiempo de reintento.

    Args:
        attempt: Número de intento (1-based)

    Returns:
        Datetime del próximo intento
    """
    delay_seconds = calculate_backoff(attempt)
    next_retry = utc_now() + timedelta(seconds=delay_seconds)

    logger.info(
        "cima.retry.next_attempt_scheduled",
        event_type="retry_strategy",
        attempt=attempt,
        delay_seconds=delay_seconds,
        next_retry_at=next_retry.isoformat(),
    )

    return next_retry


def should_retry(attempt: int) -> bool:
    """
    Determinar si se debe continuar reintentando.

    Args:
        attempt: Número de intento actual (1-based)

    Returns:
        True si se debe reintentar, False si se alcanzó el máximo
    """
    can_retry = attempt < MAX_RECOVERY_ATTEMPTS

    if not can_retry:
        logger.warning(
            "cima.retry.max_attempts_reached",
            event_type="retry_strategy",
            current_attempt=attempt,
            max_attempts=MAX_RECOVERY_ATTEMPTS,
        )

    return can_retry


def get_retry_context(attempt: int) -> dict:
    """
    Obtener contexto completo de reintento para logging.

    Args:
        attempt: Número de intento actual

    Returns:
        Diccionario con información del reintento
    """
    delay = calculate_backoff(attempt)
    next_retry = get_next_retry_time(attempt)
    can_retry = should_retry(attempt)

    return {
        "attempt": attempt,
        "max_attempts": MAX_RECOVERY_ATTEMPTS,
        "delay_seconds": delay,
        "next_retry_at": next_retry.isoformat(),
        "can_retry": can_retry,
        "attempts_remaining": MAX_RECOVERY_ATTEMPTS - attempt,
    }


def log_retry_attempt(attempt: int, reason: str, additional_context: Optional[dict] = None) -> None:
    """
    Registrar un intento de reintento con contexto completo.

    Args:
        attempt: Número de intento
        reason: Razón del reintento
        additional_context: Contexto adicional opcional
    """
    retry_context = get_retry_context(attempt)

    log_data = {"event_type": "retry_attempt", "reason": reason, **retry_context}

    if additional_context:
        log_data.update(additional_context)

    logger.info(f"cima.retry.attempt_{attempt}", **log_data)


def calculate_total_retry_time(max_attempts: int = MAX_RECOVERY_ATTEMPTS) -> Tuple[int, int]:
    """
    Calcular el tiempo total máximo de reintentos.

    Args:
        max_attempts: Número máximo de intentos

    Returns:
        Tupla (tiempo_minimo_segundos, tiempo_maximo_segundos)
    """
    # Sin jitter (tiempo mínimo)
    min_total = sum(min(BASE_DELAY_SECONDS * (2 ** (i - 1)), MAX_DELAY_SECONDS) for i in range(1, max_attempts))

    # Con jitter máximo (tiempo máximo)
    max_total = sum(
        int(min(BASE_DELAY_SECONDS * (2 ** (i - 1)), MAX_DELAY_SECONDS) * (1 + JITTER_PERCENTAGE / 2))
        for i in range(1, max_attempts)
    )

    logger.debug(
        "cima.retry.total_time_calculated",
        event_type="retry_strategy",
        max_attempts=max_attempts,
        min_total_seconds=min_total,
        max_total_seconds=max_total,
        min_total_hours=round(min_total / 3600, 2),
        max_total_hours=round(max_total / 3600, 2),
    )

    return min_total, max_total
