﻿"""
Circuit Breaker Pattern para sincronización CIMA.

Este módulo implementa un circuit breaker para proteger el sistema
de intentos repetidos de sincronización cuando el servicio CIMA está fallando.

Estados:
- CLOSED: Funcionamiento normal, permite intentos
- OPEN: Servicio fallando, bloquea intentos
- HALF_OPEN: Fase de recuperación, permite intentos limitados
"""

from datetime import datetime
from enum import Enum
from typing import Any, Dict, Optional

import structlog

from app.utils.datetime_utils import utc_now

logger = structlog.get_logger(__name__)


class CircuitState(Enum):
    """Estados posibles del circuit breaker."""

    CLOSED = "CLOSED"
    OPEN = "OPEN"
    HALF_OPEN = "HALF_OPEN"


class CimaSyncCircuitBreaker:
    """
    Circuit breaker para proteger el sistema de sincronizaciones fallidas repetidas.

    El circuit breaker previene llamadas repetidas a un servicio que está fallando,
    dándole tiempo para recuperarse antes de intentar nuevamente.

    **IMPORTANTE - LIMITACIÓN DE ESTADO IN-MEMORY**:

    Estado del Circuit Breaker:
    - **NO persistente**: El estado (CLOSED/OPEN/HALF_OPEN) se almacena solo en memoria
    - **NO sincronizado**: No se comparte entre workers en entornos multi-proceso
    - **Reinicio en cada restart**: El estado se resetea completamente al reiniciar el servicio
    - **No usa Redis/PostgreSQL**: No hay persistencia externa del estado del circuito

    Implicaciones en Producción (Render.com con múltiples workers):
    - Cada worker de FastAPI mantiene su PROPIO circuit breaker independiente
    - Un worker puede tener el circuito OPEN mientras otro tiene CLOSED
    - Los contadores de fallos (failure_count) no se acumulan entre workers
    - El umbral de fallos (failure_threshold) se aplica POR WORKER, no globalmente
    - Un reinicio del servicio resetea completamente el estado de protección

    Ejemplo de comportamiento en producción con 2 workers:
    ```
    Worker 1: 5 fallos → circuito OPEN → bloquea intentos
    Worker 2: 0 fallos → circuito CLOSED → permite intentos
    → El sistema sigue intentando sincronización vía Worker 2
    ```

    Recomendaciones para Producción:
    1. **Aceptar limitación actual**: Para workloads de bajo volumen, la protección
       por worker puede ser suficiente. Cada worker se protege independientemente.

    2. **Persistir estado en futuro (si es necesario)**:
       - Opción A: Redis con clave compartida para estado del circuito
         ```python
         redis.hset("circuit_breaker:cima", "state", state.value)
         redis.hincrby("circuit_breaker:cima", "failure_count", 1)
         ```
       - Opción B: Tabla PostgreSQL con row-level locking
         ```sql
         CREATE TABLE circuit_breaker_state (
             service VARCHAR PRIMARY KEY,
             state VARCHAR,
             failure_count INT,
             last_failure_time TIMESTAMP
         );
         ```
       - Opción C: Redis Pub/Sub para sincronización entre workers
         ```python
         redis.publish("circuit_breaker:state_change", json.dumps(state))
         ```

    3. **Monitoreo distribuido**: Agregar métricas que muestren el estado del
       circuito POR WORKER para detectar inconsistencias:
       ```python
       logger.info("circuit_state_per_worker", worker_id=os.getpid(), state=state)
       ```

    Para la mayoría de casos de uso de xFarma (sincronización CIMA diaria/semanal),
    la implementación in-memory actual es adecuada. Solo considerar persistencia
    si se detectan problemas de coordinación entre workers en producción.
    """

    def __init__(self, failure_threshold: int = 5, timeout_seconds: int = 300, half_open_max_attempts: int = 3):
        """
        Inicializar el circuit breaker.

        Args:
            failure_threshold: Número de fallos antes de abrir el circuito
            timeout_seconds: Tiempo en segundos antes de pasar a HALF_OPEN
            half_open_max_attempts: Intentos máximos en estado HALF_OPEN
        """
        self.state = CircuitState.CLOSED
        self.failure_count = 0
        self.success_count = 0
        self.failure_threshold = failure_threshold
        self.timeout_seconds = timeout_seconds
        self.half_open_max_attempts = half_open_max_attempts
        self.half_open_attempts = 0
        self.last_failure_time: Optional[datetime] = None
        self.last_success_time: Optional[datetime] = None
        self.state_changed_at: datetime = utc_now()

        logger.info(
            "cima.circuit_breaker.initialized",
            event_type="circuit_breaker",
            failure_threshold=failure_threshold,
            timeout_seconds=timeout_seconds,
        )

    def can_attempt(self) -> bool:
        """
        Verificar si se puede intentar una operación.

        Returns:
            True si se permite el intento, False en caso contrario
        """
        self._check_state_transition()

        if self.state == CircuitState.OPEN:
            logger.warning(
                "cima.circuit_breaker.attempt_blocked",
                event_type="circuit_breaker",
                state=self.state.value,
                time_until_half_open=self._time_until_half_open(),
            )
            return False

        if self.state == CircuitState.HALF_OPEN:
            if self.half_open_attempts >= self.half_open_max_attempts:
                logger.warning(
                    "cima.circuit_breaker.half_open_limit_reached",
                    event_type="circuit_breaker",
                    attempts=self.half_open_attempts,
                    max_attempts=self.half_open_max_attempts,
                )
                return False
            self.half_open_attempts += 1

        return True

    def record_success(self) -> None:
        """Registrar una operación exitosa."""
        self.last_success_time = utc_now()
        self.success_count += 1

        if self.state == CircuitState.HALF_OPEN:
            # En HALF_OPEN, un éxito cierra el circuito
            self._transition_to_closed()

            # Issue #114 - Fase 3: Logging estructurado con contexto temporal
            logger.info(
                "cima.circuit_breaker.recovery_successful",
                event_type="circuit_breaker",
                previous_state=CircuitState.HALF_OPEN.value,
                new_state=CircuitState.CLOSED.value,
                half_open_attempts=self.half_open_attempts,
                total_success_count=self.success_count,
                last_failure_time=self.last_failure_time.isoformat() if self.last_failure_time else None,
            )
        elif self.state == CircuitState.CLOSED:
            # Reset del contador de fallos en estado CLOSED
            self.failure_count = 0

        logger.debug(
            "cima.circuit_breaker.success_recorded",
            event_type="circuit_breaker",
            state=self.state.value,
            success_count=self.success_count,
        )

    def record_failure(self) -> None:
        """Registrar una operación fallida."""
        self.last_failure_time = utc_now()
        self.failure_count += 1

        if self.state == CircuitState.CLOSED:
            if self.failure_count >= self.failure_threshold:
                self._transition_to_open()

                # Issue #114 - Fase 3: Logging estructurado con contexto temporal completo
                logger.error(
                    "cima.circuit_breaker.opened",
                    event_type="circuit_breaker",
                    failure_count=self.failure_count,
                    threshold=self.failure_threshold,
                    timeout_seconds=self.timeout_seconds,
                    last_success_time=self.last_success_time.isoformat() if self.last_success_time else None,
                    state_changed_at=self.state_changed_at.isoformat(),
                )
        elif self.state == CircuitState.HALF_OPEN:
            # En HALF_OPEN, un fallo reabre el circuito
            self._transition_to_open()
            logger.warning(
                "cima.circuit_breaker.reopened",
                event_type="circuit_breaker",
                half_open_attempts=self.half_open_attempts,
            )

        logger.warning(
            "cima.circuit_breaker.failure_recorded",
            event_type="circuit_breaker",
            state=self.state.value,
            failure_count=self.failure_count,
        )

    def get_status(self) -> Dict[str, Any]:
        """
        Obtener el estado actual del circuit breaker.

        Returns:
            Diccionario con información del estado actual
        """
        self._check_state_transition()

        return {
            "state": self.state.value,
            "failure_count": self.failure_count,
            "success_count": self.success_count,
            "failure_threshold": self.failure_threshold,
            "last_failure_time": self.last_failure_time.isoformat() if self.last_failure_time else None,
            "last_success_time": self.last_success_time.isoformat() if self.last_success_time else None,
            "state_changed_at": self.state_changed_at.isoformat(),
            "time_in_current_state_seconds": (utc_now() - self.state_changed_at).total_seconds(),
            "can_attempt": self.can_attempt(),
            "time_until_half_open_seconds": self._time_until_half_open() if self.state == CircuitState.OPEN else None,
        }

    def reset(self) -> None:
        """Resetear el circuit breaker a su estado inicial."""
        self.state = CircuitState.CLOSED
        self.failure_count = 0
        self.success_count = 0
        self.half_open_attempts = 0
        self.last_failure_time = None
        self.last_success_time = None
        self.state_changed_at = utc_now()

        logger.info("cima.circuit_breaker.reset", event_type="circuit_breaker")

    def _check_state_transition(self) -> None:
        """Verificar si es necesario cambiar el estado del circuit breaker."""
        if self.state == CircuitState.OPEN:
            if self._should_transition_to_half_open():
                self._transition_to_half_open()
                logger.info(
                    "cima.circuit_breaker.half_open", event_type="circuit_breaker", timeout_seconds=self.timeout_seconds
                )

    def _should_transition_to_half_open(self) -> bool:
        """Verificar si debe transicionar de OPEN a HALF_OPEN."""
        if not self.last_failure_time:
            return False

        time_since_failure = (utc_now() - self.last_failure_time).total_seconds()
        return time_since_failure >= self.timeout_seconds

    def _transition_to_open(self) -> None:
        """Transicionar al estado OPEN."""
        self.state = CircuitState.OPEN
        self.state_changed_at = utc_now()
        self.half_open_attempts = 0

    def _transition_to_half_open(self) -> None:
        """Transicionar al estado HALF_OPEN."""
        self.state = CircuitState.HALF_OPEN
        self.state_changed_at = utc_now()
        self.half_open_attempts = 0

    def _transition_to_closed(self) -> None:
        """Transicionar al estado CLOSED."""
        self.state = CircuitState.CLOSED
        self.state_changed_at = utc_now()
        self.failure_count = 0
        self.half_open_attempts = 0

    def _time_until_half_open(self) -> Optional[int]:
        """Calcular tiempo restante hasta pasar a HALF_OPEN."""
        if not self.last_failure_time:
            return None

        time_since_failure = (utc_now() - self.last_failure_time).total_seconds()
        remaining = self.timeout_seconds - time_since_failure
        return max(0, int(remaining))
