# backend/app/services/category_alias_service.py
"""
Service for category alias management.

Issue #459: Category normalization en base de datos.

Este servicio gestiona los aliases de categorías que normalizan
los outputs del clasificador NECESIDAD a los nombres esperados en DB.

Ejemplo:
    "aftas_llagas" (output clasificador) → "aftas" (nombre en DB)

Diferente de KeywordOverrideService:
    - Este normaliza OUTPUT del clasificador
    - KeywordOverride hace pattern matching en INPUT (nombre producto)
"""

import logging
from datetime import datetime, timezone
from typing import Optional

from sqlalchemy import func
from sqlalchemy.orm import Session

from app.models.category_alias import CategoryAlias
from app.schemas.category_alias import CategoryAliasCreate, CategoryAliasUpdate

logger = logging.getLogger(__name__)


class CategoryAliasService:
    """
    Service para CRUD de category aliases.

    Mantiene fallback a dict estático para robustez.
    """

    def __init__(self, db: Session):
        self.db = db

    # =========================================================================
    # CRUD Operations
    # =========================================================================

    def list_aliases(
        self,
        is_active: Optional[bool] = None,
        search: Optional[str] = None,
        limit: int = 100,
        offset: int = 0,
    ) -> dict:
        """
        Lista aliases con filtros y estadísticas.

        Args:
            is_active: Filtrar por estado activo/inactivo
            search: Buscar en source_category o target_category
            limit: Máximo de resultados
            offset: Paginación

        Returns:
            dict con items, total, offset, limit, stats
        """
        query = self.db.query(CategoryAlias)

        # Aplicar filtros
        if is_active is not None:
            query = query.filter(CategoryAlias.is_active == is_active)
        if search:
            escaped_search = search.replace("\\", "\\\\").replace("%", "\\%").replace("_", "\\_")
            search_filter = f"%{escaped_search}%"
            query = query.filter(
                (CategoryAlias.source_category.ilike(search_filter, escape="\\"))
                | (CategoryAlias.target_category.ilike(search_filter, escape="\\"))
            )

        # Contar total
        total = query.count()

        # Ordenar: por uso descendente, luego alfabético
        items = (
            query.order_by(
                CategoryAlias.usage_count.desc(),
                CategoryAlias.source_category,
            )
            .offset(offset)
            .limit(limit)
            .all()
        )

        # Calcular estadísticas
        stats = self._get_stats()

        return {
            "items": items,
            "total": total,
            "offset": offset,
            "limit": limit,
            "stats": stats,
        }

    def get_alias(self, alias_id: int) -> Optional[CategoryAlias]:
        """Obtener un alias por ID."""
        return self.db.query(CategoryAlias).filter(CategoryAlias.id == alias_id).first()

    def get_by_source(self, source_category: str) -> Optional[CategoryAlias]:
        """Obtener alias por source_category."""
        return (
            self.db.query(CategoryAlias)
            .filter(CategoryAlias.source_category == source_category.lower())
            .first()
        )

    def create_alias(self, data: CategoryAliasCreate) -> CategoryAlias:
        """
        Crea un nuevo alias.

        Args:
            data: Datos del alias

        Returns:
            CategoryAlias creado

        Raises:
            ValueError: Si el source_category ya existe
        """
        # Verificar duplicado
        existing = self.get_by_source(data.source_category)
        if existing:
            raise ValueError(
                f"Ya existe un alias para '{data.source_category}'"
            )

        # Crear alias
        alias = CategoryAlias(
            source_category=data.source_category.lower(),
            target_category=data.target_category.lower(),
            reason=data.reason,
            is_active=True,
        )

        self.db.add(alias)
        self.db.commit()
        self.db.refresh(alias)

        logger.info(
            f"Created category alias: {alias.source_category} → {alias.target_category}"
        )

        return alias

    def update_alias(
        self, alias_id: int, data: CategoryAliasUpdate
    ) -> Optional[CategoryAlias]:
        """
        Actualiza un alias existente.

        Args:
            alias_id: ID del alias a actualizar
            data: Datos a actualizar

        Returns:
            CategoryAlias actualizado o None si no existe

        Raises:
            ValueError: Si el nuevo source_category ya existe en otro alias
        """
        alias = self.get_alias(alias_id)
        if not alias:
            return None

        # Si cambia source_category, verificar que no exista
        if data.source_category and data.source_category.lower() != alias.source_category:
            existing = self.get_by_source(data.source_category)
            if existing and existing.id != alias_id:
                raise ValueError(
                    f"Ya existe un alias para '{data.source_category}'"
                )

        # Aplicar actualizaciones
        update_data = data.model_dump(exclude_unset=True)
        for field, value in update_data.items():
            if value is not None:
                setattr(alias, field, value)

        self.db.commit()
        self.db.refresh(alias)

        logger.info(f"Updated category alias id={alias_id}")

        return alias

    def delete_alias(self, alias_id: int, hard: bool = False) -> bool:
        """
        Elimina un alias.

        Args:
            alias_id: ID del alias a eliminar
            hard: Si True, elimina permanentemente. Si False, desactiva (soft delete)

        Returns:
            True si se eliminó/desactivó, False si no existe
        """
        alias = self.get_alias(alias_id)
        if not alias:
            return False

        if hard:
            self.db.delete(alias)
            logger.info(f"Hard deleted category alias id={alias_id}")
        else:
            alias.is_active = False
            logger.info(f"Soft deleted (deactivated) category alias id={alias_id}")

        self.db.commit()
        return True

    def toggle_active(self, alias_id: int) -> Optional[CategoryAlias]:
        """
        Alternar estado activo/inactivo de un alias.

        Args:
            alias_id: ID del alias

        Returns:
            CategoryAlias actualizado o None si no existe
        """
        alias = self.get_alias(alias_id)
        if not alias:
            return None

        alias.is_active = not alias.is_active
        self.db.commit()
        self.db.refresh(alias)

        logger.info(
            f"Toggled category alias id={alias_id} to is_active={alias.is_active}"
        )

        return alias

    # =========================================================================
    # Normalization (Core functionality)
    # =========================================================================

    def normalize(self, category: str) -> str:
        """
        Normaliza una categoría usando aliases activos.

        Args:
            category: Categoría del clasificador

        Returns:
            Categoría normalizada (o la original si no hay alias)
        """
        alias = (
            self.db.query(CategoryAlias)
            .filter(
                CategoryAlias.source_category == category.lower(),
                CategoryAlias.is_active == True,  # noqa: E712
            )
            .first()
        )

        if alias:
            # Incrementar contador de uso
            alias.increment_usage()
            self.db.commit()
            return alias.target_category

        return category

    def normalize_batch(self, categories: list[str]) -> dict[str, str]:
        """
        Normaliza múltiples categorías en una sola operación.

        Optimizado para alto volumen: un solo commit para todos los
        incrementos de usage_count.

        Args:
            categories: Lista de categorías a normalizar

        Returns:
            Dict {categoria_original: categoria_normalizada}
        """
        if not categories:
            return {}

        # Obtener todos los aliases activos de una vez
        unique_cats = list(set(cat.lower() for cat in categories))
        aliases = (
            self.db.query(CategoryAlias)
            .filter(
                CategoryAlias.source_category.in_(unique_cats),
                CategoryAlias.is_active == True,  # noqa: E712
            )
            .all()
        )

        # Crear lookup dict
        alias_map = {a.source_category: a for a in aliases}

        # Construir resultados e incrementar usage
        results = {}
        for category in categories:
            cat_lower = category.lower()
            if cat_lower in alias_map:
                alias = alias_map[cat_lower]
                results[category] = alias.target_category
                alias.increment_usage()
            else:
                results[category] = category

        # Un solo commit para todos los incrementos
        if aliases:
            self.db.commit()

        return results

    def get_all_active_mappings(self) -> dict[str, str]:
        """
        Obtener todos los mappings activos como dict.

        Returns:
            Dict {source_category: target_category}
        """
        aliases = (
            self.db.query(CategoryAlias)
            .filter(CategoryAlias.is_active == True)  # noqa: E712
            .all()
        )

        return {a.source_category: a.target_category for a in aliases}

    # =========================================================================
    # Statistics
    # =========================================================================

    def _get_stats(self) -> dict:
        """Calcular estadísticas de aliases."""
        total = self.db.query(CategoryAlias).count()
        active = (
            self.db.query(CategoryAlias)
            .filter(CategoryAlias.is_active == True)  # noqa: E712
            .count()
        )
        inactive = total - active

        total_usage = (
            self.db.query(func.sum(CategoryAlias.usage_count)).scalar() or 0
        )

        # Más usado
        most_used = (
            self.db.query(CategoryAlias)
            .filter(CategoryAlias.usage_count > 0)
            .order_by(CategoryAlias.usage_count.desc())
            .first()
        )

        # Menos usado (de los activos)
        least_used = (
            self.db.query(CategoryAlias)
            .filter(CategoryAlias.is_active == True)  # noqa: E712
            .order_by(CategoryAlias.usage_count.asc())
            .first()
        )

        return {
            "total": total,
            "active": active,
            "inactive": inactive,
            "total_usage": total_usage,
            "most_used": most_used.source_category if most_used else None,
            "least_used": least_used.source_category if least_used else None,
        }

    def get_stats(self) -> dict:
        """Obtener estadísticas públicas."""
        return self._get_stats()
