"""
Servicio para gestión del historial de sincronización del catálogo.

Maneja el registro y consulta del historial de sincronizaciones
con CIMA y nomenclator, proporcionando visibilidad sobre el estado
y rendimiento de las operaciones de actualización del catálogo.
"""

import logging
from datetime import datetime, timezone
from typing import Dict, Optional, Any

from sqlalchemy.orm import Session
from sqlalchemy.exc import SQLAlchemyError

from app.models.catalog_sync_history import (
    CatalogSyncHistory,
    SyncType,
    SyncStatus,
    TriggerType
)

logger = logging.getLogger(__name__)


class CatalogSyncHistoryService:
    """
    Servicio para gestionar el historial de sincronización del catálogo.

    Proporciona métodos para:
    - Registrar eventos de sincronización
    - Consultar el historial con filtros y paginación
    - Obtener estadísticas de sincronización
    """

    def __init__(self, db: Session):
        """
        Inicializa el servicio con una sesión de base de datos.

        Args:
            db: Sesión de SQLAlchemy
        """
        self.db = db

    def log_sync_event(
        self,
        sync_type: SyncType,
        status: SyncStatus,
        records_updated: int,
        duration_seconds: float,
        triggered_by: TriggerType,
        error_message: Optional[str] = None
    ) -> CatalogSyncHistory:
        """
        Registra un evento de sincronización en el historial.

        Args:
            sync_type: Tipo de sincronización (CIMA o nomenclator)
            status: Estado de la sincronización (success, failure, partial)
            records_updated: Número de registros actualizados
            duration_seconds: Duración de la operación en segundos
            triggered_by: Origen del trigger (automatic o manual)
            error_message: Mensaje de error (si aplica)

        Returns:
            CatalogSyncHistory: El registro creado

        Raises:
            Exception: Si hay un error al guardar en la base de datos
        """
        try:
            # Crear el registro de sincronización
            sync_history = CatalogSyncHistory(
                sync_date=datetime.now(timezone.utc),
                sync_type=sync_type,
                status=status,
                records_updated=records_updated,
                duration_seconds=duration_seconds,
                triggered_by=triggered_by,
                error_message=error_message
            )

            # Guardar en la base de datos
            self.db.add(sync_history)
            self.db.commit()

            logger.info(
                f"Registrado evento de sincronización: {sync_type.value} - "
                f"Estado: {status.value} - Registros: {records_updated} - "
                f"Duración: {duration_seconds:.2f}s"
            )

            return sync_history

        except SQLAlchemyError as e:
            logger.error(f"Error al registrar evento de sincronización: {str(e)}")
            self.db.rollback()
            raise

    def get_sync_history(
        self,
        limit: int = 50,
        offset: int = 0,
        sync_type: Optional[str] = None,
        status: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Obtiene el historial de sincronización con filtros opcionales.

        Args:
            limit: Número máximo de registros a retornar (max 100)
            offset: Número de registros a saltar (para paginación)
            sync_type: Filtrar por tipo de sincronización ("cima" o "nomenclator")
            status: Filtrar por estado ("success", "failure", "partial")

        Returns:
            Dict con:
                - history: Lista de diccionarios con los registros de sincronización
                - total_count: Número total de registros (sin paginación)
        """
        try:
            # Construir query base
            query = self.db.query(CatalogSyncHistory)

            # Aplicar filtros si se proporcionan
            if sync_type:
                # Validar tipo de sincronización
                if sync_type.lower() in [t.value for t in SyncType]:
                    query = query.filter(
                        CatalogSyncHistory.sync_type == sync_type.lower()
                    )

            if status:
                # Validar estado
                if status.lower() in [s.value for s in SyncStatus]:
                    query = query.filter(
                        CatalogSyncHistory.status == status.lower()
                    )

            # Obtener el total de registros (antes de paginación)
            total_count = query.count()

            # Aplicar ordenamiento (más reciente primero), límite y offset
            histories = (
                query
                .order_by(CatalogSyncHistory.sync_date.desc())
                .limit(limit)
                .offset(offset)
                .all()
            )

            # Convertir a diccionarios
            history_list = [history.to_dict() for history in histories]

            logger.debug(
                f"Recuperados {len(history_list)} registros de sincronización "
                f"(total: {total_count}, offset: {offset}, limit: {limit})"
            )

            return {
                "history": history_list,
                "total_count": total_count
            }

        except SQLAlchemyError as e:
            logger.error(f"Error al obtener historial de sincronización: {str(e)}")
            # Retornar respuesta vacía en caso de error
            return {
                "history": [],
                "total_count": 0
            }

    def get_last_successful_sync(
        self,
        sync_type: Optional[SyncType] = None
    ) -> Optional[CatalogSyncHistory]:
        """
        Obtiene la última sincronización exitosa.

        Args:
            sync_type: Tipo de sincronización específico (opcional)

        Returns:
            CatalogSyncHistory o None si no hay sincronizaciones exitosas
        """
        try:
            query = self.db.query(CatalogSyncHistory).filter(
                CatalogSyncHistory.status == SyncStatus.SUCCESS
            )

            if sync_type:
                query = query.filter(CatalogSyncHistory.sync_type == sync_type)

            return query.order_by(CatalogSyncHistory.sync_date.desc()).first()

        except SQLAlchemyError as e:
            logger.error(
                f"Error al obtener última sincronización exitosa: {str(e)}"
            )
            return None

    def get_sync_statistics(self) -> Dict[str, Any]:
        """
        Obtiene estadísticas del historial de sincronización.

        Returns:
            Dict con estadísticas agregadas por tipo y estado
        """
        try:
            # Obtener estadísticas por tipo de sincronización
            stats = {
                "cima": {
                    "total": 0,
                    "success": 0,
                    "failure": 0,
                    "partial": 0,
                    "last_success": None,
                    "average_duration": 0.0
                },
                "nomenclator": {
                    "total": 0,
                    "success": 0,
                    "failure": 0,
                    "partial": 0,
                    "last_success": None,
                    "average_duration": 0.0
                }
            }

            for sync_type in SyncType:
                # Contar por estado
                for status in SyncStatus:
                    count = self.db.query(CatalogSyncHistory).filter(
                        CatalogSyncHistory.sync_type == sync_type,
                        CatalogSyncHistory.status == status
                    ).count()

                    stats[sync_type.value][status.value] = count
                    stats[sync_type.value]["total"] += count

                # Obtener última sincronización exitosa
                last_success = self.get_last_successful_sync(sync_type)
                if last_success:
                    stats[sync_type.value]["last_success"] = (
                        last_success.sync_date.isoformat()
                    )

                # Calcular duración promedio de sincronizaciones exitosas
                if stats[sync_type.value]["success"] > 0:
                    avg_duration = self.db.query(
                        CatalogSyncHistory.duration_seconds
                    ).filter(
                        CatalogSyncHistory.sync_type == sync_type,
                        CatalogSyncHistory.status == SyncStatus.SUCCESS
                    ).all()

                    if avg_duration:
                        durations = [d[0] for d in avg_duration if d[0] is not None]
                        if durations:
                            stats[sync_type.value]["average_duration"] = (
                                sum(durations) / len(durations)
                            )

            logger.debug(f"Estadísticas de sincronización calculadas: {stats}")
            return stats

        except SQLAlchemyError as e:
            logger.error(f"Error al obtener estadísticas de sincronización: {str(e)}")
            return {}
