# backend/app/services/catalog_startup.py
"""
Funciones para sincronización del catálogo al inicio
"""

import logging
import threading

from app.database import SessionLocal
from app.services.catalog_maintenance_service import catalog_maintenance_service

logger = logging.getLogger(__name__)


async def check_and_update_catalog_if_needed() -> None:
    """
    Lógica común para verificar si el catálogo necesita actualizarse
    Si los datos tienen más de 10 días, actualiza nomenclator, CIMA y grupos homogéneos
    """
    try:
        db = SessionLocal()
        try:
            # Obtener estado del catálogo
            status = catalog_maintenance_service.get_catalog_status(db)
            needs_update = status.get("needs_update", True)

            if status["catalog"]["total_products"] == 0:
                needs_update = True
                logger.warning("[CATALOG] Catálogo vacío, iniciando sincronización inicial")
            elif needs_update:
                logger.warning("[CATALOG] Catálogo antiguo (>10 días), actualizando...")
            else:
                logger.info(f"[CATALOG] Catálogo actualizado: {status['catalog']['total_products']} productos")
        finally:
            db.close()

        if needs_update:
            # Ejecutar actualización en thread separado
            def update_catalog():
                try:
                    logger.info("[CATALOG] Iniciando sincronización completa del catálogo...")
                    db = SessionLocal()
                    try:
                        result = catalog_maintenance_service.check_and_update(db)
                        if result.get("status") == "success":
                            logger.info("[CATALOG] Sincronización completada exitosamente")
                            logger.info(f"[CATALOG] Nomenclator: {result.get('nomenclator', {})}")
                            logger.info(f"[CATALOG] CIMA: {result.get('cima', {})}")
                            logger.info(f"[CATALOG] Grupos homogéneos: {result.get('homogeneous_groups', {})}")
                        elif result.get("status") == "up_to_date":
                            logger.info("[CATALOG] Catálogo ya está actualizado")
                        else:
                            logger.error(f"[CATALOG] Error en sincronización: {result.get('message')}")
                    finally:
                        db.close()
                except Exception as e:
                    logger.error(f"[CATALOG] Error durante sincronización: {str(e)}")

            # Ejecutar en background SIN BLOQUEAR
            # Usar threading.Thread en lugar de ThreadPoolExecutor con 'with'
            # para evitar que el context manager espere a que termine
            background_thread = threading.Thread(target=update_catalog, daemon=True)
            background_thread.start()
            logger.info("[CATALOG] Sincronización iniciada en background (thread daemon)")

    except Exception as e:
        logger.error(f"[CATALOG] Error en verificación: {str(e)}")
