# backend/app/services/brand_detection_service.py
"""
Servicio de detección de marca y línea de producto.

Issue #446: Taxonomía jerárquica de venta libre
ADR-001: docs/architecture/ADR-001-VENTA-LIBRE-TAXONOMY-HIERARCHY.md

Este servicio detecta:
- detected_brand: Marca del producto (ej: "ISDIN", "Heliocare")
- brand_line: Línea de producto (ej: "Fusion Water", "360")
- ml_subcategory: Zona/subcategoría (ej: "facial", "corporal")

Usa el conocimiento estructurado de notebooks/brand_product_lines.py
"""

import re
from dataclasses import dataclass, field
from typing import Dict, List, Optional, Tuple

import structlog

logger = structlog.get_logger(__name__)


@dataclass
class ProductLineInfo:
    """Información de una línea de producto."""
    necesidad: str
    zona: str = ""
    tipo: str = ""
    keywords: List[str] = field(default_factory=list)
    descripcion: str = ""


@dataclass
class BrandDetectionResult:
    """Resultado de la detección de marca."""
    detected_brand: Optional[str] = None
    brand_line: Optional[str] = None
    ml_subcategory: Optional[str] = None  # Mapped from 'zona'
    necesidad: Optional[str] = None
    confidence: float = 0.0

    def to_dict(self) -> Dict:
        return {
            "detected_brand": self.detected_brand,
            "brand_line": self.brand_line,
            "ml_subcategory": self.ml_subcategory,
            "necesidad": self.necesidad,
            "confidence": self.confidence,
        }


# =============================================================================
# BRAND PRODUCT LINES DATABASE
# Extracted from notebooks/brand_product_lines.py for production use
# =============================================================================

# ISDIN
ISDIN_LINES = {
    'fotoprotector': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
    'fusion water': ProductLineInfo(necesidad='proteccion_solar', zona='facial', tipo='fluido'),
    'fusion gel': ProductLineInfo(necesidad='proteccion_solar', zona='corporal', tipo='gel'),
    'fusion air': ProductLineInfo(necesidad='proteccion_solar', zona='facial', tipo='fluido'),
    'transparent spray': ProductLineInfo(necesidad='proteccion_solar', zona='corporal', tipo='spray'),
    'hydro oil': ProductLineInfo(necesidad='proteccion_solar', zona='corporal', tipo='aceite'),
    'pediatrics': ProductLineInfo(necesidad='proteccion_solar_infantil', zona='corporal'),
    'eryfotona': ProductLineInfo(necesidad='proteccion_solar', zona='facial', keywords=['actinica']),
    'foto ultra': ProductLineInfo(necesidad='proteccion_solar', zona='facial', keywords=['age repair']),
    'isdinceutics': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'flavo-c': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial', keywords=['vitamina c']),
    'melaclear': ProductLineInfo(necesidad='manchas', zona='facial'),
    'melatonik': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial', keywords=['noche']),
    'glicoisdin': ProductLineInfo(necesidad='exfoliacion', zona='facial', keywords=['glicolico']),
    'retinal': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial', keywords=['retinol']),
    'k-ox': ProductLineInfo(necesidad='contorno_ojos', zona='facial'),
    'acniben': ProductLineInfo(necesidad='acne', zona='facial'),
    'ureadin': ProductLineInfo(necesidad='hidratacion_corporal', zona='corporal', keywords=['urea']),
    'ureadin podos': ProductLineInfo(necesidad='cuidado_pies', zona='pies'),
    'nutradeica': ProductLineInfo(necesidad='dermatitis_seborreica', zona='facial'),
    'lambdapil': ProductLineInfo(necesidad='caida_cabello', zona='capilar'),
    'woman isdin': ProductLineInfo(necesidad='higiene_intima', zona='intimo'),
}

# HELIOCARE
HELIOCARE_LINES = {
    '360': ProductLineInfo(necesidad='proteccion_solar', zona='facial'),
    '360 water gel': ProductLineInfo(necesidad='proteccion_solar', zona='facial', tipo='gel'),
    '360 fluid cream': ProductLineInfo(necesidad='proteccion_solar', zona='facial', tipo='crema'),
    '360 color': ProductLineInfo(necesidad='proteccion_solar', zona='facial', keywords=['maquillaje']),
    'ultra': ProductLineInfo(necesidad='proteccion_solar', zona='facial'),
    'advanced': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
    'mineral': ProductLineInfo(necesidad='proteccion_solar', zona='facial', keywords=['filtros fisicos']),
    'capsulas': ProductLineInfo(necesidad='proteccion_solar', zona='oral', tipo='capsulas'),
}

# BIODERMA
BIODERMA_LINES = {
    'sensibio': ProductLineInfo(necesidad='piel_sensible', zona='facial'),
    'sensibio h2o': ProductLineInfo(necesidad='limpieza_facial', zona='facial', tipo='micelar'),
    'atoderm': ProductLineInfo(necesidad='piel_atopica', zona='corporal'),
    'sebium': ProductLineInfo(necesidad='acne', zona='facial'),
    'cicabio': ProductLineInfo(necesidad='cicatrizacion', zona='corporal'),
    'photoderm': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
    'pigmentbio': ProductLineInfo(necesidad='manchas', zona='facial'),
    'hydrabio': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'node': ProductLineInfo(necesidad='caida_cabello', zona='capilar'),
}

# LA ROCHE POSAY
LAROCHE_LINES = {
    'effaclar': ProductLineInfo(necesidad='acne', zona='facial'),
    'toleriane': ProductLineInfo(necesidad='piel_sensible', zona='facial'),
    'anthelios': ProductLineInfo(necesidad='proteccion_solar', zona='facial'),
    'cicaplast': ProductLineInfo(necesidad='cicatrizacion', zona='corporal'),
    'lipikar': ProductLineInfo(necesidad='piel_atopica', zona='corporal'),
    'hyalu b5': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'redermic': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'pigmentclar': ProductLineInfo(necesidad='manchas', zona='facial'),
}

# AVENE
AVENE_LINES = {
    'cleanance': ProductLineInfo(necesidad='acne', zona='facial'),
    'cicalfate': ProductLineInfo(necesidad='cicatrizacion', zona='corporal'),
    'xeracalm': ProductLineInfo(necesidad='piel_atopica', zona='corporal'),
    'tolerance': ProductLineInfo(necesidad='piel_sensible', zona='facial'),
    'hydrance': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'dermabsolu': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'solaire': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
}

# EUCERIN
EUCERIN_LINES = {
    'hyaluron-filler': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'atopicontrol': ProductLineInfo(necesidad='piel_atopica', zona='corporal'),
    'dermopure': ProductLineInfo(necesidad='acne', zona='facial'),
    'urea repair': ProductLineInfo(necesidad='piel_seca', zona='corporal', keywords=['urea']),
    'sun': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
    'pigment control': ProductLineInfo(necesidad='manchas', zona='facial'),
    'aquaporin': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
}

# SESDERMA
SESDERMA_LINES = {
    'c-vit': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial', keywords=['vitamina c']),
    'azelac': ProductLineInfo(necesidad='acne', zona='facial'),
    'reti age': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial', keywords=['retinol']),
    'factor g': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'hidraderm': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'repaskin': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
    'daeses': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
}

# CERAVE
CERAVE_LINES = {
    'hidratante': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'limpiadora': ProductLineInfo(necesidad='limpieza_facial', zona='facial'),
    'sa': ProductLineInfo(necesidad='piel_seca', zona='corporal', keywords=['salicilico']),
    'espumosa': ProductLineInfo(necesidad='limpieza_facial', zona='facial'),
    'pm': ProductLineInfo(necesidad='hidratacion_facial', zona='facial', keywords=['noche']),
    'am': ProductLineInfo(necesidad='hidratacion_facial', zona='facial', keywords=['dia', 'spf']),
}

# GUM - Dental
GUM_LINES = {
    'trav-ler': ProductLineInfo(necesidad='interdental', zona='dental'),
    'soft-picks': ProductLineInfo(necesidad='interdental', zona='dental'),
    'activital': ProductLineInfo(necesidad='higiene_dental', zona='dental'),
    'paroex': ProductLineInfo(necesidad='encias', zona='dental'),
    'gingidex': ProductLineInfo(necesidad='encias', zona='dental'),
    'ortho': ProductLineInfo(necesidad='higiene_dental', zona='dental', keywords=['ortodoncia']),
    'aftaclear': ProductLineInfo(necesidad='aftas', zona='dental'),
    'sensivital': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
}

# LACER - Dental
LACER_LINES = {
    'gingilacer': ProductLineInfo(necesidad='encias', zona='dental'),
    'fluocaril': ProductLineInfo(necesidad='higiene_dental', zona='dental'),
    'sensilacer': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
    'blanc': ProductLineInfo(necesidad='blanqueamiento', zona='dental'),
    'ortolacer': ProductLineInfo(necesidad='higiene_dental', zona='dental', keywords=['ortodoncia']),
    'lacer fresh': ProductLineInfo(necesidad='higiene_dental', zona='dental'),
    'interdental': ProductLineInfo(necesidad='interdental', zona='dental'),
}

# BLEMIL - Bebé
BLEMIL_LINES = {
    'optimum': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'evolution': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'confort': ProductLineInfo(necesidad='colicos_bebe', zona='infantil'),
    'plus': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'ar': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil', keywords=['antiregurgitacion']),
}

# KIN - Dental
KIN_LINES = {
    'gingival': ProductLineInfo(necesidad='encias', zona='dental'),
    'fluor-kin': ProductLineInfo(necesidad='caries', zona='dental'),
    'orthokin': ProductLineInfo(necesidad='higiene_dental', zona='dental', keywords=['ortodoncia']),
    'kin b5': ProductLineInfo(necesidad='aftas', zona='dental'),
    'sensikin': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
}

# COREGA - Dental
COREGA_LINES = {
    'ultra': ProductLineInfo(necesidad='protesis_dental', zona='dental'),
    'tabs': ProductLineInfo(necesidad='protesis_dental', zona='dental', tipo='tabletas'),
    'crema': ProductLineInfo(necesidad='protesis_dental', zona='dental', tipo='crema'),
    'sin sabor': ProductLineInfo(necesidad='protesis_dental', zona='dental'),
}

# KUKIDENT - Dental
KUKIDENT_LINES = {
    'pro': ProductLineInfo(necesidad='protesis_dental', zona='dental'),
    'complete': ProductLineInfo(necesidad='protesis_dental', zona='dental'),
    'fresh': ProductLineInfo(necesidad='protesis_dental', zona='dental'),
}

# OLISTIC - Capilar
OLISTIC_LINES = {
    'for women': ProductLineInfo(necesidad='caida_cabello', zona='capilar', keywords=['mujer']),
    'for men': ProductLineInfo(necesidad='caida_cabello', zona='capilar', keywords=['hombre']),
    'forte': ProductLineInfo(necesidad='caida_cabello', zona='capilar'),
}

# THEALOZ - Ocular
THEALOZ_LINES = {
    'duo': ProductLineInfo(necesidad='ojo_seco', zona='ocular'),
    'duo gel': ProductLineInfo(necesidad='ojo_seco', zona='ocular', tipo='gel'),
}

# IVB - Suplementos
IVB_LINES = {
    'collagen': ProductLineInfo(necesidad='colageno', zona='oral'),
    'vitamina d': ProductLineInfo(necesidad='vitaminas_general', zona='oral'),
    'melatonina': ProductLineInfo(necesidad='sueno_insomnio', zona='oral'),
    'omega': ProductLineInfo(necesidad='omega3', zona='oral'),
}

# NS CINFA - Suplementos
NS_LINES = {
    'gineprotect': ProductLineInfo(necesidad='infeccion_urinaria', zona='intimo'),
    'cisprenicina': ProductLineInfo(necesidad='infeccion_urinaria', zona='intimo'),
    'vitans': ProductLineInfo(necesidad='vitaminas_general', zona='oral'),
    'sonadol': ProductLineInfo(necesidad='sueno_insomnio', zona='oral'),
    'lipoless': ProductLineInfo(necesidad='perdida_peso', zona='oral'),
    'lipocarbo': ProductLineInfo(necesidad='perdida_peso', zona='oral'),
    'digestconfort': ProductLineInfo(necesidad='gases_digestion', zona='oral'),
    'florabiotic': ProductLineInfo(necesidad='flora_intestinal', zona='oral'),
}

# GH COSMETICS
GH_LINES = {
    '25 peptidos': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'mascarilla': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'azelaico': ProductLineInfo(necesidad='acne', zona='facial'),
    '5 azelaico': ProductLineInfo(necesidad='acne', zona='facial'),
    'kiro': ProductLineInfo(necesidad='limpieza_facial', zona='facial'),
    'cristina mitre': ProductLineInfo(necesidad='limpieza_facial', zona='facial'),
    'cofre': ProductLineInfo(necesidad='set_regalo', zona='facial'),
    '10 aha': ProductLineInfo(necesidad='exfoliacion', zona='facial'),
    'esencia': ProductLineInfo(necesidad='limpieza_facial', zona='facial'),
    'triple h': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'serum': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
}

# =============================================================================
# NUEVAS MARCAS (2025-12-16) - Identificadas por analyze_unmapped_brands.py
# =============================================================================

# VOLTADOL - Antiinflamatorio tópico
VOLTADOL_LINES = {
    'forte': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
    'gel': ProductLineInfo(necesidad='dolor_muscular', zona='corporal', tipo='gel'),
}

# BE+ - Dermocosmética
BEPLUS_LINES = {
    'energifique': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'med': ProductLineInfo(necesidad='cicatrizacion', zona='corporal'),
    'capilar': ProductLineInfo(necesidad='caida_cabello', zona='capilar'),
    'cuidado solar': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
}

# FRENADOL - Antigripal OTC
FRENADOL_LINES = {
    'complex': ProductLineInfo(necesidad='gripe_resfriado', zona='oral'),
    'descongestivo': ProductLineInfo(necesidad='congestion_nasal', zona='oral'),
    'hot lemon': ProductLineInfo(necesidad='gripe_resfriado', zona='oral'),
}

# MEDIK8 - Dermocosmética premium
MEDIK8_LINES = {
    'crystal retinal': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial', keywords=['retinol']),
    'c-tetra': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial', keywords=['vitamina c']),
    'liquid peptides': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'clarity peptides': ProductLineInfo(necesidad='manchas', zona='facial'),
    'press & glow': ProductLineInfo(necesidad='exfoliacion', zona='facial'),
}

# BEXIDENT - Dental
BEXIDENT_LINES = {
    'encias': ProductLineInfo(necesidad='encias', zona='dental'),
    'aftas': ProductLineInfo(necesidad='aftas', zona='dental'),
    'anticaries': ProductLineInfo(necesidad='caries', zona='dental'),
    'blanqueante': ProductLineInfo(necesidad='blanqueamiento', zona='dental'),
}

# CUMLAUDE - Ginecología/Dermocosmética
CUMLAUDE_LINES = {
    'gynelaude': ProductLineInfo(necesidad='higiene_intima', zona='intimo'),
    'summum': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'sunlaude': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
}

# PEDIALAC - Alimentación infantil
PEDIALAC_LINES = {
    'leche': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'cereales': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'potitos': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
}

# KLORANE - Capilar/Dermocosmética
KLORANE_LINES = {
    'quinina': ProductLineInfo(necesidad='caida_cabello', zona='capilar'),
    'camomila': ProductLineInfo(necesidad='cabello_rubio', zona='capilar'),
    'ortiga': ProductLineInfo(necesidad='cabello_graso', zona='capilar'),
    'mango': ProductLineInfo(necesidad='cabello_seco', zona='capilar'),
    'cedrat': ProductLineInfo(necesidad='cabello_graso', zona='capilar'),
    'aciano': ProductLineInfo(necesidad='limpieza_facial', zona='facial'),
}

# ESTHEDERM - Dermocosmética premium
ESTHEDERM_LINES = {
    'intensive': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'osmoclean': ProductLineInfo(necesidad='limpieza_facial', zona='facial'),
    'photo': ProductLineInfo(necesidad='proteccion_solar', zona='facial'),
    'excellage': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
}

# FLUIMUCIL - Mucolítico OTC
FLUIMUCIL_LINES = {
    'forte': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='oral'),
    'complex': ProductLineInfo(necesidad='gripe_resfriado', zona='oral'),
}

# CRISTALMINA - Antiséptico
CRISTALMINA_LINES = {
    'film': ProductLineInfo(necesidad='heridas', zona='corporal'),
    'spray': ProductLineInfo(necesidad='heridas', zona='corporal', tipo='spray'),
}

# ALMAX - Antiácido OTC
ALMAX_LINES = {
    'forte': ProductLineInfo(necesidad='acidez_estomago', zona='oral'),
    'suspension': ProductLineInfo(necesidad='acidez_estomago', zona='oral'),
}

# NHCO - Suplementos nutricosmética
NHCO_LINES = {
    'artixine': ProductLineInfo(necesidad='articulaciones', zona='oral'),
    'dermactiv': ProductLineInfo(necesidad='piel_unas_cabello', zona='oral'),
    'aminoscience': ProductLineInfo(necesidad='energia_cansancio', zona='oral'),
}

# ZZZQUIL - Sueño OTC
ZZZQUIL_LINES = {
    'natura': ProductLineInfo(necesidad='sueno_insomnio', zona='oral'),
    'plus': ProductLineInfo(necesidad='sueno_insomnio', zona='oral'),
}

# IBUDOL - Ibuprofeno OTC
IBUDOL_LINES = {
    'forte': ProductLineInfo(necesidad='dolor_cabeza', zona='oral'),
    'rapid': ProductLineInfo(necesidad='dolor_cabeza', zona='oral'),
}

# RESPIBIEN - Descongestionante nasal
RESPIBIEN_LINES = {
    'antialergico': ProductLineInfo(necesidad='congestion_nasal', zona='nasal'),
    'freshmint': ProductLineInfo(necesidad='congestion_nasal', zona='nasal'),
}


# =============================================================================
# MARCAS ADICIONALES - 2025-12-16 (Issue #446 - Fase 2)
# =============================================================================

# MITOSYL - Higiene bebé
MITOSYL_LINES = {
    'pomada': ProductLineInfo(necesidad='dermatitis_panal', zona='corporal'),
    'protectora': ProductLineInfo(necesidad='dermatitis_panal', zona='corporal'),
    'balsam': ProductLineInfo(necesidad='dermatitis_panal', zona='corporal'),
}

# VICKS/VAPORUB - Respiratorio
VICKS_LINES = {
    'vaporub': ProductLineInfo(necesidad='balsamos_respiratorios', zona='pecho'),
    'inhalador': ProductLineInfo(necesidad='congestion_nasal', zona='nasal'),
    'jarabe': ProductLineInfo(necesidad='tos', zona='oral'),
}

# ORAL-B - Higiene dental
ORALB_LINES = {
    # Cepillos eléctricos y recambios
    'io': ProductLineInfo(necesidad='cepillo_electrico', zona='dental'),  # ORAL-B IO Series
    'vitality': ProductLineInfo(necesidad='cepillo_electrico', zona='dental'),
    'pro 1': ProductLineInfo(necesidad='cepillo_electrico', zona='dental'),
    'pro 3': ProductLineInfo(necesidad='cepillo_electrico', zona='dental'),
    'recambio': ProductLineInfo(necesidad='cepillo_electrico', zona='dental'),
    'electrico': ProductLineInfo(necesidad='cepillo_electrico', zona='dental'),
    'recargable': ProductLineInfo(necesidad='cepillo_electrico', zona='dental'),
    # Otros productos Oral-B
    'pro': ProductLineInfo(necesidad='higiene_dental', zona='dental'),
    'sensitive': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
    'pro-expert': ProductLineInfo(necesidad='higiene_dental', zona='dental'),
    'cepillo': ProductLineInfo(necesidad='higiene_dental', zona='dental'),
}

# OMRON - Dispositivos médicos
OMRON_LINES = {
    'm3': ProductLineInfo(necesidad='monitorizacion', zona='brazo'),
    'm6': ProductLineInfo(necesidad='monitorizacion', zona='brazo'),
    'm7': ProductLineInfo(necesidad='monitorizacion', zona='brazo'),
    'x3': ProductLineInfo(necesidad='monitorizacion', zona='brazo'),
    'inhalador': ProductLineInfo(necesidad='asma', zona='respiratorio'),
}

# PREDICTOR - Tests diagnóstico
PREDICTOR_LINES = {
    'early': ProductLineInfo(necesidad='test_embarazo', zona='diagnostico'),
    'test': ProductLineInfo(necesidad='test_embarazo', zona='diagnostico'),
}

# HYLO - Ocular
HYLO_LINES = {
    'dual': ProductLineInfo(necesidad='ojo_seco', zona='ocular'),
    'comod': ProductLineInfo(necesidad='ojo_seco', zona='ocular'),
    'gel': ProductLineInfo(necesidad='ojo_seco', zona='ocular'),
    'fresh': ProductLineInfo(necesidad='ojo_seco', zona='ocular'),
}

# BIVOS - Probióticos bebé
BIVOS_LINES = {
    'gotas': ProductLineInfo(necesidad='probioticos', zona='digestivo'),
    'lactobacillus': ProductLineInfo(necesidad='probioticos', zona='digestivo'),
}

# DULCOSOFT - Digestivo/estreñimiento
DULCOSOFT_LINES = {
    'polvo': ProductLineInfo(necesidad='estrenimiento', zona='digestivo'),
    'solucion': ProductLineInfo(necesidad='estrenimiento', zona='digestivo'),
    'plus': ProductLineInfo(necesidad='estrenimiento', zona='digestivo'),
}

# MEGALEVURE - Probióticos
MEGALEVURE_LINES = {
    'sticks': ProductLineInfo(necesidad='probioticos', zona='digestivo'),
    'capsulas': ProductLineInfo(necesidad='probioticos', zona='digestivo'),
}

# NUTRACEL - Dermofarmacia
NUTRACEL_LINES = {
    'gel': ProductLineInfo(necesidad='anticelulitis', zona='corporal'),
    'crema': ProductLineInfo(necesidad='anticelulitis', zona='corporal'),
}

# ARTURO ALBA / DR ARTHOUROS - Dermofarmacia premium local
ARTUROALBA_LINES = {
    'exfoliante': ProductLineInfo(necesidad='exfoliacion', zona='facial'),
    'exfolia': ProductLineInfo(necesidad='exfoliacion', zona='facial'),
    'locion': ProductLineInfo(necesidad='exfoliacion', zona='facial'),  # Lociones exfoliantes
    'pate': ProductLineInfo(necesidad='exfoliacion', zona='facial'),  # Pate exfoliante
    'rapsodia': ProductLineInfo(necesidad='antiedad', zona='facial'),
    'factores': ProductLineInfo(necesidad='antiedad', zona='facial'),
    'contorno': ProductLineInfo(necesidad='contorno_ojos', zona='facial'),
    'serum': ProductLineInfo(necesidad='antiedad', zona='facial'),
}

# DAFLON - Circulación venosa
DAFLON_LINES = {
    '500': ProductLineInfo(necesidad='circulacion', zona='oral'),
    '1000': ProductLineInfo(necesidad='circulacion', zona='oral'),
}

# FORTASEC - Antidiarreico
FORTASEC_LINES = {
    'capsulas': ProductLineInfo(necesidad='diarrea', zona='digestivo'),
    'plus': ProductLineInfo(necesidad='diarrea', zona='digestivo'),
}

# NUROFEN - Analgésico/antiinflamatorio
NUROFEN_LINES = {
    'rapid': ProductLineInfo(necesidad='dolor', zona='oral'),
    'forte': ProductLineInfo(necesidad='dolor', zona='oral'),
    'express': ProductLineInfo(necesidad='dolor', zona='oral'),
    'junior': ProductLineInfo(necesidad='dolor_infantil', zona='oral'),
}

# THROMBOCID - Circulación/hematomas
THROMBOCID_LINES = {
    'forte': ProductLineInfo(necesidad='hematomas', zona='corporal'),
    'pomada': ProductLineInfo(necesidad='hematomas', zona='corporal'),
}

# DORMIDINA - Sueño
DORMIDINA_LINES = {
    'comprimidos': ProductLineInfo(necesidad='insomnio', zona='oral'),
    'doxilamina': ProductLineInfo(necesidad='insomnio', zona='oral'),
}

# BUCOMAX - Dolor de garganta
BUCOMAX_LINES = {
    'lidocaina': ProductLineInfo(necesidad='dolor_garganta', zona='bucofaringea'),
    'pastillas': ProductLineInfo(necesidad='dolor_garganta', zona='bucofaringea'),
}

# DULCOLAXO - Laxante
DULCOLAXO_LINES = {
    'bisacodilo': ProductLineInfo(necesidad='estrenimiento', zona='digestivo'),
    'supositorios': ProductLineInfo(necesidad='estrenimiento', zona='rectal'),
}

# GINE-CANESTEN - Antifúngico vaginal
GINECANESTEN_LINES = {
    'crema': ProductLineInfo(necesidad='candidiasis', zona='vaginal'),
    'ovulos': ProductLineInfo(necesidad='candidiasis', zona='vaginal'),
}


# =============================================================================
# NUEVAS MARCAS (2025-12-17) - Issue #446 Expansión
# =============================================================================

# NEUTROGENA - Solar y Corporal
NEUTROGENA_LINES = {
    'hydro boost': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'beach defense': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
    'ultra sheer': ProductLineInfo(necesidad='proteccion_solar', zona='facial'),
    'norwegian formula': ProductLineInfo(necesidad='piel_muy_seca', zona='corporal'),
}

# SENSODYNE - Dental Sensibilidad
SENSODYNE_LINES = {
    'rapid': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
    'repair': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
    'pronamel': ProductLineInfo(necesidad='esmalte_dental', zona='dental'),
    'complete': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
}

# PARODONTAX - Dental Encías
PARODONTAX_LINES = {
    'original': ProductLineInfo(necesidad='encias', zona='dental'),
    'complete': ProductLineInfo(necesidad='encias', zona='dental'),
    'extra fresh': ProductLineInfo(necesidad='encias', zona='dental'),
}

# AQUILEA - Suplementos
AQUILEA_LINES = {
    'sueno': ProductLineInfo(necesidad='sueno_insomnio', zona='oral'),
    'magnesio': ProductLineInfo(necesidad='magnesio', zona='oral'),
    'colageno': ProductLineInfo(necesidad='colageno', zona='oral'),
    'articulaciones': ProductLineInfo(necesidad='articulaciones', zona='oral'),
    'inmuno': ProductLineInfo(necesidad='defensas', zona='oral'),
    'piernas ligeras': ProductLineInfo(necesidad='circulacion', zona='oral'),
    'stagutt': ProductLineInfo(necesidad='circulacion', zona='oral'),
}

# ARKOPHARMA - Fitoterapia
ARKOPHARMA_LINES = {
    'arkocapsulas': ProductLineInfo(necesidad='fitoterapia', zona='oral'),
    'arkovital': ProductLineInfo(necesidad='vitaminas_general', zona='oral'),
    'arkofluido': ProductLineInfo(necesidad='detox', zona='oral'),
    'arkorelax': ProductLineInfo(necesidad='sueno_insomnio', zona='oral'),
}

# SOLGAR - Vitaminas Premium
SOLGAR_LINES = {
    'vitamina c': ProductLineInfo(necesidad='defensas', zona='oral'),
    'vitamina d': ProductLineInfo(necesidad='vitamina_d', zona='oral'),
    'omega 3': ProductLineInfo(necesidad='omega3', zona='oral'),
    'magnesio': ProductLineInfo(necesidad='magnesio', zona='oral'),
    'colageno': ProductLineInfo(necesidad='colageno', zona='oral'),
}

# CENTRUM - Multivitaminas
CENTRUM_LINES = {
    'complete': ProductLineInfo(necesidad='vitaminas_general', zona='oral'),
    'women': ProductLineInfo(necesidad='vitaminas_general', zona='oral'),
    'men': ProductLineInfo(necesidad='vitaminas_general', zona='oral'),
    'select': ProductLineInfo(necesidad='vitaminas_general', zona='oral'),
}

# SUPRADYN - Energía
SUPRADYN_LINES = {
    'activo': ProductLineInfo(necesidad='energia_cansancio', zona='oral'),
    'energy': ProductLineInfo(necesidad='energia_cansancio', zona='oral'),
    'memoria': ProductLineInfo(necesidad='memoria', zona='oral'),
    'junior': ProductLineInfo(necesidad='vitaminas_infantil', zona='oral'),
}

# NUXE - Antiedad Natural
NUXE_LINES = {
    'huile prodigieuse': ProductLineInfo(necesidad='hidratacion_corporal', zona='corporal'),
    'nuxuriance': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'creme fraiche': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'merveillance': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
}

# CAUDALIE - Vinoterapia
CAUDALIE_LINES = {
    'vinoperfect': ProductLineInfo(necesidad='manchas', zona='facial'),
    'resveratrol': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'premier cru': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'vinosource': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
}

# FILORGA - Antiedad Premium
FILORGA_LINES = {
    'time-filler': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
    'optim-eyes': ProductLineInfo(necesidad='contorno_ojos', zona='facial'),
    'hydra-hyal': ProductLineInfo(necesidad='hidratacion_facial', zona='facial'),
    'ncef': ProductLineInfo(necesidad='arrugas_antiedad', zona='facial'),
}

# URIAGE - Agua Termal
URIAGE_LINES = {
    'eau thermale': ProductLineInfo(necesidad='piel_sensible', zona='facial'),
    'hyseac': ProductLineInfo(necesidad='acne', zona='facial'),
    'xemose': ProductLineInfo(necesidad='piel_atopica', zona='corporal'),
    'bariesun': ProductLineInfo(necesidad='proteccion_solar', zona='corporal'),
}

# A-DERMA - Avena
ADERMA_LINES = {
    'exomega': ProductLineInfo(necesidad='piel_atopica', zona='corporal'),
    'epitheliale': ProductLineInfo(necesidad='cicatrizacion', zona='corporal'),
    'rheacalm': ProductLineInfo(necesidad='piel_sensible', zona='facial'),
    'phys-ac': ProductLineInfo(necesidad='acne', zona='facial'),
}

# PHB - Dental
PHB_LINES = {
    'total': ProductLineInfo(necesidad='higiene_dental', zona='dental'),
    'white': ProductLineInfo(necesidad='blanqueamiento', zona='dental'),
    'sensitive': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
}

# VITIS - Dental Encías
VITIS_LINES = {
    'encias': ProductLineInfo(necesidad='encias', zona='dental'),
    'sensible': ProductLineInfo(necesidad='sensibilidad_dental', zona='dental'),
    'blanqueadora': ProductLineInfo(necesidad='blanqueamiento', zona='dental'),
    'orthodontic': ProductLineInfo(necesidad='higiene_dental', zona='dental'),
}

# FLUOCARIL - Dental Caries
FLUOCARIL_LINES = {
    'bi-fluore': ProductLineInfo(necesidad='caries', zona='dental'),
    'noche': ProductLineInfo(necesidad='caries', zona='dental'),
    'blanqueador': ProductLineInfo(necesidad='blanqueamiento', zona='dental'),
}

# =============================================================================
# SECONDARY BRANDS (2025-12-17) - Issue #446 Fase 2
# Validated from sales data: €15,000+ combined sales
# =============================================================================

# NUTRIBEN - Alimentación Bebé (€2,613)
NUTRIBEN_LINES = {
    'innova': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'leche': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'potitos': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'papillas': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
    'innova zero': ProductLineInfo(necesidad='alimentacion_bebe', zona='infantil'),
}

# FISIOCREM - Dolor muscular (€2,399)
FISIOCREM_LINES = {
    'solugel': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
    'spray': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
    'active': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
}

# COMPEED - Heridas/Ampollas (€2,113)
COMPEED_LINES = {
    'ampollas': ProductLineInfo(necesidad='heridas_apositos', zona='pies'),
    'callos': ProductLineInfo(necesidad='heridas_apositos', zona='pies'),
    'herpes': ProductLineInfo(necesidad='herpes_labial', zona='facial'),
    'juanetes': ProductLineInfo(necesidad='heridas_apositos', zona='pies'),
}

# FILVIT - Pediculosis (€1,792)
FILVIT_LINES = {
    'kit': ProductLineInfo(necesidad='pediculosis', zona='capilar'),
    'locion': ProductLineInfo(necesidad='pediculosis', zona='capilar'),
    'champu': ProductLineInfo(necesidad='pediculosis', zona='capilar'),
    'spray': ProductLineInfo(necesidad='pediculosis', zona='capilar'),
}

# LACTOFLORA - Probióticos (€1,582)
LACTOFLORA_LINES = {
    'intestinal': ProductLineInfo(necesidad='flora_intestinal', zona='digestivo'),
    'intimo': ProductLineInfo(necesidad='higiene_intima', zona='intimo'),
    'protector': ProductLineInfo(necesidad='flora_intestinal', zona='digestivo'),
    'kids': ProductLineInfo(necesidad='flora_intestinal', zona='infantil'),
}

# AFTERBITE - Picaduras (€1,552)
AFTERBITE_LINES = {
    'gel': ProductLineInfo(necesidad='picaduras', zona='corporal'),
    'xtreme': ProductLineInfo(necesidad='picaduras', zona='corporal'),
    'kids': ProductLineInfo(necesidad='picaduras', zona='infantil'),
    'original': ProductLineInfo(necesidad='picaduras', zona='corporal'),
}

# MUSTELA - Bebé Dermofarmacia (€1,536)
MUSTELA_LINES = {
    'stelatopia': ProductLineInfo(necesidad='piel_atopica', zona='infantil'),
    'bebe': ProductLineInfo(necesidad='higiene_bebe', zona='infantil'),
    'cicastela': ProductLineInfo(necesidad='heridas_apositos', zona='infantil'),
    'hydra': ProductLineInfo(necesidad='hidratacion', zona='infantil'),
    'solar': ProductLineInfo(necesidad='proteccion_solar', zona='infantil'),
    'maternidad': ProductLineInfo(necesidad='maternidad', zona='corporal'),
}

# DODOT - Pañales (€1,371)
DODOT_LINES = {
    'sensitive': ProductLineInfo(necesidad='higiene_bebe', zona='infantil'),
    'activity': ProductLineInfo(necesidad='higiene_bebe', zona='infantil'),
    'pro': ProductLineInfo(necesidad='higiene_bebe', zona='infantil'),
    'aqua': ProductLineInfo(necesidad='higiene_bebe', zona='infantil'),
}

# SUAVINEX - Bebé/Lactancia (adicional)
SUAVINEX_LINES = {
    'zero': ProductLineInfo(necesidad='lactancia', zona='infantil'),
    'fusion': ProductLineInfo(necesidad='lactancia', zona='infantil'),
    'tetina': ProductLineInfo(necesidad='lactancia', zona='infantil'),
    'chupete': ProductLineInfo(necesidad='confort_bebe', zona='infantil'),
}

# HANSAPLAST - Heridas (adicional)
HANSAPLAST_LINES = {
    'sensitive': ProductLineInfo(necesidad='heridas_apositos', zona='corporal'),
    'classic': ProductLineInfo(necesidad='heridas_apositos', zona='corporal'),
    'sport': ProductLineInfo(necesidad='heridas_apositos', zona='corporal'),
    'aqua protect': ProductLineInfo(necesidad='heridas_apositos', zona='corporal'),
}


# =============================================================================
# NUEVAS MARCAS (2025-12-17) - Issue #446 Fase 2 Expansión
# Identificadas por análisis de ventas de productos sin marca detectada
# =============================================================================

# NORMOGRIP - Antigripal Normon (€3,419)
NORMOGRIP_LINES = {
    'antitusivo': ProductLineInfo(necesidad='gripe_resfriado', zona='oral'),
    'granulado': ProductLineInfo(necesidad='gripe_resfriado', zona='oral'),
}

# NORMOTUS - Antitusivo Normon (€1,583)
NORMOTUS_LINES = {
    'expectorante': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='oral'),
    'antitusivo': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='oral'),
}

# FRIONEX - Mucolítico (€3,264)
FRIONEX_LINES = {
    'efervescente': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='oral'),
}

# OTOCERUM - Gotas óticas (€2,192)
OTOCERUM_LINES = {
    'gotas': ProductLineInfo(necesidad='higiene_oidos', zona='auricular'),
}

# MEDEBIOTIN - Biotina (€1,599)
MEDEBIOTIN_LINES = {
    'fuerte': ProductLineInfo(necesidad='caida_cabello', zona='oral'),
}

# REFLEX - Spray muscular (€1,547)
REFLEX_LINES = {
    'aerosol': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
    'spray': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
}

# DICLOKERN - Diclofenaco Kern (€1,527)
DICLOKERN_LINES = {
    'forte': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
    'spray': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
    'gel': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
}

# ESPIDIDOL - Ibuprofeno rápido (€1,477)
ESPIDIDOL_LINES = {
    'granulado': ProductLineInfo(necesidad='dolor', zona='oral'),
}

# HIDROXIL - Vitaminas B (€1,453)
HIDROXIL_LINES = {
    'b1-b6-b12': ProductLineInfo(necesidad='vitaminas_general', zona='oral'),
}

# ELLAONE - Anticoncepción emergencia (€1,419)
ELLAONE_LINES = {
    'comprimido': ProductLineInfo(necesidad='anticoncepcion_emergencia', zona='oral'),
}

# FLUTOX - Antitusivo (€1,289)
FLUTOX_LINES = {
    'jarabe': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='oral'),
}

# VARILIV - Circulación (€1,234)
VARILIV_LINES = {
    'diosmina': ProductLineInfo(necesidad='circulacion_piernas', zona='oral'),
}

# SALVACOLINA - Antidiarreico (€1,159)
SALVACOLINA_LINES = {
    'comprimidos': ProductLineInfo(necesidad='diarrea', zona='oral'),
}

# ZARBEES - Productos naturales (€1,130)
ZARBEES_LINES = {
    'jarabe': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='oral'),
    'kids': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='infantil'),
}

# TAVONIN - Circulación (€1,087)
TAVONIN_LINES = {
    'comprimidos': ProductLineInfo(necesidad='circulacion_piernas', zona='oral'),
}

# VENOVID - Circulación (€1,048)
VENOVID_LINES = {
    'comprimidos': ProductLineInfo(necesidad='circulacion_piernas', zona='oral'),
}

# NUADHA - Omega DHA (€975)
NUADHA_LINES = {
    '1000': ProductLineInfo(necesidad='omega3', zona='oral'),
    'dha': ProductLineInfo(necesidad='omega3', zona='oral'),
}

# NATALBEN - Embarazo (€919)
NATALBEN_LINES = {
    'supra': ProductLineInfo(necesidad='embarazo_prenatal', zona='oral'),
    'preconceptivo': ProductLineInfo(necesidad='fertilidad', zona='oral'),
    'lactancia': ProductLineInfo(necesidad='lactancia', zona='oral'),
}

# MICRALAX - Laxantes (€908)
MICRALAX_LINES = {
    'microenema': ProductLineInfo(necesidad='estrenimiento', zona='rectal'),
    'citrato': ProductLineInfo(necesidad='estrenimiento', zona='rectal'),
}

# AERO RED - Gases (€893)
AERORED_LINES = {
    'masticables': ProductLineInfo(necesidad='gases_digestion', zona='oral'),
    'gotas': ProductLineInfo(necesidad='gases_digestion', zona='oral'),
}

# FISIOGEN - Hierro (€880)
FISIOGEN_LINES = {
    'ferro': ProductLineInfo(necesidad='hierro', zona='oral'),
    'forte': ProductLineInfo(necesidad='hierro', zona='oral'),
}

# FRESUBIN - Nutrición clínica Fresenius (€871)
FRESUBIN_LINES = {
    'creme': ProductLineInfo(necesidad='nutricion_clinica', zona='oral'),
    '2kcal': ProductLineInfo(necesidad='nutricion_clinica', zona='oral'),
    'protein': ProductLineInfo(necesidad='nutricion_clinica', zona='oral'),
}

# FONTACTIV - Nutrición clínica (€862)
FONTACTIV_LINES = {
    'forte': ProductLineInfo(necesidad='nutricion_clinica', zona='oral'),
    'complete': ProductLineInfo(necesidad='nutricion_clinica', zona='oral'),
}

# CINFAMUCOL - Mucolítico Cinfa (€858)
CINFAMUCOL_LINES = {
    'acetilcisteina': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='oral'),
    'forte': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='oral'),
}

# CINFADOL - Antiinflamatorio Cinfa
CINFADOL_LINES = {
    'gel': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
}

# CINFATOS - Antitusivo Cinfa
CINFATOS_LINES = {
    'pastillas': ProductLineInfo(necesidad='mucosidad_respiratoria', zona='bucofaringea'),
}

# MELILAX - Laxante natural (€853)
MELILAX_LINES = {
    'adult': ProductLineInfo(necesidad='estrenimiento', zona='rectal'),
    'pediatric': ProductLineInfo(necesidad='estrenimiento', zona='rectal'),
}

# SEIDIVID - Fertilidad (€826)
SEIDIVID_LINES = {
    'ferty': ProductLineInfo(necesidad='fertilidad', zona='oral'),
}

# FLOGOPROFEN - Antiinflamatorio (€799)
FLOGOPROFEN_LINES = {
    'gel': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
    'spray': ProductLineInfo(necesidad='dolor_muscular', zona='corporal'),
}

# DIFENADOL - Ibuprofeno (€793)
DIFENADOL_LINES = {
    'comprimidos': ProductLineInfo(necesidad='dolor', zona='oral'),
}

# FLAVIA - Menopausia (€782)
FLAVIA_LINES = {
    'nocta': ProductLineInfo(necesidad='menopausia', zona='oral'),
    'plus': ProductLineInfo(necesidad='menopausia', zona='oral'),
}

# ACTRON - Analgésico (€774)
ACTRON_LINES = {
    'compuesto': ProductLineInfo(necesidad='dolor', zona='oral'),
}

# IBUFEN - Ibuprofeno (€767)
IBUFEN_LINES = {
    'comprimidos': ProductLineInfo(necesidad='dolor', zona='oral'),
}

# OMEREFLUX - Omeprazol (€760)
OMEREFLUX_LINES = {
    'capsulas': ProductLineInfo(necesidad='acidez_reflujo', zona='oral'),
}

# EXELVIT - Embarazo (€758)
EXELVIT_LINES = {
    'esencial': ProductLineInfo(necesidad='embarazo_prenatal', zona='oral'),
}

# RELIVE - Ojos secos (€756)
RELIVE_LINES = {
    'monodosis': ProductLineInfo(necesidad='ojo_seco', zona='ocular'),
}

# ZOVIDUO - Herpes (€739)
ZOVIDUO_LINES = {
    'crema': ProductLineInfo(necesidad='herpes_labial', zona='labial'),
}

# OPTIBEN - Ocular (verificar si ya existe)
OPTIBEN_LINES = {
    'ojos secos': ProductLineInfo(necesidad='ojo_seco', zona='ocular'),
    'irritados': ProductLineInfo(necesidad='irritacion_ocular', zona='ocular'),
}

# LINDOR - Incontinencia/cremas (€862)
LINDOR_LINES = {
    'crema': ProductLineInfo(necesidad='dermatitis_panal', zona='corporal'),
    'protectora': ProductLineInfo(necesidad='incontinencia', zona='corporal'),
}

# KABI - Nutrición clínica (€1,423)
KABI_LINES = {
    'control': ProductLineInfo(necesidad='nutricion_clinica', zona='oral'),
    'protein': ProductLineInfo(necesidad='nutricion_clinica', zona='oral'),
}

# AMINOCOMPLEX/AMINOHAIR - Suplementos aminoácidos (€2,354 combinado)
AMINOCOMPLEX_LINES = {
    'capsulas': ProductLineInfo(necesidad='caida_cabello', zona='oral'),
}

AMINOHAIR_LINES = {
    'capsulas': ProductLineInfo(necesidad='caida_cabello', zona='oral'),
}


# =============================================================================
# ALL BRANDS MAPPING
# =============================================================================

ALL_BRAND_LINES: Dict[str, Dict[str, ProductLineInfo]] = {
    # Dermofarmacia
    'isdin': ISDIN_LINES,
    'heliocare': HELIOCARE_LINES,
    'bioderma': BIODERMA_LINES,
    'la roche posay': LAROCHE_LINES,
    'la roche-posay': LAROCHE_LINES,
    'avene': AVENE_LINES,
    'eucerin': EUCERIN_LINES,
    'sesderma': SESDERMA_LINES,
    'cerave': CERAVE_LINES,
    # Dermofarmacia premium (2025-12-16)
    'medik8': MEDIK8_LINES,
    'esthederm': ESTHEDERM_LINES,
    'cumlaude': CUMLAUDE_LINES,
    'klorane': KLORANE_LINES,
    'be+': BEPLUS_LINES,
    # Dental
    'gum': GUM_LINES,
    'lacer': LACER_LINES,
    'kin': KIN_LINES,
    'corega': COREGA_LINES,
    'kukident': KUKIDENT_LINES,
    'bexident': BEXIDENT_LINES,  # 2025-12-16
    # Bebé
    'blemil': BLEMIL_LINES,
    'pedialac': PEDIALAC_LINES,  # 2025-12-16
    # Capilar
    'olistic': OLISTIC_LINES,
    # Ocular
    'thealoz': THEALOZ_LINES,
    # Suplementos
    'ivb': IVB_LINES,
    'ivb wellness': IVB_LINES,
    'ns': NS_LINES,
    'nhco': NHCO_LINES,  # 2025-12-16
    # Dermofarmacia especializada
    'gh': GH_LINES,
    'gema herrerias': GH_LINES,
    # OTC Farmacia (2025-12-16)
    'voltadol': VOLTADOL_LINES,
    'frenadol': FRENADOL_LINES,
    'fluimucil': FLUIMUCIL_LINES,
    'cristalmina': CRISTALMINA_LINES,
    'almax': ALMAX_LINES,
    'zzzquil': ZZZQUIL_LINES,
    'ibudol': IBUDOL_LINES,
    'respibien': RESPIBIEN_LINES,
    # Marcas adicionales (2025-12-16 Fase 2)
    'mitosyl': MITOSYL_LINES,
    'vicks': VICKS_LINES,
    'vaporub': VICKS_LINES,  # alias
    'oral-b': ORALB_LINES,
    'oral b': ORALB_LINES,   # alias sin guión
    'omron': OMRON_LINES,
    'predictor': PREDICTOR_LINES,
    'hylo': HYLO_LINES,
    'bivos': BIVOS_LINES,
    'dulcosoft': DULCOSOFT_LINES,
    'megalevure': MEGALEVURE_LINES,
    'nutracel': NUTRACEL_LINES,
    'arturo alba': ARTUROALBA_LINES,
    'dr arthouros': ARTUROALBA_LINES,  # alias
    'dr arthouros alba': ARTUROALBA_LINES,  # alias completo
    'daflon': DAFLON_LINES,
    'fortasec': FORTASEC_LINES,
    'nurofen': NUROFEN_LINES,
    'thrombocid': THROMBOCID_LINES,
    'dormidina': DORMIDINA_LINES,
    'bucomax': BUCOMAX_LINES,
    'dulcolaxo': DULCOLAXO_LINES,
    'gine-canesten': GINECANESTEN_LINES,
    'gine canesten': GINECANESTEN_LINES,  # alias sin guión
    # Nuevas marcas (2025-12-17) - Issue #446 Expansión
    'neutrogena': NEUTROGENA_LINES,
    'sensodyne': SENSODYNE_LINES,
    'parodontax': PARODONTAX_LINES,
    'aquilea': AQUILEA_LINES,
    'arkopharma': ARKOPHARMA_LINES,
    'arko': ARKOPHARMA_LINES,  # alias
    'arkocapsulas': ARKOPHARMA_LINES,  # alias
    'arkovital': ARKOPHARMA_LINES,  # alias
    'solgar': SOLGAR_LINES,
    'centrum': CENTRUM_LINES,
    'multicentrum': CENTRUM_LINES,  # alias
    'supradyn': SUPRADYN_LINES,
    'nuxe': NUXE_LINES,
    'caudalie': CAUDALIE_LINES,
    'filorga': FILORGA_LINES,
    'uriage': URIAGE_LINES,
    'a-derma': ADERMA_LINES,
    'aderma': ADERMA_LINES,  # alias
    'phb': PHB_LINES,
    'vitis': VITIS_LINES,
    'fluocaril': FLUOCARIL_LINES,
    # Secondary brands (2025-12-17) - Issue #446 Fase 2
    'nutriben': NUTRIBEN_LINES,
    'nutribén': NUTRIBEN_LINES,  # con tilde
    'fisiocrem': FISIOCREM_LINES,
    'compeed': COMPEED_LINES,
    'filvit': FILVIT_LINES,
    'lactoflora': LACTOFLORA_LINES,
    'afterbite': AFTERBITE_LINES,
    'after bite': AFTERBITE_LINES,  # alias con espacio
    'mustela': MUSTELA_LINES,
    'dodot': DODOT_LINES,
    'suavinex': SUAVINEX_LINES,
    'hansaplast': HANSAPLAST_LINES,
    # Nuevas marcas (2025-12-17) - Issue #446 Fase 2 Expansión
    'normogrip': NORMOGRIP_LINES,
    'normotus': NORMOTUS_LINES,
    'frionex': FRIONEX_LINES,
    'otocerum': OTOCERUM_LINES,
    'medebiotin': MEDEBIOTIN_LINES,
    'reflex': REFLEX_LINES,
    'diclokern': DICLOKERN_LINES,
    'espididol': ESPIDIDOL_LINES,
    'hidroxil': HIDROXIL_LINES,
    'ellaone': ELLAONE_LINES,
    'flutox': FLUTOX_LINES,
    'variliv': VARILIV_LINES,
    'salvacolina': SALVACOLINA_LINES,
    'zarbees': ZARBEES_LINES,
    'tavonin': TAVONIN_LINES,
    'venovid': VENOVID_LINES,
    'nuadha': NUADHA_LINES,
    'natalben': NATALBEN_LINES,
    'micralax': MICRALAX_LINES,
    'aero red': AERORED_LINES,
    'aerored': AERORED_LINES,  # alias sin espacio
    'fisiogen': FISIOGEN_LINES,
    'fresubin': FRESUBIN_LINES,
    'fontactiv': FONTACTIV_LINES,
    'cinfamucol': CINFAMUCOL_LINES,
    'cinfadol': CINFADOL_LINES,
    'cinfatos': CINFATOS_LINES,
    'melilax': MELILAX_LINES,
    'seidivid': SEIDIVID_LINES,
    'flogoprofen': FLOGOPROFEN_LINES,
    'difenadol': DIFENADOL_LINES,
    'flavia': FLAVIA_LINES,
    'actron': ACTRON_LINES,
    'ibufen': IBUFEN_LINES,
    'omereflux': OMEREFLUX_LINES,
    'exelvit': EXELVIT_LINES,
    'relive': RELIVE_LINES,
    'zoviduo': ZOVIDUO_LINES,
    'optiben': OPTIBEN_LINES,
    'lindor': LINDOR_LINES,
    'kabi': KABI_LINES,
    'aminocomplex': AMINOCOMPLEX_LINES,
    'aminohair': AMINOHAIR_LINES,
}


# Orden de prioridad para detección de marca (más específicas primero)
BRAND_PRIORITY = [
    # Multi-palabra primero (más específicas)
    'la roche posay', 'la roche-posay', 'ivb wellness', 'gema herrerias',
    'arturo alba', 'dr arthouros alba', 'dr arthouros',
    'gine-canesten', 'gine canesten', 'oral-b', 'oral b', 'a-derma',
    'huile prodigieuse',  # Nuxe product line
    # Dermofarmacia
    'heliocare', 'isdin', 'bioderma', 'eucerin', 'avene',
    'sesderma', 'cerave', 'neutrogena',  # 🆕
    # Dermofarmacia premium
    'medik8', 'esthederm', 'cumlaude', 'klorane', 'be+',
    'nuxe', 'caudalie', 'filorga', 'uriage', 'aderma',  # 🆕
    # Dental
    'sensodyne', 'parodontax',  # 🆕 Primero por ser más específicos
    'gum', 'lacer', 'kukident', 'corega', 'kin', 'bexident',
    'phb', 'vitis', 'fluocaril',  # 🆕
    # Capilar
    'olistic',
    # Suplementos
    'arkopharma', 'arkocapsulas', 'arkovital',  # 🆕 Arkopharma variantes
    'aquilea', 'solgar', 'centrum', 'multicentrum', 'supradyn',  # 🆕
    'ivb', 'ns', 'nhco',
    # Dermofarmacia especializada
    'gh', 'nutracel',
    # Ocular
    'thealoz', 'hylo',
    # Bebé
    'blemil', 'pedialac', 'mitosyl', 'bivos',
    'nutriben', 'nutribén', 'mustela', 'dodot', 'suavinex',  # 🆕 Secondary
    # OTC Farmacia
    'voltadol', 'frenadol', 'fluimucil', 'cristalmina',
    'almax', 'zzzquil', 'ibudol', 'respibien',
    'vicks', 'vaporub', 'omron', 'predictor',
    'dulcosoft', 'megalevure', 'daflon', 'fortasec',
    'nurofen', 'thrombocid', 'dormidina', 'bucomax', 'dulcolaxo',
    # Secundarias Issue #446 Fase 2
    'fisiocrem', 'compeed', 'filvit', 'lactoflora',
    'afterbite', 'after bite', 'hansaplast',
    # Nuevas marcas Issue #446 Fase 2 Expansión (2025-12-17)
    'normogrip', 'normotus', 'frionex', 'otocerum', 'medebiotin',
    'reflex', 'diclokern', 'espididol', 'hidroxil', 'ellaone',
    'flutox', 'variliv', 'salvacolina', 'zarbees', 'tavonin',
    'venovid', 'nuadha', 'natalben', 'micralax', 'aero red', 'aerored',
    'fisiogen', 'fresubin', 'fontactiv', 'cinfamucol', 'cinfadol', 'cinfatos',
    'melilax', 'seidivid', 'flogoprofen', 'difenadol', 'flavia',
    'actron', 'ibufen', 'omereflux', 'exelvit', 'relive', 'zoviduo',
    'optiben', 'lindor', 'kabi', 'aminocomplex', 'aminohair',
]


# =============================================================================
# SIMPLE BRANDS (2025-12-17) - Issue #446 Fase 2 - 500 Brand Expansion
# Mapeo simplificado: marca → necesidad (sin líneas de producto detalladas)
# Estas marcas se detectan pero no tienen líneas de producto configuradas
# =============================================================================

SIMPLE_BRANDS: Dict[str, str] = {
    # ACIDEZ/REFLUJO
    'acidrine': 'acidez_reflujo', 'alka-seltzer': 'acidez_reflujo', 'arcid': 'acidez_reflujo',
    'digest': 'acidez_reflujo', 'esophacare': 'acidez_reflujo', 'esoxx': 'acidez_reflujo',
    'gavidigest': 'acidez_reflujo', 'gaviscon': 'acidez_reflujo', 'iberogast': 'acidez_reflujo',
    'maalox': 'acidez_reflujo', 'motilium': 'acidez_reflujo', 'mylanta': 'acidez_reflujo',
    'neobianacid': 'acidez_reflujo', 'oliovita gastro': 'acidez_reflujo', 'omekaste': 'acidez_reflujo',
    'omeprazol': 'acidez_reflujo', 'pantoprazol': 'acidez_reflujo', 'phosphalugel': 'acidez_reflujo',
    'primperan': 'acidez_reflujo', 'rennie': 'acidez_reflujo', 'sal de fruta': 'acidez_reflujo',
    'eno': 'acidez_reflujo', 'lansoprazol': 'acidez_reflujo',

    # ACNÉ
    'acmed': 'acne', 'acniben': 'acne', 'azeane': 'acne', 'benzac': 'acne', 'boreade': 'acne',
    'cleanance': 'acne', 'keracnyl': 'acne', 'martiderm acniover': 'acne', 'normaderm': 'acne',
    'panoxyl': 'acne', 'peroxiben': 'acne', 'eucerin dermopure': 'acne', 'sesderma azelac': 'acne',

    # AFTAS
    'afta repair': 'aftas', 'aftaclear': 'aftas', 'aftasone': 'aftas', 'aloclair': 'aftas',
    'bexident aftas': 'aftas', 'aftamer': 'aftas',

    # ALERGIA
    'abrilia': 'alergia', 'aerius': 'alergia', 'atarax': 'alergia', 'bactil': 'alergia',
    'bilaxten': 'alergia', 'cetirizina': 'alergia', 'clarityne': 'alergia', 'ebastel': 'alergia',
    'loratadina': 'alergia', 'polaramine': 'alergia', 'reactine': 'alergia', 'rinoebastel': 'alergia',
    'telfast': 'alergia', 'xazal': 'alergia', 'zyrtec': 'alergia',

    # ALIMENTACIÓN BEBÉ
    'almiron': 'alimentacion_bebe', 'enfamil': 'alimentacion_bebe', 'gerber': 'alimentacion_bebe',
    'hero baby': 'alimentacion_bebe', 'hipp': 'alimentacion_bebe', 'nan': 'alimentacion_bebe',
    'nidina': 'alimentacion_bebe', 'novalac': 'alimentacion_bebe', 'smileat': 'alimentacion_bebe',

    # ARRUGAS/ANTIEDAD
    'acm': 'arrugas_antiedad', 'anne moller': 'arrugas_antiedad', 'babaria': 'arrugas_antiedad',
    'bella aurora': 'arrugas_antiedad', 'carita': 'arrugas_antiedad', 'collistar': 'arrugas_antiedad',
    'd\'shila': 'arrugas_antiedad', 'dshila': 'arrugas_antiedad', 'deliplus': 'arrugas_antiedad',
    'embryolisse': 'arrugas_antiedad', 'flavo-c': 'arrugas_antiedad', 'fridda dorsch': 'arrugas_antiedad',
    'hyalu b5': 'arrugas_antiedad', 'isis pharma': 'arrugas_antiedad', 'lancaster': 'arrugas_antiedad',
    'leti': 'arrugas_antiedad', 'liftactiv': 'arrugas_antiedad', 'massada': 'arrugas_antiedad',
    'neovadiol': 'arrugas_antiedad', 'nezeni': 'arrugas_antiedad', 'noreva': 'arrugas_antiedad',
    'olyan farma': 'arrugas_antiedad', 'ozoaqua': 'arrugas_antiedad', 'phyto': 'arrugas_antiedad',
    'pigmentclar': 'arrugas_antiedad', 'redermic': 'arrugas_antiedad', 'retincare': 'arrugas_antiedad',
    'singuladerm': 'arrugas_antiedad', 'talika': 'arrugas_antiedad', 'th pharma': 'arrugas_antiedad',
    'toleriane': 'arrugas_antiedad', 'topicrem': 'arrugas_antiedad', 'xhekpon': 'arrugas_antiedad',
    'effaclar': 'acne', 'sebium': 'acne', 'sensibio': 'piel_sensible', 'atoderm': 'piel_atopica',
    'cicabio': 'cicatrizacion', 'cicaplast': 'cicatrizacion', 'epitheliale': 'cicatrizacion',
    'cicalfate': 'cicatrizacion', 'skinceuticals': 'arrugas_antiedad',

    # ARTICULACIONES
    'aminocollax': 'articulaciones', 'aminoflex': 'articulaciones', 'arkoflex': 'articulaciones',
    'bonflex': 'articulaciones', 'boswellia': 'articulaciones', 'cartilago de tiburon': 'articulaciones',
    'colageno hidrolizado': 'articulaciones', 'colagenova': 'articulaciones', 'colnatur': 'articulaciones',
    'condroitina': 'articulaciones', 'curcuma': 'articulaciones', 'drasanvi colageno': 'articulaciones',
    'epaplus': 'articulaciones', 'glucosamina': 'articulaciones', 'harpagofito': 'articulaciones',
    'marine collagen': 'articulaciones', 'movial': 'articulaciones', 'msm': 'articulaciones',
    'peptan': 'articulaciones', 'siik': 'articulaciones', 'unique pink collagen': 'articulaciones',
    'vitanatur collagen': 'articulaciones', 'zentrum colageno': 'articulaciones',
    'ivb articare': 'articulaciones',

    # CAÍDA CABELLO
    'alpecin': 'caida_cabello', 'aminexil': 'caida_cabello', 'anacaps': 'caida_cabello',
    'arkocapil': 'caida_cabello', 'complidermol': 'caida_cabello', 'densiage': 'caida_cabello',
    'dercos aminexil': 'caida_cabello', 'forticea': 'caida_cabello', 'inneov': 'caida_cabello',
    'iraltone': 'caida_cabello', 'kerastase': 'caida_cabello', 'keratincaps': 'caida_cabello',
    'kerzo': 'caida_cabello', 'l\'oreal serioxyl': 'caida_cabello', 'lambdapil': 'caida_cabello',
    'lazartigue': 'caida_cabello', 'minoxidil vinas': 'caida_cabello', 'moncho moreno': 'caida_cabello',
    'nanogen': 'caida_cabello', 'neogenic': 'caida_cabello', 'nioxin': 'caida_cabello',
    'nuggela & sule': 'caida_cabello', 'nuggela': 'caida_cabello', 'oenobiol capilar': 'caida_cabello',
    'pilexil': 'caida_cabello', 'redken cerafill': 'caida_cabello', 'regaxidil': 'caida_cabello',
    'rene furterer': 'caida_cabello', 'revita': 'caida_cabello', 'svenson': 'caida_cabello',
    'triphasic': 'caida_cabello', 'valquer': 'caida_cabello', 'vitalfan': 'caida_cabello',
    'vichy dercos': 'caida_cabello', 'lacer pilopeptan': 'caida_cabello', 'isdin lambdapil': 'caida_cabello',

    # CIRCULACIÓN
    'antistax': 'circulacion_piernas', 'edemox': 'circulacion_piernas', 'flebinec': 'circulacion_piernas',
    'flebostasin': 'circulacion_piernas', 'ginkor': 'circulacion_piernas',
    'martiderm legvass': 'circulacion_piernas', 'veinotonyl': 'circulacion_piernas',
    'venen': 'circulacion_piernas', 'venoplus': 'circulacion_piernas', 'venoruton': 'circulacion_piernas',
    'venosmil': 'circulacion_piernas', 'aquilea piernas ligeras': 'circulacion_piernas',

    # CUIDADO LABIOS
    'avene cold cream labios': 'cuidado_labios', 'blistex': 'cuidado_labios', 'burts bees': 'cuidado_labios',
    'carmex': 'cuidado_labios', 'eos': 'cuidado_labios', 'himalaya': 'cuidado_labios',
    'labello': 'cuidado_labios', 'letibalm': 'cuidado_labios', 'repavar labial': 'cuidado_labios',
    'neutrogena labios': 'cuidado_labios', 'isdin labial': 'cuidado_labios',
    'eucerin lip active': 'cuidado_labios', 'vichy aqualia labios': 'cuidado_labios',
    'be+ med labios': 'cuidado_labios',

    # DIARREA
    'bioralsuero': 'diarrea', 'dialiv': 'diarrea', 'electrolit': 'diarrea', 'megasmect': 'diarrea',
    'probiobal': 'diarrea', 'sindiar': 'diarrea', 'smecta': 'diarrea', 'suero oral normon': 'diarrea',
    'sueroral': 'diarrea', 'tanagel': 'diarrea', 'tiorfan': 'diarrea', 'ultra-levura': 'diarrea',
    'ns suerobiotic': 'diarrea',

    # DOLOR
    'adolonta': 'dolor', 'antidol': 'dolor', 'apiretal': 'dolor', 'arnidol': 'dolor',
    'aspirina': 'dolor', 'calmazine': 'dolor', 'dalsy': 'dolor', 'dexketoprofeno': 'dolor',
    'diclofenaco topico': 'dolor_muscular', 'difenatil': 'dolor', 'dolocatil': 'dolor',
    'dolostop': 'dolor', 'dolovanz': 'dolor', 'efferalgan': 'dolor', 'enantyum': 'dolor',
    'endolex': 'dolor', 'gelocatil': 'dolor', 'geloprofen': 'dolor', 'hirudoid': 'hematomas',
    'ibuprofeno': 'dolor', 'neobrufen': 'dolor', 'paracetamol': 'dolor', 'radio salil': 'dolor_muscular',
    'termalgin': 'dolor',

    # DOLOR GARGANTA
    'angal': 'dolor_garganta', 'angileptol': 'dolor_garganta', 'anginovag': 'dolor_garganta',
    'bucomax': 'dolor_garganta', 'bucometasana': 'dolor_garganta', 'deratin': 'dolor_garganta', 'drill': 'dolor_garganta',
    'faringesic': 'dolor_garganta', 'faringospray': 'dolor_garganta', 'garydol': 'dolor_garganta',
    'juanolgar': 'dolor_garganta', 'lizipaina': 'dolor_garganta', 'lysopaine': 'dolor_garganta',
    'mebucaina': 'dolor_garganta', 'pharysol': 'dolor_garganta', 'praims': 'dolor_garganta',
    'propalgar': 'dolor_garganta', 'propoleo': 'dolor_garganta', 'propolis': 'dolor_garganta',
    'ricola': 'dolor_garganta', 'sediflu garganta': 'dolor_garganta', 'strefen': 'dolor_garganta',
    'strepsils': 'dolor_garganta', 'tantum verde': 'dolor_garganta', 'vivis': 'dolor_garganta',

    # DOLOR MUSCULAR TÓPICO
    'algesal': 'dolor_muscular', 'balsamo de tigre': 'dolor_muscular', 'bengue': 'dolor_muscular',
    'calmatel': 'dolor_muscular', 'counterpain': 'dolor_muscular', 'dolalgial': 'dolor_muscular',
    'fastum': 'dolor_muscular', 'feldene': 'dolor_muscular', 'hidrotelial luxoben': 'dolor_muscular',
    'kytta': 'dolor_muscular', 'menaven': 'dolor_muscular', 'tiger balm': 'dolor_muscular',
    'traumeel': 'dolor_muscular', 'voltaren emulgel': 'dolor_muscular', 'zenavan': 'dolor_muscular',

    # EMBARAZO
    'acido folico': 'embarazo_prenatal', 'femibion': 'embarazo_prenatal', 'fertifem': 'embarazo_prenatal',
    'fertybiotic': 'embarazo_prenatal', 'gestacare': 'embarazo_prenatal', 'gestagyn': 'embarazo_prenatal',
    'gynea': 'embarazo_prenatal', 'inofolic': 'embarazo_prenatal', 'lactanza': 'lactancia',
    'maternum': 'embarazo_prenatal', 'natele': 'embarazo_prenatal', 'ovusitol': 'fertilidad',
    'seidibion mater': 'embarazo_prenatal', 'vitafolin': 'embarazo_prenatal', 'yoduk': 'embarazo_prenatal',
    'solgar prenatal': 'embarazo_prenatal', 'multicentrum pre-mama': 'embarazo_prenatal',
    'suavinex natal': 'embarazo_prenatal',

    # ENERGÍA
    'aminoenergy': 'energia_cansancio', 'apiserum': 'energia_cansancio', 'astenolit': 'energia_cansancio',
    'berocca boost': 'energia_cansancio', 'black bee pharmacy': 'energia_cansancio',
    'energynat': 'energia_cansancio', 'guarana': 'energia_cansancio', 'jalea real': 'energia_cansancio',
    'leotron': 'energia_cansancio', 'maca': 'energia_cansancio', 'rhodiola': 'energia_cansancio',
    'royal vit': 'energia_cansancio', 'taurina': 'energia_cansancio', 'vitaminor': 'energia_cansancio',
    'vitanatur energia': 'energia_cansancio', 'vitasprint': 'energia_cansancio',
    'ginseng': 'energia_cansancio', 'arkocapsulas ginseng': 'energia_cansancio',

    # ESTREÑIMIENTO
    'agiolax': 'estrenimiento', 'aliviolas': 'estrenimiento', 'bekunis': 'estrenimiento',
    'casenlax': 'estrenimiento', 'duphalac': 'estrenimiento', 'emportal': 'estrenimiento',
    'evacuol': 'estrenimiento', 'fave de fuca': 'estrenimiento', 'fisioenema': 'estrenimiento',
    'glycilax': 'estrenimiento', 'lactulosa': 'estrenimiento', 'laxante salud': 'estrenimiento',
    'laxante bescansa': 'estrenimiento', 'linomed': 'estrenimiento', 'macrogol': 'estrenimiento',
    'movicol': 'estrenimiento', 'plantaben': 'estrenimiento', 'regulax': 'estrenimiento',
    'roha max': 'estrenimiento', 'supositorios glicerina': 'estrenimiento',
    'vilardell digest': 'estrenimiento',

    # ESTRÉS
    '5-htp': 'estres_ansiedad', 'ashwagandha': 'estres_ansiedad', 'gaba': 'estres_ansiedad',
    'l-teanina': 'estres_ansiedad', 'pasiflora': 'estres_ansiedad', 'relaxing': 'estres_ansiedad',
    'rescue remedy': 'estres_ansiedad', 'triptofano': 'estres_ansiedad',
    'ns equirelax': 'estres_ansiedad', 'aquilea tranquilidad': 'estres_ansiedad',
    'ivb stress off': 'estres_ansiedad',

    # FLORA INTESTINAL / PROBIÓTICOS
    'alflorex': 'flora_intestinal', 'arkoprobiotics': 'flora_intestinal', 'bioithas': 'flora_intestinal',
    'enterogermina': 'flora_intestinal', 'florabiotic': 'flora_intestinal', 'florastor': 'flora_intestinal',
    'kijimea': 'flora_intestinal', 'lactibiane': 'flora_intestinal', 'lamberts': 'vitaminas_general',
    'multibionta': 'flora_intestinal', 'optibac': 'flora_intestinal', 'pearls': 'flora_intestinal',
    'prodefen': 'flora_intestinal', 'produo': 'flora_intestinal', 'profaes4': 'flora_intestinal',
    'reuteri': 'flora_intestinal', 'rotagermine': 'flora_intestinal', 'sibok': 'flora_intestinal',
    'solaray': 'vitaminas_general', 'vitanatur': 'vitaminas_general', 'vivomixx': 'flora_intestinal',
    'vsl#3': 'flora_intestinal', 'yovis': 'flora_intestinal', 'aquilea probio': 'flora_intestinal',
    'supradyn probio': 'flora_intestinal',

    # GASES
    'benegast': 'gases_digestion', 'carbon vegetal': 'gases_digestion', 'digerflat': 'gases_digestion',
    'dimeticona': 'gases_digestion', 'fructaid': 'gases_digestion', 'fuca': 'gases_digestion',
    'lactasa': 'gases_digestion', 'lactoben': 'gases_digestion', 'naturflat': 'gases_digestion',
    'nutira': 'gases_digestion', 'simethicone': 'gases_digestion', 'aquilea gases': 'gases_digestion',

    # GRIPE/RESFRIADO
    'actigrip': 'gripe_resfriado', 'afrin': 'congestion_nasal', 'antigripine': 'gripe_resfriado',
    'aspirina complex': 'gripe_resfriado', 'bisolgrip': 'gripe_resfriado', 'couldina': 'gripe_resfriado',
    'flulenza': 'gripe_resfriado', 'gelocatil gripe': 'gripe_resfriado', 'grippal': 'gripe_resfriado',
    'ilvico': 'gripe_resfriado', 'inhalvicks': 'congestion_nasal', 'marimer': 'congestion_nasal',
    'nasalvicks': 'congestion_nasal', 'normonasal': 'congestion_nasal', 'otrivina': 'congestion_nasal',
    'pharmagrip': 'gripe_resfriado', 'physiomer': 'congestion_nasal', 'propalgina': 'gripe_resfriado',
    'respidina': 'congestion_nasal', 'rhinodouche': 'congestion_nasal', 'rhinomer': 'congestion_nasal',
    'rhinospray': 'congestion_nasal', 'sterimar': 'congestion_nasal', 'termalgin gripe': 'gripe_resfriado',
    'utabon': 'congestion_nasal', 'vincigrip': 'gripe_resfriado', 'vicks vaporub': 'balsamos_respiratorios',

    # HEMORROIDES
    'anso': 'hemorroides', 'detramax': 'hemorroides', 'faktu': 'hemorroides', 'hemoal': 'hemorroides',
    'hemorrane': 'hemorroides', 'proctoial': 'hemorroides', 'proctolog': 'hemorroides',
    'proctosan': 'hemorroides', 'ruscus llorens': 'hemorroides', 'sante verte hemoless': 'hemorroides',

    # HERIDAS
    'betadine': 'heridas_apositos', 'bio-oil': 'cicatrizacion', 'blastoestimulina': 'heridas_apositos',
    'contractubex': 'cicatrizacion', 'dermatix': 'cicatrizacion', 'iruxol': 'heridas_apositos',
    'kelo-cote': 'cicatrizacion', 'linitul': 'heridas_apositos', 'mederma': 'cicatrizacion',
    'repavar': 'cicatrizacion', 'trofolastin': 'cicatrizacion', 'tulgrasum': 'heridas_apositos',
    'suaviskin': 'cicatrizacion', 'lacer gel clorhexidina': 'heridas_apositos',

    # HERPES LABIAL
    'viruderm': 'herpes_labial', 'zovicrem': 'herpes_labial', 'zovirax': 'herpes_labial',
    'compeed herpes': 'herpes_labial',

    # HIDRATACIÓN CORPORAL
    'acofar': 'hidratacion_corporal', 'dove': 'hidratacion_corporal', 'farline': 'hidratacion_corporal',
    'genove': 'hidratacion_corporal', 'germisdin': 'higiene_corporal', 'inibsa': 'hidratacion_corporal',
    'johnson\'s': 'higiene_corporal', 'lactovit': 'hidratacion_corporal',
    'lipikar': 'piel_atopica', 'mussvital': 'hidratacion_corporal', 'nivea': 'hidratacion_corporal',
    'sanex': 'higiene_corporal', 'sebamed': 'piel_sensible', 'topialyse': 'piel_atopica',
    'ureadin': 'piel_seca', 'xemose': 'piel_atopica', 'xeracalm': 'piel_atopica',

    # HIERRO
    'fero-gradumet': 'hierro', 'ferroglicina': 'hierro', 'ferroprotect': 'hierro',
    'ferrosanol': 'hierro', 'floradix': 'hierro', 'floravital': 'hierro', 'iromax': 'hierro',
    'tardyferon': 'hierro', 'ivb ferroplus': 'hierro', 'fisiogen ferro': 'hierro',

    # HIGIENE BEBÉ
    'bepanthol baby': 'higiene_bebe', 'bio baby': 'higiene_bebe', 'cattier baby': 'higiene_bebe',
    'chelino': 'higiene_bebe', 'chicco': 'bebes_infantil', 'eryplast': 'dermatitis_panal',
    'halibut': 'dermatitis_panal', 'huggies': 'higiene_bebe', 'lutsine': 'dermatitis_panal',
    'moltex': 'higiene_bebe', 'pingo': 'higiene_bebe', 'weleda baby': 'higiene_bebe',
    'isdin baby naturals': 'higiene_bebe', 'nutraisdin': 'higiene_bebe', 'klorane bebe': 'higiene_bebe',
    'uriage bebe': 'higiene_bebe',

    # HIGIENE DENTAL
    'algasiv': 'protesis_dental', 'cb12': 'higiene_dental', 'clorhexidina': 'encias',
    'corsodyl': 'encias', 'curaprox': 'higiene_dental', 'dentaid': 'higiene_dental',
    'elmex': 'higiene_dental', 'halita': 'higiene_dental', 'interprox': 'interdental',
    'listerine': 'higiene_dental', 'meridol': 'encias', 'oraldine': 'higiene_dental',
    'oralkin': 'higiene_dental', 'perio-aid': 'encias', 'philips sonicare': 'higiene_dental',
    'tepe': 'interdental', 'waterpik': 'higiene_dental', 'kin oro': 'protesis_dental',
    'lacerfresh': 'higiene_dental', 'sensilacer': 'sensibilidad_dental', 'xerolacer': 'xerostomia',
    'gingilacer': 'encias',

    # HIGIENE ÍNTIMA
    'ainara': 'higiene_intima', 'chilly': 'higiene_intima', 'durex intima': 'higiene_intima',
    'enna': 'higiene_intima', 'feminella': 'higiene_intima', 'ginecanesten': 'candidiasis',
    'ginexin': 'higiene_intima', 'idracare': 'higiene_intima', 'intimina': 'higiene_intima',
    'intimo': 'higiene_intima', 'lactacyd': 'higiene_intima', 'ozointim': 'higiene_intima',
    'palomacare': 'higiene_intima', 'papilocare': 'higiene_intima', 'saforelle': 'higiene_intima',
    'saugella': 'higiene_intima', 'tantum rosa': 'higiene_intima', 'vagisil': 'higiene_intima',
    'zelesse': 'higiene_intima', 'isdin woman': 'higiene_intima',

    # HOMEOPATÍA
    'a. vogel': 'homeopatia', 'boiron': 'homeopatia', 'dhu': 'homeopatia', 'heel': 'homeopatia',
    'lehning': 'homeopatia', 'nelsons': 'homeopatia', 'oscillococcinum': 'homeopatia',
    'schwabe': 'homeopatia', 'weleda': 'homeopatia',

    # HONGOS
    'canesten': 'hongos_pies', 'farmafeet': 'hongos_pies', 'fungarest': 'hongos_pies',
    'funsol': 'hongos_pies', 'ketoconazol': 'hongos_pies', 'lamisil': 'hongos_pies',
    'mycoben': 'hongos_pies', 'po polvos pedicos': 'hongos_pies', 'terbinafina': 'hongos_pies',
    'zeliderm': 'hongos_pies',

    # INCONTINENCIA
    'abena': 'incontinencia', 'attends': 'incontinencia', 'ausonia discreet': 'incontinencia',
    'depend': 'incontinencia', 'hartmann': 'incontinencia', 'id expert': 'incontinencia',
    'molicare': 'incontinencia', 'ontex': 'incontinencia', 'tena': 'incontinencia',

    # INFECCIÓN URINARIA
    'cistitus': 'infeccion_urinaria', 'cispren': 'infeccion_urinaria', 'cranberry': 'infeccion_urinaria',
    'cystiberry': 'infeccion_urinaria', 'cysticlean': 'infeccion_urinaria', 'cystine': 'infeccion_urinaria',
    'd-manosa': 'infeccion_urinaria', 'ellura': 'infeccion_urinaria', 'feminabiane': 'infeccion_urinaria',
    'lactofem': 'infeccion_urinaria', 'monurelle': 'infeccion_urinaria', 'utipro': 'infeccion_urinaria',
    'urosens': 'infeccion_urinaria', 'aquilea cistitis': 'infeccion_urinaria',
    'arkocapsulas arandano': 'infeccion_urinaria',

    # LACTANCIA
    'dr. brown\'s': 'lactancia', 'dr browns': 'lactancia', 'lansinoh': 'lactancia',
    'mam': 'lactancia', 'medela': 'lactancia', 'nuk': 'lactancia', 'philips avent': 'lactancia',
    'tommee tippee': 'lactancia',

    # MAGNESIO
    'magmax': 'magnesio', 'magnesia san pellegrino': 'magnesio', 'magnesioboi': 'magnesio',
    'magnesium': 'magnesio', 'magnogene': 'magnesio', 'ana maria lajusticia': 'magnesio',
    'ivb magnesio total': 'magnesio', 'ns vitans magnesio': 'magnesio', 'aquilea magnesio': 'magnesio',
    'solgar magnesio': 'magnesio',

    # MAREO
    'biodramina': 'mareo', 'cinfamar': 'mareo', 'navicalm': 'mareo', 'normostop': 'mareo',
    'sweetsin stop mareo': 'mareo',

    # MATERIAL SANITARIO / ORTOPEDIA
    'actimove': 'material_sanitario', 'aposan': 'material_sanitario', 'cosmopor': 'heridas_apositos',
    'emo': 'material_sanitario', 'farmalastic': 'material_ortopedico', 'futuro': 'material_sanitario',
    'leukoplast': 'heridas_apositos', 'medilast': 'material_sanitario', 'nexcare': 'heridas_apositos',
    'orliman': 'material_ortopedico', 'actius': 'material_ortopedico', 'tiritas': 'heridas_apositos', 'varisan': 'material_sanitario',
    '3m': 'material_sanitario',

    # MEMORIA
    'bacopa': 'memoria', 'brain plus': 'memoria', 'claudia': 'memoria', 'dememory': 'memoria',
    'fosfatidilserina': 'memoria', 'ginkgo biloba': 'memoria', 'intellan': 'memoria',
    'memorup': 'memoria', 'mind master': 'memoria', 'neuralex': 'memoria', 'tanakene': 'memoria',
    'arkocapsulas ginkgo': 'memoria', 'nhco aminobrain': 'memoria', 'ivb memory-on': 'memoria',

    # MENOPAUSIA
    'aminopause': 'menopausia', 'cimicifuga': 'menopausia', 'ciminocta': 'menopausia',
    'climafort': 'menopausia', 'donna plus': 'menopausia', 'estromineral': 'menopausia',
    'femikliman': 'menopausia', 'gynecomplex': 'menopausia', 'isoflavonas': 'menopausia',
    'menofem': 'menopausia', 'menopro': 'menopausia', 'menostop': 'menopausia',
    'onagra': 'menopausia', 'seidibion': 'menopausia', 'serelys': 'menopausia', 'ymea': 'menopausia',
    'aquilea menopausia': 'menopausia', 'ns menopausia': 'menopausia', 'ivb menomaster': 'menopausia',

    # OJO SECO
    'aquoral': 'ojo_seco', 'estila': 'ojo_seco', 'hyabak': 'ojo_seco', 'lacryvisc': 'ojo_seco',
    'lagrinet': 'ojo_seco', 'lagroben': 'ojo_seco', 'lephanet': 'ojo_seco', 'lipolac': 'ojo_seco',
    'ocualiv': 'ojo_seco', 'ocular': 'ojo_seco', 'ojosbel': 'ojo_seco', 'orcex': 'ojo_seco',
    'recugel': 'ojo_seco', 'siccafluid': 'ojo_seco', 'systane': 'ojo_seco', 'viscofresh': 'ojo_seco',
    'visuectoine': 'ojo_seco',

    # OMEGA 3
    'enerzona omega': 'omega3', 'epaplus omega 3': 'omega3', 'krill oil': 'omega3',
    'lamberts omega': 'omega3', 'mollers': 'omega3', 'nordic naturals': 'omega3',
    'norsan': 'omega3', 'now foods omega': 'omega3', 'omacor': 'omega3', 'zeus genesis': 'omega3',
    'ivb omega': 'omega3', 'aquilea omega 3': 'omega3', 'solgar omega 3': 'omega3',

    # PIEL ATÓPICA
    'eucerin atopicontrol': 'piel_atopica', 'exomega': 'piel_atopica', 'leti at4': 'piel_atopica',
    'nutratopic': 'piel_atopica', 'primalba': 'piel_atopica', 'rilastil xerolact': 'piel_atopica',
    'stelatopia': 'piel_atopica', 'bepanthol sensidaily': 'piel_atopica', 'be+ atopia': 'piel_atopica',

    # PIOJOS
    'full marks': 'piojos', 'goibi antipiojos': 'piojos', 'itax': 'piojos', 'kitapic': 'piojos',
    'milice': 'piojos', 'neositrin': 'piojos', 'nyda': 'piojos', 'otc antipiojos': 'piojos',
    'paranix': 'piojos', 'permetrina': 'piojos',

    # PROTECCIÓN SOLAR
    '5punto5': 'proteccion_solar', 'ducray': 'dermatitis_seborreica', 'endocare': 'arrugas_antiedad',
    'galenic': 'arrugas_antiedad', 'lierac': 'arrugas_antiedad', 'martiderm': 'arrugas_antiedad',
    'neostrata': 'arrugas_antiedad', 'rilastil': 'proteccion_solar', 'sensilis': 'proteccion_solar',
    'svr': 'piel_sensible', 'two poles': 'proteccion_solar', 'isdinceutics': 'arrugas_antiedad',
    'apivita suncare': 'proteccion_solar', 'australian gold': 'proteccion_solar',
    'avene solar': 'proteccion_solar', 'babaria solar': 'proteccion_solar',
    'biotherm solaire': 'proteccion_solar', 'clinique sun': 'proteccion_solar',
    'coppertone': 'proteccion_solar', 'delial': 'proteccion_solar', 'ecran': 'proteccion_solar',
    'garnier delial': 'proteccion_solar', 'hawaiian tropic': 'proteccion_solar',
    'ladival': 'proteccion_solar', 'lancaster sun': 'proteccion_solar', 'nivea sun': 'proteccion_solar',
    'piz buin': 'proteccion_solar', 'rilastil sun': 'proteccion_solar', 'shiseido sun': 'proteccion_solar',
    'sensilis sun': 'proteccion_solar', 'be+ skin protect': 'proteccion_solar',
    'mustela solar': 'proteccion_solar', 'caudalie solaire': 'proteccion_solar',
    'nuxe sun': 'proteccion_solar', 'clarins sun': 'proteccion_solar', 'vichy capital soleil': 'proteccion_solar',
    'bioderma photoderm': 'proteccion_solar', 'eucerin sun': 'proteccion_solar',
    'la roche-posay anthelios': 'proteccion_solar', 'isdin fotoprotector': 'proteccion_solar',

    # REPELENTES
    'autan': 'repelente_insectos', 'cer8': 'repelente_insectos', 'chicco natural': 'repelente_insectos',
    'cinq sur cinq': 'repelente_insectos', 'goibi': 'repelente_insectos',
    'moskito guard': 'repelente_insectos', 'parasidose': 'repelente_insectos',
    'relec': 'repelente_insectos', 'isdin antimosquitos': 'repelente_insectos',

    # SALUD INFANTIL
    'colimil': 'colicos_bebe', 'reuteri gotas': 'flora_intestinal', 'rhinomer baby': 'congestion_nasal',
    'melamil': 'sueno_insomnio', 'ns florabiotic gotas': 'flora_intestinal',
    'vicks babyrub': 'congestion_nasal', 'pedialac suero': 'diarrea',

    # SUEÑO
    'circadin': 'sueno_insomnio', 'dormirel': 'sueno_insomnio', 'drasanvi melatonina': 'sueno_insomnio',
    'epaplus sleepcare': 'sueno_insomnio', 'esi melatonina': 'sueno_insomnio',
    'kneipp valeriana': 'sueno_insomnio', 'lasea': 'sueno_insomnio', 'naturtierra melatonina': 'sueno_insomnio',
    'nytol': 'sueno_insomnio', 'passiflora': 'sueno_insomnio', 'sedatif pc': 'sueno_insomnio',
    'serenity': 'sueno_insomnio', 'sonodor': 'sueno_insomnio', 'soria natural valeriana': 'sueno_insomnio',
    'valdispert': 'sueno_insomnio', 'valeriana': 'sueno_insomnio', 'vitanatur sueno': 'sueno_insomnio',
    'rescue night': 'sueno_insomnio', 'arkocapsulas pasiflora': 'sueno_insomnio',
    'ns melatonina': 'sueno_insomnio', 'aquilea sueno': 'sueno_insomnio', 'ns sonaben': 'sueno_insomnio',

    # TABAQUISMO
    'champix': 'tabaquismo', 'nicorette': 'tabaquismo', 'nicotinell': 'tabaquismo',
    'niquitin': 'tabaquismo', 'todacitan': 'tabaquismo',

    # TOS/MUCOSIDAD
    'aboca': 'mucosidad_respiratoria', 'acetilcisteina': 'mucosidad_respiratoria', 'ambroxol': 'mucosidad_respiratoria',
    'bisolvon': 'mucosidad_respiratoria', 'bromhexina': 'mucosidad_respiratoria', 'bronchostop': 'mucosidad_respiratoria',
    'carbocisteina': 'mucosidad_respiratoria', 'cloperastina': 'mucosidad_respiratoria', 'codeisan': 'mucosidad_respiratoria',
    'drosetux': 'mucosidad_respiratoria', 'grintuss': 'mucosidad_respiratoria', 'iniston': 'mucosidad_respiratoria',
    'juanola': 'mucosidad_respiratoria', 'mucibron': 'mucosidad_respiratoria', 'mucosan': 'mucosidad_respiratoria',
    'pectox': 'mucosidad_respiratoria', 'propolaid': 'mucosidad_respiratoria', 'prospantus': 'mucosidad_respiratoria',
    'romilar': 'mucosidad_respiratoria', 'sante verte sediflu': 'balsamos_respiratorios', 'seditus': 'mucosidad_respiratoria',
    'stodal': 'mucosidad_respiratoria', 'toseina': 'mucosidad_respiratoria',

    # VERRUGAS/CALLOS
    'duofilm': 'verrugas_callos', 'excilor': 'verrugas_callos', 'farmapodo': 'verrugas_callos',
    'pointts': 'verrugas_callos', 'salvelox callos': 'verrugas_callos', 'scholl': 'verrugas_callos',
    'urgo verrugas': 'verrugas_callos', 'wartner': 'verrugas_callos',
    'isdin antiverrugas': 'verrugas_callos', 'compeed callos': 'verrugas_callos',

    # VETERINARIA
    'advantix': 'veterinaria', 'bravecto': 'veterinaria', 'fortekor': 'veterinaria',
    'frontline': 'veterinaria', 'nexgard': 'veterinaria', 'otomax': 'veterinaria',
    'recicort': 'veterinaria', 'scalibor': 'veterinaria', 'seresto': 'veterinaria',
    'upcard': 'veterinaria',

    # VISIÓN
    'astaxantina': 'vision', 'brudy macula': 'vision', 'luteina': 'vision', 'nutrof': 'vision',
    'ocuvite': 'vision', 'oftan macula': 'vision', 'preservision': 'vision',
    'retinacaps': 'vision', 'vitreoclar': 'vision', 'ivb vision': 'vision',

    # IVB WELLNESS LAB (Isabel Viña) - Suplementación completa
    # Metabolismo y Tiroides
    'ivb tyroenergy': 'energia_vitalidad',     # Tiroides/energía
    'ivb metabolic': 'colesterol',              # Metabolic-Max (colesterol, triglicéridos)
    'ivb satisens': 'control_peso',             # Control apetito/saciedad
    # Sueño y Estrés
    'ivb somnilove': 'sueno_insomnio',          # Sueño sin melatonina
    'ivb calm': 'estres_ansiedad',              # Ashwagandha
    'ivb stress': 'estres_ansiedad',            # Estrés general
    # Salud de la Mujer
    'ivb fertiup': 'fertilidad',                # Calidad ovárica
    'ivb pcos': 'ciclo_menstrual',              # SOP
    'ivb woman': 'ciclo_menstrual',             # Woman's Balance
    'ivb meno': 'menopausia',                   # Menopausia
    # Vitaminas/Minerales específicos
    'ivb d3': 'calcio_vitamina_d',              # D3+K2
    'ivb magnesio': 'magnesio',                 # Magnesio Total
    'ivb omega': 'omega3',                      # Omega
    'ivb hierro': 'hierro',                     # Hierro
    'ivb ferroplus': 'hierro',                  # FerroPlus
    # Piel/Pelo (Nutricosmética)
    'ivb skin': 'colageno',                     # Suplementos piel
    'ivb hair': 'caida_cabello',                # Suplementos pelo
    'ivb colageno': 'colageno',                 # Colágeno
    # Digestivo
    'ivb digestive': 'digestion_pesada',        # Enzimas digestivas
    'ivb probio': 'probioticos',                # Probióticos
    # Vitaminas generales
    'ivb metil': 'vitaminas_minerales',         # Metil-B12
    'ivb b12': 'vitaminas_minerales',           # Variantes B12
    'ivb wellness': 'vitaminas_minerales',      # Marca general fallback
    'isabel vina': 'vitaminas_minerales',       # Creadora
    'isabel viña': 'vitaminas_minerales',       # Con ñ

    # NHCO NUTRITION - Aminoscience (22 referencias)
    # Capilar
    'nhco aminohair': 'caida_cabello',          # Cabello fuerte
    'nhco hair': 'caida_cabello',
    # Piel (Nutricosmética)
    'nhco aminoskin': 'colageno',               # Piel desde dentro
    'nhco skin': 'colageno',
    'nhco celluoff': 'celulitis',               # AminoSkin CelluOff
    # Energía y Vitalidad
    'nhco aminoenergy': 'energia_vitalidad',    # Ginseng, mate, L-glutamina
    'nhco energy': 'energia_vitalidad',
    # Memoria y Concentración
    'nhco aminomemory': 'concentracion_memoria', # Salud cerebral
    'nhco aminobrain': 'concentracion_memoria',  # Variante
    'nhco memory': 'concentracion_memoria',
    # Menopausia
    'nhco aminopause': 'menopausia',            # Perimenopausia
    'nhco pause': 'menopausia',
    # Digestivo / Deporte
    'nhco glutamine': 'proteinas_deporte',      # L-Glutamine 5
    'nhco l-glutamine': 'proteinas_deporte',
    # Multivitamínico
    'nhco aminocomplex': 'vitaminas_minerales', # Equilibrio nutricional
    'nhco complex': 'vitaminas_minerales',
    # Sueño / Estrés
    'nhco aminorelax': 'sueno_insomnio',        # Relajación
    'nhco relax': 'estres_ansiedad',
    # Marca general
    'nhco nutrition': 'vitaminas_minerales',    # Fallback marca
    'nhco': 'vitaminas_minerales',              # Prefijo general

    # VITAMINAS GENERALES
    '100% natural': 'vitaminas_general', 'ana maria lajusticia': 'vitaminas_general',
    'becozyme': 'vitaminas_general', 'berocca': 'vitaminas_general', 'cfn': 'vitaminas_general',
    'dayamineral': 'vitaminas_general', 'drasanvi': 'vitaminas_general',
    'el granero integral': 'vitaminas_general', 'eladiet': 'vitaminas_general',
    'equisalud': 'vitaminas_general', 'esi': 'vitaminas_general', 'healthaid': 'vitaminas_general',
    'hidropolivit': 'vitaminas_general', 'nature\'s plus': 'vitaminas_general',
    'natures bounty': 'vitaminas_general', 'natysal': 'vitaminas_general',
    'now foods': 'vitaminas_general', 'nutergia': 'vitaminas_general', 'nutersin': 'vitaminas_general',
    'pharmaton': 'vitaminas_general', 'phytogreen': 'vitaminas_general', 'pileje': 'vitaminas_general',
    'redoxon': 'vitaminas_general', 'salengei': 'vitaminas_general', 'santiveri': 'vitaminas_general',
    'soria natural': 'vitaminas_general', 'vitagobens': 'vitaminas_general',
    'ns vitans': 'vitaminas_general',

    # =============================================================================
    # MARCAS ADICIONALES (detectadas de productos top sin marca - 2025-12-17)
    # =============================================================================

    # TEST DIAGNÓSTICO
    'clearblue': 'test_diagnostico',

    # ESTREÑIMIENTO
    'vilardell': 'estrenimiento',

    # OJO SECO / OFTALMOLOGÍA
    'lubristil': 'ojo_seco', 'ozonest': 'ojo_seco', 'vita-pos': 'ojo_seco',
    'vispring': 'ojo_seco', 'optialerg': 'alergia',

    # CICATRIZACIÓN
    'bepanthol': 'cicatrizacion',

    # ROSACEA / PIEL SENSIBLE
    'rosacure': 'piel_sensible', 'rosaderm': 'piel_sensible',

    # ACNÉ
    'biretix': 'acne', 'azelac': 'acne',

    # HIGIENE NASAL / CONGESTIÓN
    'normomar': 'congestion_nasal', 'respir': 'congestion_nasal', 'respimar': 'congestion_nasal',
    'physiodose': 'higiene_nasal', 'normobucal': 'dolor_garganta',

    # VITAMINAS INFANTIL
    'pediakid': 'vitaminas_infantil',

    # INMUNIDAD
    'imunoglukan': 'inmunidad', 'inmunoferon': 'inmunidad',

    # COLESTEROL
    'armolipid': 'colesterol', 'tecnigen': 'colesterol',

    # IRRITACIONES / PICADURAS
    'talquistina': 'irritacion_piel', 'calmiox': 'picaduras', 'dermapic': 'picaduras',

    # HONGOS
    'canespie': 'hongos_pies',

    # GRIPE / RESFRIADO
    'ilvigrip': 'gripe_resfriado', 'desenfriol': 'gripe_resfriado',

    # PERFUMERÍA
    'iap pharma': 'perfumeria',

    # SUPLEMENTOS VARIOS
    'soniase': 'vitaminas_general', 'aminovital': 'vitaminas_general', 'barimix': 'vitaminas_general',
    'hemoliv': 'vitaminas_general', 'superlativa': 'vitaminas_general',
    'ferrum': 'vitaminas_general', 'aminobalance': 'vitaminas_general', 'bonusan': 'vitaminas_general',
    'aminobrain': 'vitaminas_general',
    # 'advance nutrition programme' -> movido a sección FEEDBACK LOOP (colageno)

    # MEMORIA/COGNITIVO (reclasificados 2025-12-18)
    'cognex': 'memoria_cognitivo',  # Farmolab - Theracurmin para función cerebral
    'mindex': 'memoria_cognitivo',  # Bioksan - Omega-3 + Bacopa para concentración

    # ARTICULACIONES (reclasificado 2025-12-18)
    'nexala': 'articulaciones',  # Farmolab - Curcumina + Boswellia para artrosis

    # SALUD HEPÁTICA (reclasificado 2025-12-18)
    'livex': 'salud_hepatica',  # Bioksan - Cardo mariano para hígado/detox

    # SUEÑO
    'aminosleep': 'sueno_insomnio', 'passiflorine': 'estres_ansiedad', 'sed-tif': 'estres_ansiedad',

    # DIGESTIVO / GASES
    'pankreoflat': 'gases', 'gelsectan': 'diarrea', 'noraflat': 'gases',

    # CAPILAR
    'normopil': 'caida_cabello', 'pirotex': 'caspa', 'minoxidil cinfa': 'caida_cabello',

    # HERPES
    'herpix': 'herpes', 'lavirk': 'herpes',

    # LACTANCIA
    'purelan': 'lactancia',

    # BEBÉS
    'baby naturals': 'higiene_bebe',

    # CBD / DOLOR
    'cannabiben': 'dolor_muscular',

    # DOLOR
    'actromadol': 'dolor', 'difenadex': 'dolor',

    # INFUSIONES
    'bie3': 'infusiones',

    # HIGIENE
    'quilian': 'higiene_corporal',

    # =============================================================================
    # LÍNEAS DE PRODUCTO SIN MARCA MADRE EN NOMBRE (2025-12-17)
    # Productos que se venden como "ERYFOTONA..." sin "ISDIN" en el nombre
    # =============================================================================

    # ISDIN líneas vendidas sin marca
    'eryfotona': 'proteccion_solar',

    # SESDERMA líneas vendidas sin marca
    'c-vit': 'arrugas_antiedad', 'azelac ru': 'acne',

    # CANTABRIA LABS líneas
    'endocare': 'arrugas_antiedad', 'neoretin': 'manchas',

    # MINOXIDIL variantes ortográficas
    'minoxidil viñas': 'caida_cabello',

    # OTRAS MARCAS/LÍNEAS STANDALONE
    'afta clear': 'aftas', 'beter': 'cuidado_cabello', 'perio aid': 'higiene_dental',
    'rosalgin': 'higiene_intima', 'melagyn': 'higiene_intima', 'finisher': 'nutricion_deportiva',
    'plenocapil': 'caida_cabello', 'tonucia': 'caida_cabello', 'lipolasic': 'ojo_seco',
    'yeloin': 'ojo_seco', 'recudrop': 'ojo_seco', 'coqun': 'ojo_seco',
    'dentispray': 'dolor_dental', 'nervala': 'estres_ansiedad', 'tinnitan': 'tinnitus',
    'ibustick': 'dolor_muscular', 'acticrem': 'dolor_muscular',

    # =============================================================================
    # REGLAS FEEDBACK LOOP (Issue #457 - 2025-12-19)
    # Generadas automáticamente del Human-in-the-Loop con 75 productos revisados
    # =============================================================================

    # BÁLSAMOS RESPIRATORIOS (nueva categoría)
    'vaporub': 'balsamos_respiratorios',  # Corregido de congestion_nasal

    # JUANOLA - diferenciación por tipo de producto
    'juanola pastillas': 'tos_seca',      # Pastillas clásicas = antitusivo
    'juanola caramelos': 'dolor_garganta', # Caramelos = suavizante garganta

    # GARGANTA
    'zarbees garganta': 'dolor_garganta',  # Corregido de mucosidad_respiratoria

    # ENCÍAS (colutorio clorhexidina)
    'lacer colutorio clorhexidina': 'encias',  # Corregido de higiene_dental

    # CUIDADO LABIAL (nueva categoría)
    'gh balsamo funcion barrera': 'cuidado_labial',  # Bálsamos labiales

    # ACCESORIOS BAÑO (nueva categoría - Issue #457)
    'beter esponja': 'accesorios_bano',   # Esponjas corporales
    'beter cepillo bano': 'accesorios_bano',  # Cepillos de baño (no pelo)

    # =============================================================================
    # REGLAS FEEDBACK LOOP - Mining Correcciones (2025-12-19)
    # 109 correcciones analizadas, 82.6% accuracy
    # =============================================================================

    # SERVICIOS FARMACIA - SPD (Sistema Personalizado de Dosificación)
    # Confusión: interno_no_venta -> servicios_farmacia (6 casos)
    'spd': 'servicios_farmacia',           # Dosificador semanal
    'spd semanal': 'servicios_farmacia',   # SPD explícito
    'dosificador semanal': 'servicios_farmacia',
    'pastillero semanal': 'servicios_farmacia',

    # COLÁGENO - Advance Nutrition Programme
    # Confusión: vitaminas_general -> colageno (2 casos)
    'advance nutrition programme': 'colageno',  # Marca premium (principalmente colágeno)
    'advance nutrition': 'colageno',            # Variante corta
    'collagen synergy': 'colageno',             # Producto específico
    'nutrition programme collagen': 'colageno', # Variante

    # OPTICA - Gafas, lupas, accesorios
    'gafas presbicia': 'optica',
    'gafas lectura': 'optica',
    'gafas vista cansada': 'optica',
    'lupa': 'optica',
    'cordon gafas': 'optica',
    'cordón gafas': 'optica',
    'funda gafas': 'optica',
    'gamuza gafas': 'optica',
    'limpiagafas': 'optica',
    'gafas sol': 'optica',

    # LIMPIEZA FACIAL - Tónicos, aguas termales
    'agua termal': 'limpieza_facial',
    'tonico facial': 'limpieza_facial',
    'bruma facial': 'limpieza_facial',
    'agua micelar': 'limpieza_facial',

    # HIGIENE NASAL - Lavados, suero fisiológico nasal
    'suero fisiologico nasal': 'higiene_nasal',
    'suero fisiologico monodosis': 'higiene_nasal',  # Por defecto nasal (uso más común)
    'agua de mar': 'higiene_nasal',
    'rhinomer': 'higiene_nasal',
    'nasalmer': 'higiene_nasal',
    'rinowash': 'higiene_nasal',
    'lavado nasal': 'higiene_nasal',
    'irrigador nasal': 'higiene_nasal',
    'spray nasal salino': 'higiene_nasal',
    'physiomer': 'higiene_nasal',
    'respibien agua mar': 'higiene_nasal',
    'quinton nasal': 'higiene_nasal',

    # REGALOS/BAZAR - Productos no relacionados con salud
    'juguete': 'regalos_bazar',
    'peluche': 'regalos_bazar',
    'muñeco': 'regalos_bazar',
    'puzzle': 'regalos_bazar',
    'figura decorativa': 'regalos_bazar',
    'adorno': 'regalos_bazar',
    'llavero': 'regalos_bazar',
    'imán nevera': 'regalos_bazar',
    'merchandising': 'regalos_bazar',
    'regalo': 'regalos_bazar',
    'set regalo': 'regalos_bazar',
    'cofre regalo': 'regalos_bazar',
    'neceser': 'regalos_bazar',            # Bolsas/neceseres vacíos
    'bolsa regalo': 'regalos_bazar',

    # =========================================================================
    # MARCAS BASE FALTANTES (Issue #486) - 2025-12-28
    # Marcas que existían como líneas específicas pero no como marca genérica
    # =========================================================================

    # Marcas con líneas que faltaba la base
    'vichy': 'cosmetica_facial',           # Fallback si no matchea línea específica
    'durex': 'preservativos',              # Preservativos y lubricantes
    'hero': 'alimentacion_bebe',           # Hero Baby sin "baby"
    'blevit': 'alimentacion_bebe',         # Papillas y cereales
    'nosa': 'antipiojos',                  # Tratamientos antipiojos

    # Marcas nuevas detectadas por análisis de ventas
    'corny': 'alimentacion_saludable',     # Barritas de cereales
    'eroxon': 'disfuncion_erectil',        # Gel para disfunción eréctil
    'efferaldol': 'dolor_fiebre',          # Analgésico efervescente
    'bucomed': 'dolor_garganta',           # Pastillas para garganta
    'ozobaby': 'cosmetica_bebe',           # Cosmética bebé con ozono
    'libicare': 'suplementos_mujer',       # Suplementos femeninos
    'genotergente': 'higiene_corporal',    # Gel de higiene
    'bexidermil': 'dermatologia',          # Tratamiento dermatológico
    'sediflu': 'gripe_resfriado',          # Antigripal
    'frenodiar': 'diarrea',                # Antidiarreico
    'pipia': 'joyeria_pendientes',         # Pendientes infantiles
    'pipía': 'joyeria_pendientes',         # Pendientes (con tilde)
    'quies': 'tapones_oidos',              # Tapones para oídos
    'rovi': 'medicamento_generico',        # Laboratorio Rovi
    'aminodetox': 'detox_depurativo',      # Suplemento detox
    'regenintest': 'digestivo',            # Salud intestinal
    'ginecomplex': 'suplementos_mujer',    # Suplementos ginecológicos
    'vea': 'dermatologia',                 # Aceites dermatológicos
    'emolienta': 'piel_seca',              # Emolientes piel seca
    'pranzo': 'pediatrico',                # Medicamento pediátrico
    'entero': 'digestivo',                 # Silicona digestiva
}

# =============================================================================
# BLACKLIST DE SERVICIOS (Issue #486) - 2025-12-28
# Patrones regex para detectar servicios internos de farmacia (no productos)
# Estos se marcan como 'interno_no_venta' y NO se clasifican como venta_libre
# =============================================================================

# Virtual brand for generic products without detectable brand (Issue #486)
MARCA_BLANCA = "marca_blanca"

SERVICE_BLACKLIST_PATTERNS = [
    re.compile(r'^spd[\s\-]', re.IGNORECASE),        # "SPD DANIEL", "SPD-ANA"
    re.compile(r'^servicio\s', re.IGNORECASE),       # "SERVICIO NUTRICION"
    re.compile(r'^peso$', re.IGNORECASE),            # "PESO" exacto
    re.compile(r'^analisis\s', re.IGNORECASE),       # "ANALISIS SANGRE"
    re.compile(r'^medicion\s', re.IGNORECASE),       # "MEDICION TENSION"
    re.compile(r'^formula\s?magistral', re.IGNORECASE),  # "FORMULA MAGISTRAL"
    re.compile(r'^alquiler\s', re.IGNORECASE),       # "ALQUILER SILLA"
    re.compile(r'^fianza\s', re.IGNORECASE),         # "FIANZA ALQUILER"
]

def is_service_product(product_name: str) -> bool:
    """
    Detecta si un producto es realmente un servicio interno de farmacia.

    Args:
        product_name: Nombre del producto

    Returns:
        True si es un servicio (SPD, SERVICIO NUTRICION, etc.)
    """
    if not product_name:
        return False
    for pattern in SERVICE_BLACKLIST_PATTERNS:
        if pattern.search(product_name):
            return True
    return False

# Lista adicional de marcas simples para BRAND_PRIORITY (orden alfabético)
SIMPLE_BRAND_PRIORITY = sorted(SIMPLE_BRANDS.keys())


class BrandDetectionService:
    """
    Servicio de detección de marca y línea de producto.

    Uso:
        service = BrandDetectionService()
        result = service.detect("ISDIN FOTOPROTECTOR FUSION WATER SPF50+ 50ML")
        # result.detected_brand = "isdin"
        # result.brand_line = "fusion water"
        # result.ml_subcategory = "facial"
        # result.necesidad = "proteccion_solar"

    Issue #XXX: Soporta BrandAlias para correcciones de marcas:
        - Fusionar marcas: "oralkin" → "kin"
        - Excluir falsos positivos: "clorhexidina" → None (no es marca)
    """

    def __init__(self):
        self.brand_lines = ALL_BRAND_LINES
        self.brand_priority = BRAND_PRIORITY
        self.simple_brands = SIMPLE_BRANDS
        # Combined priority: detailed brands first, then simple brands
        self.all_brand_priority = BRAND_PRIORITY + [
            b for b in SIMPLE_BRAND_PRIORITY if b not in BRAND_PRIORITY
        ]
        # Issue #XXX: Cache de brand aliases (cargado desde DB)
        # Dict: source_brand -> (target_brand, action)
        # target_brand es None para action="exclude"
        self._brand_aliases: Dict[str, Tuple[Optional[str], str]] = {}

    def load_brand_aliases(self, db) -> None:
        """
        Carga los brand aliases desde la base de datos al cache.

        Llamar al inicio de la aplicación o después de cambios en aliases.

        Args:
            db: SQLAlchemy Session
        """
        from app.models.brand_alias import BrandAlias, BrandAliasAction

        aliases = (
            db.query(BrandAlias)
            .filter(BrandAlias.is_active == True)  # noqa: E712
            .all()
        )

        self._brand_aliases = {}
        for alias in aliases:
            if alias.action == BrandAliasAction.EXCLUDE:
                self._brand_aliases[alias.source_brand] = (None, "exclude")
            else:
                self._brand_aliases[alias.source_brand] = (alias.target_brand, "alias")

        logger.info(f"Loaded {len(self._brand_aliases)} brand aliases into cache")

    def _apply_brand_alias(self, brand: str) -> Tuple[Optional[str], bool]:
        """
        Aplica alias a una marca detectada usando el cache.

        Args:
            brand: Marca detectada

        Returns:
            Tuple[Optional[str], bool]:
                - str or None: La marca resultante (target o None si excluded)
                - bool: True si se aplicó un alias
        """
        brand_lower = brand.lower()

        if brand_lower in self._brand_aliases:
            target, action = self._brand_aliases[brand_lower]
            return (target, True)

        return (brand, False)

    def detect(self, product_name: str) -> BrandDetectionResult:
        """
        Detecta marca, línea de producto y subcategoría.

        Args:
            product_name: Nombre del producto

        Returns:
            BrandDetectionResult con información detectada
        """
        if not product_name:
            return BrandDetectionResult()

        result = BrandDetectionResult()
        name_lower = product_name.lower()

        # 1. Detectar marca
        brand = self._detect_brand(name_lower)
        if not brand:
            return result

        result.detected_brand = brand
        result.confidence = 0.70  # Base confidence for brand detection

        # 2. Detectar línea de producto (detailed brands)
        line_match = self._detect_line(name_lower, brand)
        if line_match:
            line_name, line_info = line_match
            result.brand_line = line_name
            result.ml_subcategory = line_info.zona if line_info.zona else None
            result.necesidad = line_info.necesidad
            result.confidence = 0.85  # Higher confidence with line detection
        # 3. For simple brands, set necesidad from SIMPLE_BRANDS
        elif brand in self.simple_brands:
            result.necesidad = self.simple_brands[brand]
            result.confidence = 0.75  # Medium confidence for simple brands

        return result

    def _detect_brand(self, name_lower: str) -> Optional[str]:
        """
        Detecta la marca de un producto.

        Issue #XXX: Soporta BrandAlias para correcciones:
        1. Si se detecta una marca, busca en cache de aliases
        2. Si hay alias con action="alias": retorna target_brand
        3. Si hay alias con action="exclude": salta a siguiente marca (continúa loop)
        4. Si no hay alias: aplica normalizaciones legacy hardcodeadas
        """
        import re

        # Search in combined priority list (detailed brands first, then simple)
        for brand in self.all_brand_priority:
            # For short brands (<=3 chars), require word boundary
            if len(brand) <= 3:
                # Use word boundary regex for short brands to avoid false positives
                pattern = r'\b' + re.escape(brand) + r'\b'
                if re.search(pattern, name_lower):
                    # Issue #XXX: Aplicar alias si existe
                    result_brand, was_aliased = self._apply_brand_alias(brand)
                    if was_aliased:
                        if result_brand is None:
                            # Excluido (no es marca), continuar con siguiente
                            continue
                        return result_brand
                    return brand
            elif brand in name_lower:
                # Issue #XXX: Aplicar alias si existe (antes de normalizaciones legacy)
                result_brand, was_aliased = self._apply_brand_alias(brand)
                if was_aliased:
                    if result_brand is None:
                        # Excluido (no es marca), continuar con siguiente
                        continue
                    return result_brand

                # Normalizar marcas conocidas (LEGACY - migrar a BrandAlias)
                # Estas normalizaciones están seedeadas en la migración
                if 'roche' in brand:
                    return 'la roche posay'
                if brand == 'gema herrerias':
                    return 'gh'
                if brand == 'ivb wellness':
                    return 'ivb'
                return brand
        return None

    def _detect_line(self, name_lower: str, brand: str) -> Optional[Tuple[str, ProductLineInfo]]:
        """Detecta la línea de producto dentro de una marca."""
        if brand not in self.brand_lines:
            return None

        brand_lines = self.brand_lines[brand]

        # Buscar línea más específica primero (ordenar por longitud)
        sorted_lines = sorted(brand_lines.items(), key=lambda x: len(x[0]), reverse=True)

        for line_name, line_info in sorted_lines:
            if line_name in name_lower:
                return (line_name, line_info)

            # Buscar también en keywords
            for keyword in line_info.keywords:
                if keyword in name_lower:
                    return (line_name, line_info)

        return None

    def get_stats(self) -> Dict:
        """Estadísticas del servicio."""
        detailed_brands = len(set(self.brand_lines.keys()))
        simple_brands = len(self.simple_brands)
        total_brands = len(self.all_brand_priority)
        total_lines = sum(len(lines) for lines in self.brand_lines.values())
        return {
            "total_brands": total_brands,
            "detailed_brands": detailed_brands,
            "simple_brands": simple_brands,
            "total_product_lines": total_lines,
            "brands": list(set(self.all_brand_priority)),
        }


# Singleton para uso global
brand_detection_service = BrandDetectionService()


def detect_brand_info(product_name: str) -> BrandDetectionResult:
    """
    Función de conveniencia para detectar información de marca.

    Args:
        product_name: Nombre del producto

    Returns:
        BrandDetectionResult con información detectada
    """
    return brand_detection_service.detect(product_name)


def detect_brand_with_fallback(
    product_name: str,
    necesidad: Optional[str] = None,
) -> BrandDetectionResult:
    """
    Detecta marca con fallback a MARCA_BLANCA si no se encuentra marca.

    Issue #486: Asegura 100% cobertura de marca para analytics.
    - Si se detecta marca real → la devuelve normalmente
    - Si NO se detecta marca PERO hay necesidad válida → MARCA_BLANCA
    - Si NO hay necesidad válida → devuelve resultado vacío

    Args:
        product_name: Nombre del producto
        necesidad: Necesidad ya clasificada (opcional, para decidir fallback)

    Returns:
        BrandDetectionResult (con MARCA_BLANCA si aplica fallback)
    """
    result = brand_detection_service.detect(product_name)

    # Si ya tiene marca detectada, devolver normalmente
    if result.detected_brand:
        return result

    # Sin marca detectada: aplicar MARCA_BLANCA si hay necesidad válida
    # Excluir categorías de servicios/interno que no son productos reales
    excluded_categories = (
        "otros", "interno_no_venta", "desconocido",
        "servicio", "servicios_farmacia",  # Issue #486: servicios no son productos
    )
    if necesidad and necesidad not in excluded_categories:
        return BrandDetectionResult(
            detected_brand=MARCA_BLANCA,
            brand_line=None,
            ml_subcategory=None,
            necesidad=necesidad,
            confidence=0.40,  # Low confidence to trigger review
        )

    # Sin marca y sin necesidad válida: devolver vacío
    return result
