# backend/app/services/brand_alias_service.py
"""
Service for brand alias management.

Sistema de corrección de marcas para BrandDetectionService.

Este servicio gestiona los aliases de marcas que:
1. Fusionan marcas (alias): "oralkin" → "kin"
2. Excluyen falsos positivos (exclude): "clorhexidina" → (no es marca)

Diferente de CategoryAliasService:
    - Este normaliza OUTPUT de brand detection
    - Tiene acción EXCLUDE para falsos positivos (ingredientes detectados como marcas)
"""

import logging
from datetime import datetime, timezone
from typing import Dict, Optional, Tuple

from sqlalchemy import func
from sqlalchemy.orm import Session

from app.models.brand_alias import BrandAlias, BrandAliasAction
from app.models.sales_enrichment import SalesEnrichment
from app.schemas.brand_alias import BrandAliasCreate, BrandAliasUpdate

logger = logging.getLogger(__name__)


def _reload_brand_detection_cache(db: Session) -> None:
    """
    Recarga el cache de brand aliases en el singleton BrandDetectionService.

    Llamar después de cualquier operación de escritura (create, update, delete, toggle).
    """
    try:
        from app.services.brand_detection_service import brand_detection_service

        brand_detection_service.load_brand_aliases(db)
    except Exception as e:
        # Log at ERROR level - stale cache could cause incorrect brand detection
        logger.error(f"Failed to reload brand detection cache: {e}", exc_info=True)


class BrandAliasService:
    """
    Service para CRUD de brand aliases y aplicación en detección.

    Soporta dos acciones:
    - ALIAS: Renombrar marca (source → target)
    - EXCLUDE: Excluir de detección (source → NULL, no es marca)
    """

    def __init__(self, db: Session):
        self.db = db
        # Cache para evitar DB calls por producto
        self._alias_cache: Optional[Dict[str, Tuple[Optional[str], BrandAliasAction]]] = None

    # =========================================================================
    # CRUD Operations
    # =========================================================================

    def list_aliases(
        self,
        is_active: Optional[bool] = None,
        action: Optional[BrandAliasAction] = None,
        search: Optional[str] = None,
        limit: int = 100,
        offset: int = 0,
    ) -> dict:
        """
        Lista aliases con filtros y estadísticas.

        Args:
            is_active: Filtrar por estado activo/inactivo
            action: Filtrar por tipo de acción (alias/exclude)
            search: Buscar en source_brand o target_brand
            limit: Máximo de resultados
            offset: Paginación

        Returns:
            dict con items (con affected_count), total, offset, limit, stats
        """
        query = self.db.query(BrandAlias)

        # Aplicar filtros
        if is_active is not None:
            query = query.filter(BrandAlias.is_active == is_active)
        if action is not None:
            query = query.filter(BrandAlias.action == action)
        if search:
            escaped_search = search.replace("\\", "\\\\").replace("%", "\\%").replace("_", "\\_")
            search_filter = f"%{escaped_search}%"
            query = query.filter(
                (BrandAlias.source_brand.ilike(search_filter, escape="\\"))
                | (BrandAlias.target_brand.ilike(search_filter, escape="\\"))
            )

        # Contar total
        total = query.count()

        # Ordenar: por uso descendente, luego alfabético
        items = (
            query.order_by(
                BrandAlias.usage_count.desc(),
                BrandAlias.source_brand,
            )
            .offset(offset)
            .limit(limit)
            .all()
        )

        # OPTIMIZATION: Get all affected counts in ONE query (avoid N+1)
        source_brands = [alias.source_brand for alias in items]
        if source_brands:
            counts = (
                self.db.query(
                    SalesEnrichment.detected_brand,
                    func.count(SalesEnrichment.id).label("count"),
                )
                .filter(SalesEnrichment.detected_brand.in_(source_brands))
                .group_by(SalesEnrichment.detected_brand)
                .all()
            )
            count_map = {row[0]: row[1] for row in counts}
        else:
            count_map = {}

        # Build response items with affected_count from cache
        items_with_counts = []
        for alias in items:
            alias_dict = {
                "id": alias.id,
                "source_brand": alias.source_brand,
                "target_brand": alias.target_brand,
                "action": alias.action,
                "is_active": alias.is_active,
                "reason": alias.reason,
                "usage_count": alias.usage_count,
                "last_used_at": alias.last_used_at,
                "created_at": alias.created_at,
                "affected_count": count_map.get(alias.source_brand, 0),
            }
            items_with_counts.append(alias_dict)

        # Calcular estadísticas
        stats = self._get_stats()

        return {
            "items": items_with_counts,
            "total": total,
            "offset": offset,
            "limit": limit,
            "stats": stats,
        }

    def get_alias(self, alias_id: int) -> Optional[BrandAlias]:
        """Obtener un alias por ID."""
        return self.db.query(BrandAlias).filter(BrandAlias.id == alias_id).first()

    def get_by_source(self, source_brand: str) -> Optional[BrandAlias]:
        """Obtener alias por source_brand."""
        return (
            self.db.query(BrandAlias)
            .filter(BrandAlias.source_brand == source_brand.lower())
            .first()
        )

    def create_alias(self, data: BrandAliasCreate) -> BrandAlias:
        """
        Crea un nuevo alias.

        Args:
            data: Datos del alias

        Returns:
            BrandAlias creado

        Raises:
            ValueError: Si el source_brand ya existe
        """
        # Verificar duplicado
        existing = self.get_by_source(data.source_brand)
        if existing:
            raise ValueError(f"Ya existe un alias para '{data.source_brand}'")

        # Crear alias
        alias = BrandAlias(
            source_brand=data.source_brand.lower(),
            target_brand=data.target_brand.lower() if data.target_brand else None,
            action=data.action,
            reason=data.reason,
            is_active=True,
        )

        self.db.add(alias)
        self.db.commit()
        self.db.refresh(alias)

        # Invalidar cache local y recargar singleton
        self._alias_cache = None
        _reload_brand_detection_cache(self.db)

        if alias.action == BrandAliasAction.EXCLUDE:
            logger.info(f"Created brand alias: {alias.source_brand} → EXCLUDE")
        else:
            logger.info(f"Created brand alias: {alias.source_brand} → {alias.target_brand}")

        return alias

    def update_alias(
        self, alias_id: int, data: BrandAliasUpdate
    ) -> Optional[BrandAlias]:
        """
        Actualiza un alias existente.

        Args:
            alias_id: ID del alias a actualizar
            data: Datos a actualizar

        Returns:
            BrandAlias actualizado o None si no existe

        Raises:
            ValueError: Si el nuevo source_brand ya existe en otro alias
        """
        alias = self.get_alias(alias_id)
        if not alias:
            return None

        # Si cambia source_brand, verificar que no exista
        if data.source_brand and data.source_brand.lower() != alias.source_brand:
            existing = self.get_by_source(data.source_brand)
            if existing and existing.id != alias_id:
                raise ValueError(f"Ya existe un alias para '{data.source_brand}'")

        # Aplicar actualizaciones
        update_data = data.model_dump(exclude_unset=True)

        # CRITICAL FIX: Validar coherencia action-target ANTES de aplicar
        new_action = update_data.get("action", alias.action)
        new_target = update_data.get("target_brand", alias.target_brand)

        if new_action == BrandAliasAction.ALIAS and not new_target:
            raise ValueError("target_brand es requerido cuando action='alias'")

        for field, value in update_data.items():
            if value is not None:
                setattr(alias, field, value)

        # Si action=EXCLUDE, forzar target_brand a None
        if alias.action == BrandAliasAction.EXCLUDE:
            alias.target_brand = None

        self.db.commit()
        self.db.refresh(alias)

        # Invalidar cache local y recargar singleton
        self._alias_cache = None
        _reload_brand_detection_cache(self.db)

        logger.info(f"Updated brand alias id={alias_id}")

        return alias

    def delete_alias(self, alias_id: int, hard: bool = False) -> bool:
        """
        Elimina un alias.

        Args:
            alias_id: ID del alias a eliminar
            hard: Si True, elimina permanentemente. Si False, desactiva (soft delete)

        Returns:
            True si se eliminó/desactivó, False si no existe
        """
        alias = self.get_alias(alias_id)
        if not alias:
            return False

        if hard:
            self.db.delete(alias)
            logger.info(f"Hard deleted brand alias id={alias_id}")
        else:
            alias.is_active = False
            logger.info(f"Soft deleted (deactivated) brand alias id={alias_id}")

        self.db.commit()

        # Invalidar cache local y recargar singleton
        self._alias_cache = None
        _reload_brand_detection_cache(self.db)

        return True

    def toggle_active(self, alias_id: int) -> Optional[BrandAlias]:
        """
        Alternar estado activo/inactivo de un alias.

        Args:
            alias_id: ID del alias

        Returns:
            BrandAlias actualizado o None si no existe
        """
        alias = self.get_alias(alias_id)
        if not alias:
            return None

        alias.is_active = not alias.is_active
        self.db.commit()
        self.db.refresh(alias)

        # Invalidar cache local y recargar singleton
        self._alias_cache = None
        _reload_brand_detection_cache(self.db)

        logger.info(f"Toggled brand alias id={alias_id} to is_active={alias.is_active}")

        return alias

    # =========================================================================
    # Core functionality: Apply Alias
    # =========================================================================

    def apply_alias(self, brand: str) -> Tuple[Optional[str], bool]:
        """
        Aplica alias a una marca detectada.

        Args:
            brand: Marca detectada por BrandDetectionService

        Returns:
            Tuple[Optional[str], bool]:
                - str or None: La marca resultante (target_brand, o None si excluded)
                - bool: True si se aplicó un alias

        Ejemplos:
            - "oralkin" → ("kin", True)      # action=ALIAS
            - "clorhexidina" → (None, True)  # action=EXCLUDE
            - "isdin" → ("isdin", False)     # No hay alias
        """
        brand_lower = brand.lower()

        # Usar cache si existe
        if self._alias_cache is not None:
            if brand_lower in self._alias_cache:
                target, action = self._alias_cache[brand_lower]
                # No incrementamos usage aquí (se hace en DB lookup para tracking)
                return (target, True)
            return (brand, False)

        # Buscar en DB
        alias = (
            self.db.query(BrandAlias)
            .filter(
                BrandAlias.source_brand == brand_lower,
                BrandAlias.is_active == True,  # noqa: E712
            )
            .first()
        )

        if not alias:
            return (brand, False)

        # Incrementar contador de uso
        alias.increment_usage()
        self.db.commit()

        if alias.action == BrandAliasAction.EXCLUDE:
            return (None, True)
        else:
            return (alias.target_brand, True)

    def load_cache(self) -> None:
        """
        Carga todos los aliases activos en cache.

        Llamar al inicio para optimizar detección en batch.
        """
        aliases = (
            self.db.query(BrandAlias)
            .filter(BrandAlias.is_active == True)  # noqa: E712
            .all()
        )

        self._alias_cache = {}
        for alias in aliases:
            if alias.action == BrandAliasAction.EXCLUDE:
                self._alias_cache[alias.source_brand] = (None, alias.action)
            else:
                self._alias_cache[alias.source_brand] = (alias.target_brand, alias.action)

        logger.info(f"Loaded {len(self._alias_cache)} brand aliases into cache")

    def get_all_active_aliases(self) -> Dict[str, Tuple[Optional[str], BrandAliasAction]]:
        """
        Obtener todos los aliases activos como dict.

        Returns:
            Dict {source_brand: (target_brand, action)}
        """
        if self._alias_cache is None:
            self.load_cache()
        return self._alias_cache or {}

    # =========================================================================
    # Reprocess: Update existing products
    # =========================================================================

    def _get_affected_count(self, source_brand: str) -> int:
        """
        Cuenta productos afectados por un alias.

        Args:
            source_brand: Marca origen del alias

        Returns:
            Número de SalesEnrichment con detected_brand = source_brand
        """
        return (
            self.db.query(SalesEnrichment)
            .filter(SalesEnrichment.detected_brand == source_brand.lower())
            .count()
        )

    def reprocess_alias(self, alias_id: int) -> dict:
        """
        Re-procesa productos afectados por un alias.

        Args:
            alias_id: ID del alias

        Returns:
            dict con updated_count y mensaje

        Raises:
            ValueError: Si el alias no existe o no está activo
        """
        alias = self.get_alias(alias_id)
        if not alias:
            raise ValueError(f"Alias id={alias_id} no encontrado")

        if not alias.is_active:
            raise ValueError(f"Alias id={alias_id} no está activo")

        source_brand = alias.source_brand

        if alias.action == BrandAliasAction.ALIAS:
            # Renombrar: source_brand → target_brand
            updated_count = (
                self.db.query(SalesEnrichment)
                .filter(SalesEnrichment.detected_brand == source_brand)
                .update(
                    {SalesEnrichment.detected_brand: alias.target_brand},
                    synchronize_session=False,
                )
            )
            message = f"Renombrado '{source_brand}' → '{alias.target_brand}' en {updated_count} productos"
        else:
            # Exclude: source_brand → NULL
            updated_count = (
                self.db.query(SalesEnrichment)
                .filter(SalesEnrichment.detected_brand == source_brand)
                .update(
                    {SalesEnrichment.detected_brand: None},
                    synchronize_session=False,
                )
            )
            message = f"Eliminado '{source_brand}' (no es marca) de {updated_count} productos"

        self.db.commit()

        logger.info(f"Reprocessed alias id={alias_id}: {message}")

        return {
            "alias_id": alias_id,
            "source_brand": source_brand,
            "target_brand": alias.target_brand,
            "action": alias.action,
            "updated_count": updated_count,
            "message": message,
        }

    # =========================================================================
    # Statistics
    # =========================================================================

    def _get_stats(self) -> dict:
        """Calcular estadísticas de aliases.

        OPTIMIZED: Single aggregated query for counts (was 7 queries → 3 queries).
        """
        from sqlalchemy import case

        # Single aggregated query for all counts
        stats = self.db.query(
            func.count(BrandAlias.id).label("total"),
            func.sum(case((BrandAlias.is_active == True, 1), else_=0)).label("active"),  # noqa: E712
            func.sum(case((BrandAlias.action == BrandAliasAction.ALIAS, 1), else_=0)).label("aliases"),
            func.sum(case((BrandAlias.action == BrandAliasAction.EXCLUDE, 1), else_=0)).label("excludes"),
            func.coalesce(func.sum(BrandAlias.usage_count), 0).label("total_usage"),
        ).first()

        total = stats.total or 0
        active = stats.active or 0
        inactive = total - active
        aliases_count = stats.aliases or 0
        excludes_count = stats.excludes or 0
        total_usage = stats.total_usage or 0

        # Separate queries for most/least used (require ORDER BY)
        most_used = (
            self.db.query(BrandAlias)
            .filter(BrandAlias.usage_count > 0)
            .order_by(BrandAlias.usage_count.desc())
            .first()
        )

        least_used = (
            self.db.query(BrandAlias)
            .filter(BrandAlias.is_active == True)  # noqa: E712
            .order_by(BrandAlias.usage_count.asc())
            .first()
        )

        return {
            "total": total,
            "active": active,
            "inactive": inactive,
            "aliases": aliases_count,
            "excludes": excludes_count,
            "total_usage": total_usage,
            "most_used": most_used.source_brand if most_used else None,
            "least_used": least_used.source_brand if least_used else None,
        }

    def get_stats(self) -> dict:
        """Obtener estadísticas públicas."""
        return self._get_stats()
