"""
BasketAnalysisService - Market Basket Analysis para productos venta libre (Issue #494)

Proporciona:
- Identificación de cestas (tickets) usando invoice_number
- Análisis de productos complementarios (Apriori simplificado)
- Productos alternativos (mismo NECESIDAD, diferente marca)
- Recomendaciones algorítmicas (Decision Box)
"""

from collections import defaultdict
from datetime import date, timedelta
from typing import Dict, List, Optional, Tuple
from uuid import UUID

import structlog
from sqlalchemy import and_, case, distinct, func, or_, text
from sqlalchemy.orm import Session

from app.models.inventory_snapshot import InventorySnapshot
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment
from app.services.seasonality_service import seasonality_service

logger = structlog.get_logger(__name__)


# Configuración de umbrales para MBA
DEFAULT_MBA_DAYS = 90  # 3 meses por defecto
MIN_SUPPORT_DEFAULT = 0.01  # 1% de cestas mínimo
MIN_CONFIDENCE_DEFAULT = 0.1  # 10% probabilidad mínima
SAMPLING_THRESHOLD = 10000  # Aplicar sampling si hay más de 10k cestas
SAMPLING_RATE = 0.2  # 20% de muestreo

# Umbrales para Decision Box (semáforo)
DANGER_COVERAGE_DAYS = 90
DANGER_TREND_THRESHOLD = -10.0
WARNING_MARGIN_VS_CATEGORY = 0.0
OPPORTUNITY_COVERAGE_DAYS = 7
OPPORTUNITY_TREND_THRESHOLD = 15.0


class BasketAnalysisService:
    """
    Servicio para análisis de cestas (Market Basket Analysis) en productos venta libre.

    Funcionalidades:
    - Identificar cestas/tickets usando invoice_number
    - Calcular productos complementarios (Support, Confidence, Lift)
    - Obtener alternativas por categoría NECESIDAD
    - Generar recomendaciones algorítmicas (semáforo + acción)
    """

    def identify_baskets(
        self,
        db: Session,
        pharmacy_id: UUID,
        date_from: date,
        date_to: date,
        product_code: Optional[str] = None,
    ) -> Tuple[List[Dict], int]:
        """
        Identificar cestas/tickets usando GROUP BY (pharmacy_id, invoice_number, sale_date::DATE).

        Excluye:
        - Devoluciones (quantity <= 0, is_return=True)
        - Cancelaciones (is_cancelled=True)

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            date_from: Fecha inicio
            date_to: Fecha fin
            product_code: Opcional - filtrar cestas que contienen este producto

        Returns:
            Tuple[List[Dict], int]: Lista de cestas y total
        """
        try:
            # Construir query base
            query = (
                db.query(
                    SalesData.invoice_number,
                    func.date(SalesData.sale_date).label("basket_date"),
                    SalesData.employee_code,
                    func.count(distinct(SalesData.codigo_nacional)).label("unique_products"),
                    func.sum(SalesData.quantity).label("total_units"),
                    func.sum(SalesData.total_amount).label("total_amount"),
                )
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    # Excluir devoluciones y cancelaciones
                    SalesData.quantity > 0,
                    or_(SalesData.is_return == False, SalesData.is_return.is_(None)),
                    or_(SalesData.is_cancelled == False, SalesData.is_cancelled.is_(None)),
                    # Excluir cestas sin invoice_number
                    SalesData.invoice_number.isnot(None),
                    SalesData.invoice_number != "",
                )
            )

            # Si se especifica producto, filtrar cestas que lo contienen
            if product_code:
                # Subquery: cestas con el producto
                baskets_with_product = (
                    db.query(
                        SalesData.invoice_number,
                        func.date(SalesData.sale_date).label("basket_date"),
                    )
                    .filter(
                        SalesData.pharmacy_id == pharmacy_id,
                        SalesData.sale_date >= date_from,
                        SalesData.sale_date <= date_to,
                        SalesData.codigo_nacional == product_code,
                        SalesData.quantity > 0,
                    )
                    .distinct()
                    .subquery()
                )

                query = query.filter(
                    and_(
                        SalesData.invoice_number == baskets_with_product.c.invoice_number,
                        func.date(SalesData.sale_date) == baskets_with_product.c.basket_date,
                    )
                )

            # Agrupar por cesta
            query = query.group_by(
                SalesData.invoice_number,
                func.date(SalesData.sale_date),
                SalesData.employee_code,
            )

            # Ejecutar query
            results = query.all()

            baskets = [
                {
                    "invoice_number": r.invoice_number,
                    "basket_date": r.basket_date.isoformat() if r.basket_date else None,
                    "employee_code": r.employee_code,
                    "unique_products": r.unique_products,
                    "total_units": r.total_units,
                    "total_amount": float(r.total_amount) if r.total_amount else 0.0,
                }
                for r in results
            ]

            logger.info(
                "basket_analysis.identify_baskets",
                pharmacy_id=str(pharmacy_id),
                date_from=str(date_from),
                date_to=str(date_to),
                product_code=product_code,
                baskets_found=len(baskets),
            )

            return baskets, len(baskets)

        except Exception as e:
            logger.error(
                "basket_analysis.identify_baskets.error",
                pharmacy_id=str(pharmacy_id),
                error=str(e),
            )
            return [], 0

    def get_complementary_products(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
        days: int = DEFAULT_MBA_DAYS,
        min_support: float = MIN_SUPPORT_DEFAULT,
        min_confidence: float = MIN_CONFIDENCE_DEFAULT,
        limit: int = 10,
    ) -> Dict:
        """
        Calcular productos complementarios usando Market Basket Analysis (Apriori simplificado).

        Métricas:
        - Support: % de cestas que contienen ambos productos
        - Confidence: P(complementario | base) = cestas con ambos / cestas con base
        - Lift: Confidence / Support(complementario) - mide asociación real vs azar

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            product_code: Código del producto base
            days: Días de análisis (default: 90 = 3 meses)
            min_support: Support mínimo para incluir producto
            min_confidence: Confidence mínimo para incluir producto
            limit: Máximo de productos a retornar

        Returns:
            Dict con productos complementarios y metadatos
        """
        try:
            date_to = date.today()
            date_from = date_to - timedelta(days=days)

            # 1. Contar total de cestas en el período
            total_baskets_query = (
                db.query(func.count(distinct(SalesData.invoice_number)))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.quantity > 0,
                    or_(SalesData.is_return == False, SalesData.is_return.is_(None)),
                    SalesData.invoice_number.isnot(None),
                )
            )
            total_baskets = total_baskets_query.scalar() or 0

            if total_baskets == 0:
                return self._empty_complementary_response(product_code, days)

            # 2. Contar cestas que contienen el producto base
            base_baskets_query = (
                db.query(func.count(distinct(SalesData.invoice_number)))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.codigo_nacional == product_code,
                    SalesData.quantity > 0,
                    or_(SalesData.is_return == False, SalesData.is_return.is_(None)),
                )
            )
            base_basket_count = base_baskets_query.scalar() or 0

            if base_basket_count == 0:
                return self._empty_complementary_response(product_code, days)

            # 3. Determinar si aplicar sampling
            apply_sampling = base_basket_count > SAMPLING_THRESHOLD
            sample_rate = SAMPLING_RATE if apply_sampling else 1.0

            # 4. Obtener cestas con el producto base
            base_baskets_subquery = (
                db.query(SalesData.invoice_number)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.codigo_nacional == product_code,
                    SalesData.quantity > 0,
                    or_(SalesData.is_return == False, SalesData.is_return.is_(None)),
                )
                .distinct()
            )

            # Aplicar sampling si es necesario
            if apply_sampling:
                # Usar TABLESAMPLE o random() para sampling
                base_baskets_subquery = base_baskets_subquery.filter(
                    func.random() < sample_rate
                )

            base_baskets_list = [r[0] for r in base_baskets_subquery.all()]
            sampled_basket_count = len(base_baskets_list)

            if sampled_basket_count == 0:
                return self._empty_complementary_response(product_code, days)

            # Limitar IN clause para evitar degradación de performance
            if len(base_baskets_list) > 5000:
                logger.warning(
                    "basket_analysis.large_in_clause_truncated",
                    original_count=len(base_baskets_list),
                    truncated_to=5000,
                    pharmacy_id=str(pharmacy_id),
                    product_code=product_code,
                    message="Truncating IN clause to 5000 baskets for performance",
                )
                base_baskets_list = base_baskets_list[:5000]

            # 5. Calcular co-ocurrencia con otros productos
            co_occurrence_query = (
                db.query(
                    SalesData.codigo_nacional.label("complementary_code"),
                    func.count(distinct(SalesData.invoice_number)).label("co_occurrence"),
                )
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.invoice_number.in_(base_baskets_list),
                    SalesData.codigo_nacional != product_code,  # Excluir producto base
                    SalesData.quantity > 0,
                    or_(SalesData.is_return == False, SalesData.is_return.is_(None)),
                )
                .group_by(SalesData.codigo_nacional)
            )

            co_occurrences = {r.complementary_code: r.co_occurrence for r in co_occurrence_query.all()}

            if not co_occurrences:
                return self._empty_complementary_response(product_code, days)

            # 6. Calcular Support individual de cada producto complementario
            individual_support_query = (
                db.query(
                    SalesData.codigo_nacional,
                    func.count(distinct(SalesData.invoice_number)).label("basket_count"),
                )
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.codigo_nacional.in_(list(co_occurrences.keys())),
                    SalesData.quantity > 0,
                    or_(SalesData.is_return == False, SalesData.is_return.is_(None)),
                )
                .group_by(SalesData.codigo_nacional)
            )

            individual_support = {
                r.codigo_nacional: r.basket_count / total_baskets
                for r in individual_support_query.all()
            }

            # 7. Calcular métricas MBA para cada producto
            complementary_products = []
            adjusted_base_count = sampled_basket_count if apply_sampling else base_basket_count

            for code, co_count in co_occurrences.items():
                # Confidence = P(complementario | base)
                confidence = co_count / adjusted_base_count

                # Support del complementario
                support = individual_support.get(code, 0)

                # Lift = Confidence / Support(complementario)
                lift = confidence / support if support > 0 else 0

                # Filtrar por umbrales
                if confidence >= min_confidence and support >= min_support:
                    complementary_products.append({
                        "codigo_nacional": code,
                        "co_occurrence_count": co_count,
                        "support": round(support, 4),
                        "confidence": round(confidence, 4),
                        "lift": round(lift, 2),
                    })

            # 8. Ordenar por Lift descendente y limitar
            complementary_products.sort(key=lambda x: x["lift"], reverse=True)
            complementary_products = complementary_products[:limit]

            # 9. Enriquecer con información de productos
            if complementary_products:
                codes = [p["codigo_nacional"] for p in complementary_products]
                product_info = self._get_product_info(db, pharmacy_id, codes)

                for p in complementary_products:
                    info = product_info.get(p["codigo_nacional"], {})
                    p["product_name"] = info.get("product_name", "Desconocido")
                    p["detected_brand"] = info.get("detected_brand")
                    p["ml_category"] = info.get("ml_category")
                    # Labels humanos
                    p["support_label"] = f"{p['support'] * 100:.1f}% cestas"
                    p["confidence_label"] = f"{p['confidence'] * 100:.0f}% probabilidad"
                    p["lift_level"] = (
                        "high" if p["lift"] > 1.5
                        else "medium" if p["lift"] > 1.0
                        else "low"
                    )

            # Obtener nombre del producto base
            base_info = self._get_product_info(db, pharmacy_id, [product_code])
            base_name = base_info.get(product_code, {}).get("product_name", "Desconocido")

            logger.info(
                "basket_analysis.complementary_products",
                pharmacy_id=str(pharmacy_id),
                product_code=product_code,
                days=days,
                total_baskets=total_baskets,
                base_basket_count=base_basket_count,
                complementary_found=len(complementary_products),
                sampling_applied=apply_sampling,
            )

            return {
                "complementary": complementary_products,
                "base_product_code": product_code,
                "base_product_name": base_name,
                "base_product_basket_count": base_basket_count,
                "analysis_period_days": days,
                "sampling_applied": apply_sampling,
                "sample_rate": sample_rate if apply_sampling else None,
            }

        except Exception as e:
            logger.error(
                "basket_analysis.complementary_products.error",
                pharmacy_id=str(pharmacy_id),
                product_code=product_code,
                error=str(e),
            )
            return self._empty_complementary_response(product_code, days)

    def get_alternatives_by_necesidad(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
        date_from: date,
        date_to: date,
        limit: int = 10,
    ) -> Dict:
        """
        Obtener productos alternativos: mismo NECESIDAD (ml_category), diferente marca.

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            product_code: Código del producto base
            date_from: Fecha inicio
            date_to: Fecha fin
            limit: Máximo de alternativas a retornar

        Returns:
            Dict con alternativas y metadatos
        """
        try:
            # 1. Obtener información del producto base
            base_product = (
                db.query(
                    SalesData.codigo_nacional,
                    SalesData.product_name,
                    SalesEnrichment.detected_brand,
                    SalesEnrichment.ml_category,
                    func.sum(SalesData.total_amount).label("total_sales"),
                    func.avg(SalesData.margin_percentage).label("avg_margin"),
                )
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.quantity > 0,
                )
                .group_by(
                    SalesData.codigo_nacional,
                    SalesData.product_name,
                    SalesEnrichment.detected_brand,
                    SalesEnrichment.ml_category,
                )
                .first()
            )

            if not base_product or not base_product.ml_category:
                return {
                    "alternatives": [],
                    "total_count": 0,
                    "category": "unknown",
                    "base_product_margin": 0.0,
                    "message": "Producto base no encontrado o sin categoría NECESIDAD",
                }

            base_category = base_product.ml_category
            base_brand = base_product.detected_brand
            base_margin = float(base_product.avg_margin) if base_product.avg_margin else 0.0

            # 2. Buscar alternativas: misma categoría, diferente marca
            alternatives_query = (
                db.query(
                    SalesData.codigo_nacional,
                    func.max(SalesData.product_name).label("product_name"),
                    func.max(SalesEnrichment.detected_brand).label("detected_brand"),
                    func.sum(SalesData.total_amount).label("total_sales"),
                    func.avg(SalesData.margin_percentage).label("margin_percent"),
                )
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.quantity > 0,
                    SalesEnrichment.ml_category == base_category,
                    SalesData.codigo_nacional != product_code,  # Excluir producto base
                )
            )

            # Excluir misma marca (si tiene marca)
            if base_brand:
                alternatives_query = alternatives_query.filter(
                    or_(
                        SalesEnrichment.detected_brand != base_brand,
                        SalesEnrichment.detected_brand.is_(None),
                    )
                )

            alternatives_query = (
                alternatives_query
                .group_by(SalesData.codigo_nacional)
                .order_by(func.sum(SalesData.total_amount).desc())
                .limit(limit)
            )

            alternatives = []
            for r in alternatives_query.all():
                margin = float(r.margin_percent) if r.margin_percent else 0.0
                alternatives.append({
                    "codigo_nacional": r.codigo_nacional,
                    "product_name": r.product_name or "Desconocido",
                    "detected_brand": r.detected_brand,
                    "total_sales": round(float(r.total_sales or 0), 2),
                    "margin_percent": round(margin, 2),
                    "margin_diff_vs_selected": round(margin - base_margin, 2),
                    "gmroi": None,  # Se calculará por separado si es necesario
                })

            # Contar total de alternativas (sin limit)
            total_count_query = (
                db.query(func.count(distinct(SalesData.codigo_nacional)))
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.quantity > 0,
                    SalesEnrichment.ml_category == base_category,
                    SalesData.codigo_nacional != product_code,
                )
            )

            if base_brand:
                total_count_query = total_count_query.filter(
                    or_(
                        SalesEnrichment.detected_brand != base_brand,
                        SalesEnrichment.detected_brand.is_(None),
                    )
                )

            total_count = total_count_query.scalar() or 0

            logger.info(
                "basket_analysis.alternatives",
                pharmacy_id=str(pharmacy_id),
                product_code=product_code,
                category=base_category,
                base_brand=base_brand,
                alternatives_found=len(alternatives),
                total_count=total_count,
            )

            return {
                "alternatives": alternatives,
                "total_count": total_count,
                "category": base_category,
                "base_product_margin": round(base_margin, 2),
            }

        except Exception as e:
            logger.error(
                "basket_analysis.alternatives.error",
                pharmacy_id=str(pharmacy_id),
                product_code=product_code,
                error=str(e),
            )
            return {
                "alternatives": [],
                "total_count": 0,
                "category": "error",
                "base_product_margin": 0.0,
            }

    def get_product_recommendation(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
    ) -> Dict:
        """
        Generar recomendación algorítmica para el Decision Box.

        Lógica del semáforo:
        - ROJO (danger): coverage_days > 90 Y trend_MAT < -10%
        - AMARILLO (warning): margin_vs_category < 0 PERO sales_volume alto
        - VERDE (success): coverage_days < 7 Y trend_MAT > 15%

        Args:
            db: Sesión de base de datos
            pharmacy_id: ID de la farmacia
            product_code: Código del producto

        Returns:
            Dict con recomendación y factores
        """
        try:
            # 1. Obtener información del producto
            product_info = self._get_product_info(db, pharmacy_id, [product_code])
            info = product_info.get(product_code, {})
            product_name = info.get("product_name", "Desconocido")

            # 2. Calcular días de cobertura
            coverage_days = self._calculate_coverage_days(db, pharmacy_id, product_code)

            # 3. Calcular tendencia MAT YoY
            trend_mat = self._calculate_mat_trend(db, pharmacy_id, product_code)

            # 4. Calcular margen vs categoría
            margin_vs_category, sales_volume_rank = self._calculate_margin_vs_category(
                db, pharmacy_id, product_code
            )

            # 5. Verificar si hay mejores alternativas
            has_better_alternatives = self._check_better_alternatives(
                db, pharmacy_id, product_code
            )

            # 6. Calcular GMROI
            gmroi = self._calculate_simple_gmroi(db, pharmacy_id, product_code)

            # 7. Calcular momentum estacional (Issue #507)
            category = info.get("ml_category")
            momentum_seasonal, seasonality_index, seasonality_context = (
                self._calculate_momentum_seasonal(db, pharmacy_id, product_code, category)
            )

            # 8. Aplicar lógica del semáforo (con contexto estacional)
            alert_level, action, confidence, reasons = self._apply_decision_logic(
                coverage_days=coverage_days,
                trend_mat=trend_mat,
                margin_vs_category=margin_vs_category,
                sales_volume_rank=sales_volume_rank,
                gmroi=gmroi,
                has_better_alternatives=has_better_alternatives,
                momentum_seasonal=momentum_seasonal,
                seasonality_index=seasonality_index,
                seasonality_context=seasonality_context,
            )

            # 9. Calcular cantidad sugerida (si aplica)
            suggested_quantity = None
            if action == "comprar":
                suggested_quantity = self._calculate_suggested_quantity(
                    db, pharmacy_id, product_code
                )

            logger.info(
                "basket_analysis.recommendation",
                pharmacy_id=str(pharmacy_id),
                product_code=product_code,
                alert_level=alert_level,
                action=action,
                confidence=confidence,
                seasonality_index=seasonality_index,
                momentum_seasonal=momentum_seasonal,
            )

            return {
                "recommendation": {
                    "action": action,
                    "alert_level": alert_level,
                    "confidence": confidence,
                    "reasons": reasons,
                    "suggested_quantity": suggested_quantity,
                },
                "factors": {
                    "coverage_days": coverage_days,
                    "trend_mat": trend_mat,
                    "margin_vs_category": margin_vs_category,
                    "sales_volume_rank": sales_volume_rank,
                    "gmroi": gmroi,
                    "has_better_alternatives": has_better_alternatives,
                    # Issue #507: Factores de estacionalidad
                    "momentum_seasonal": momentum_seasonal,
                    "seasonality_index": seasonality_index,
                    "seasonality_context": seasonality_context,
                },
                "product_code": product_code,
                "product_name": product_name,
            }

        except Exception as e:
            logger.error(
                "basket_analysis.recommendation.error",
                pharmacy_id=str(pharmacy_id),
                product_code=product_code,
                error=str(e),
            )
            return {
                "recommendation": {
                    "action": "mantener",
                    "alert_level": "info",
                    "confidence": 0.0,
                    "reasons": ["Error al calcular recomendación"],
                    "suggested_quantity": None,
                },
                "factors": {},
                "product_code": product_code,
                "product_name": "Error",
            }

    # ========== HELPER METHODS ==========

    def _empty_complementary_response(self, product_code: str, days: int) -> Dict:
        """Respuesta vacía para productos complementarios."""
        return {
            "complementary": [],
            "base_product_code": product_code,
            "base_product_name": "Desconocido",
            "base_product_basket_count": 0,
            "analysis_period_days": days,
            "sampling_applied": False,
            "sample_rate": None,
        }

    def _get_product_info(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_codes: List[str],
    ) -> Dict[str, Dict]:
        """Obtener información de productos (nombre, marca, categoría)."""
        if not product_codes:
            return {}

        query = (
            db.query(
                SalesData.codigo_nacional,
                func.max(SalesData.product_name).label("product_name"),
                func.max(SalesEnrichment.detected_brand).label("detected_brand"),
                func.max(SalesEnrichment.ml_category).label("ml_category"),
            )
            .outerjoin(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
            .filter(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.codigo_nacional.in_(product_codes),
            )
            .group_by(SalesData.codigo_nacional)
        )

        return {
            r.codigo_nacional: {
                "product_name": r.product_name or "Desconocido",
                "detected_brand": r.detected_brand,
                "ml_category": r.ml_category,
            }
            for r in query.all()
        }

    def _calculate_coverage_days(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
    ) -> Optional[float]:
        """Calcular días de cobertura con stock actual."""
        try:
            # Obtener stock actual
            stock_query = (
                db.query(func.sum(InventorySnapshot.stock_quantity))
                .filter(
                    InventorySnapshot.pharmacy_id == pharmacy_id,
                    InventorySnapshot.product_code == product_code,
                )
            )
            stock = stock_query.scalar() or 0

            if stock == 0:
                return 0.0

            # Calcular venta diaria promedio (últimos 90 días)
            date_to = date.today()
            date_from = date_to - timedelta(days=90)

            sales_query = (
                db.query(func.sum(SalesData.quantity))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.quantity > 0,
                )
            )
            total_units = sales_query.scalar() or 0

            if total_units == 0:
                return None  # Sin ventas recientes

            daily_sales = total_units / 90
            coverage = stock / daily_sales if daily_sales > 0 else None

            return round(coverage, 1) if coverage else None

        except Exception:
            return None

    def _calculate_mat_trend(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
    ) -> Optional[float]:
        """Calcular tendencia MAT Year-over-Year."""
        try:
            today = date.today()

            # MAT actual (últimos 12 meses)
            current_start = today - timedelta(days=365)
            current_mat = (
                db.query(func.sum(SalesData.total_amount))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= current_start,
                    SalesData.sale_date <= today,
                    SalesData.quantity > 0,
                )
                .scalar() or 0
            )

            # MAT año anterior
            previous_start = today - timedelta(days=730)
            previous_end = today - timedelta(days=365)
            previous_mat = (
                db.query(func.sum(SalesData.total_amount))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= previous_start,
                    SalesData.sale_date <= previous_end,
                    SalesData.quantity > 0,
                )
                .scalar() or 0
            )

            if previous_mat == 0:
                return None  # Sin datos del año anterior

            trend = ((current_mat - previous_mat) / previous_mat) * 100
            return round(trend, 1)

        except Exception:
            return None

    def _calculate_margin_vs_category(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
    ) -> Tuple[Optional[float], Optional[str]]:
        """Calcular diferencial de margen vs categoría y ranking de volumen."""
        try:
            date_from = date.today() - timedelta(days=365)
            date_to = date.today()

            # Obtener categoría y margen del producto
            product_query = (
                db.query(
                    SalesEnrichment.ml_category,
                    func.avg(SalesData.margin_percentage).label("product_margin"),
                    func.sum(SalesData.total_amount).label("product_sales"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                )
                .group_by(SalesEnrichment.ml_category)
                .first()
            )

            if not product_query or not product_query.ml_category:
                return None, None

            category = product_query.ml_category
            product_margin = float(product_query.product_margin or 0)
            product_sales = float(product_query.product_sales or 0)

            # Obtener margen promedio de la categoría
            category_margin_query = (
                db.query(
                    func.avg(SalesData.margin_percentage).label("category_margin"),
                )
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesEnrichment.ml_category == category,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                )
                .first()
            )

            category_margin = float(category_margin_query.category_margin or 0) if category_margin_query else 0
            margin_diff = product_margin - category_margin

            # Calcular ventas promedio por producto en la categoría (usando subquery)
            # Primero: subquery para ventas por producto
            product_sales_subquery = (
                db.query(
                    SalesData.codigo_nacional,
                    func.sum(SalesData.total_amount).label("product_total"),
                )
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesEnrichment.ml_category == category,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                )
                .group_by(SalesData.codigo_nacional)
                .subquery()
            )

            # Segundo: promedio de ventas por producto
            avg_sales_query = (
                db.query(func.avg(product_sales_subquery.c.product_total))
                .scalar()
            )
            avg_sales = float(avg_sales_query or 0)

            # Calcular ranking de volumen
            # Simplificado: alto (>150% promedio), medio (50-150%), bajo (<50%)
            sales_volume_rank = "medio"  # Default
            if avg_sales > 0:
                if product_sales > avg_sales * 1.5:
                    sales_volume_rank = "alto"
                elif product_sales < avg_sales * 0.5:
                    sales_volume_rank = "bajo"

            return round(margin_diff, 2), sales_volume_rank

        except Exception:
            return None, None

    def _check_better_alternatives(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
    ) -> bool:
        """Verificar si existen alternativas con mejor margen."""
        try:
            date_from = date.today() - timedelta(days=365)
            date_to = date.today()

            # Obtener margen del producto base y su categoría
            base_query = (
                db.query(
                    SalesEnrichment.ml_category,
                    func.avg(SalesData.margin_percentage).label("base_margin"),
                )
                .join(SalesData, SalesEnrichment.sales_data_id == SalesData.id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                )
                .group_by(SalesEnrichment.ml_category)
                .first()
            )

            if not base_query or not base_query.ml_category:
                return False

            base_margin = float(base_query.base_margin or 0)
            category = base_query.ml_category

            # Buscar alternativas con mejor margen usando LIMIT 1 (más eficiente que count)
            # Solo necesitamos saber si existe AL MENOS UNA alternativa mejor
            better_alternative = (
                db.query(SalesData.codigo_nacional)
                .join(SalesEnrichment, SalesData.id == SalesEnrichment.sales_data_id)
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesEnrichment.ml_category == category,
                    SalesData.codigo_nacional != product_code,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                )
                .group_by(SalesData.codigo_nacional)
                .having(func.avg(SalesData.margin_percentage) > base_margin + 3)  # >3 pp mejor
                .limit(1)
                .first()
            )

            return better_alternative is not None

        except Exception:
            return False

    def _calculate_simple_gmroi(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
    ) -> Optional[float]:
        """Calcular GMROI simplificado para el producto."""
        try:
            date_from = date.today() - timedelta(days=365)
            date_to = date.today()

            # Margen bruto anual
            margin_query = (
                db.query(func.sum(SalesData.margin_amount))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.quantity > 0,
                )
            )
            gross_margin = float(margin_query.scalar() or 0)

            # Inversión en stock
            stock_investment = (
                db.query(
                    func.sum(
                        InventorySnapshot.stock_quantity *
                        func.coalesce(InventorySnapshot.unit_cost, InventorySnapshot.unit_price, 0)
                    )
                )
                .filter(
                    InventorySnapshot.pharmacy_id == pharmacy_id,
                    InventorySnapshot.product_code == product_code,
                )
                .scalar() or 0
            )

            if stock_investment == 0:
                return None

            gmroi = gross_margin / float(stock_investment)
            return round(gmroi, 2)

        except Exception:
            return None

    def _calculate_momentum_seasonal(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
        category: Optional[str],
    ) -> Tuple[Optional[float], float, Optional[str]]:
        """
        Calcular momentum ajustado por estacionalidad (Issue #507).

        NO modifica MAT. Calcula momentum corto plazo vs expectativa estacional.

        Returns:
            Tuple[momentum_pct, seasonality_index, context_message]
        """
        try:
            today = date.today()
            current_month = today.month

            # Si no hay categoria, no aplicar estacionalidad
            if not category:
                return None, 1.0, None

            # Obtener indice estacional (batch loading para una categoria)
            index = seasonality_service.get_index(db, category, current_month, pharmacy_id)

            # Ventas ultimos 30 dias
            date_from_30 = today - timedelta(days=30)
            current_sales = (
                db.query(func.sum(SalesData.total_amount))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= date_from_30,
                    SalesData.quantity > 0,
                )
                .scalar() or 0
            )

            # Promedio mensual historico (ultimos 12 meses, excluyendo ultimo mes)
            date_from_12m = today - timedelta(days=395)
            date_to_12m = today - timedelta(days=35)

            historical_sales = (
                db.query(func.sum(SalesData.total_amount))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= date_from_12m,
                    SalesData.sale_date <= date_to_12m,
                    SalesData.quantity > 0,
                )
                .scalar() or 0
            )

            avg_monthly = float(historical_sales) / 12 if historical_sales > 0 else 0

            if avg_monthly == 0:
                return None, index, None

            # Calcular momentum
            momentum = seasonality_service.calculate_momentum(
                float(current_sales), avg_monthly, index
            )

            # Generar contexto
            context = seasonality_service.get_seasonality_context(index, momentum)

            logger.debug(
                "basket_analysis.momentum_seasonal",
                product_code=product_code,
                category=category,
                month=current_month,
                index=index,
                momentum=momentum,
                current_sales=float(current_sales),
                avg_monthly=avg_monthly,
            )

            return momentum, index, context

        except Exception as e:
            logger.warning(
                "basket_analysis.momentum_seasonal.error",
                product_code=product_code,
                error=str(e),
            )
            return None, 1.0, None

    def _apply_decision_logic(
        self,
        coverage_days: Optional[float],
        trend_mat: Optional[float],
        margin_vs_category: Optional[float],
        sales_volume_rank: Optional[str],
        gmroi: Optional[float],
        has_better_alternatives: bool,
        momentum_seasonal: Optional[float] = None,
        seasonality_index: Optional[float] = None,
        seasonality_context: Optional[str] = None,
    ) -> Tuple[str, str, float, List[str]]:
        """
        Aplicar lógica del semáforo para generar recomendación.

        Issue #507: Añade contexto estacional para evitar falsos positivos.
        Si MAT es negativo PERO momentum estacional es positivo, reduce severidad.

        Returns:
            Tuple[alert_level, action, confidence, reasons]
        """
        reasons = []
        scores = {"danger": 0, "warning": 0, "success": 0}

        # Regla ROJO: Sobrestock + tendencia negativa
        if coverage_days and trend_mat:
            if coverage_days > DANGER_COVERAGE_DAYS and trend_mat < DANGER_TREND_THRESHOLD:
                scores["danger"] += 2
                reasons.append(
                    f"Sobrestock ({coverage_days:.0f} días cobertura) con tendencia negativa ({trend_mat:+.1f}%)"
                )

        # Sobrestock solo
        if coverage_days and coverage_days > DANGER_COVERAGE_DAYS:
            scores["danger"] += 1
            if not any("Sobrestock" in r for r in reasons):
                reasons.append(f"Cobertura de {coverage_days:.0f} días (sobrestock)")

        # Tendencia muy negativa - CON CONTEXTO ESTACIONAL (Issue #507)
        if trend_mat and trend_mat < DANGER_TREND_THRESHOLD:
            # Si hay momentum positivo en temporada baja, reducir severidad
            if momentum_seasonal is not None and momentum_seasonal > 0 and seasonality_context:
                # Tendencia MAT negativa PERO rendimiento estacional positivo
                scores["warning"] += 1  # Amarillo en vez de rojo
                reasons.append(
                    f"Tendencia MAT: {trend_mat:+.1f}%, pero {seasonality_context}: {momentum_seasonal:+.1f}%"
                )
            else:
                scores["danger"] += 1
                if not any("tendencia negativa" in r for r in reasons):
                    reasons.append(f"Tendencia MAT negativa: {trend_mat:+.1f}%")

        # Regla AMARILLO: Margen bajo pero volumen alto
        if margin_vs_category and sales_volume_rank:
            if margin_vs_category < WARNING_MARGIN_VS_CATEGORY and sales_volume_rank == "alto":
                scores["warning"] += 2
                reasons.append(
                    f"Generador de tráfico con margen bajo ({margin_vs_category:+.1f}pp vs categoría)"
                )

        # Margen bajo (sin volumen alto)
        if margin_vs_category and margin_vs_category < WARNING_MARGIN_VS_CATEGORY - 5:
            scores["warning"] += 1
            if not any("margen bajo" in r.lower() for r in reasons):
                reasons.append(f"Margen {margin_vs_category:+.1f}pp vs categoría")

        # Mejores alternativas disponibles
        if has_better_alternatives:
            scores["warning"] += 1
            reasons.append("Existen alternativas con mejor margen (+3pp o más)")

        # Regla VERDE: Stock bajo + tendencia positiva
        if coverage_days and trend_mat:
            if coverage_days < OPPORTUNITY_COVERAGE_DAYS and trend_mat > OPPORTUNITY_TREND_THRESHOLD:
                scores["success"] += 2
                reasons.append(
                    f"Riesgo de rotura ({coverage_days:.0f} días) en producto en crecimiento ({trend_mat:+.1f}%)"
                )

        # Stock muy bajo solo
        if coverage_days and coverage_days < OPPORTUNITY_COVERAGE_DAYS:
            scores["success"] += 1
            if not any("rotura" in r.lower() for r in reasons):
                reasons.append(f"Stock bajo: {coverage_days:.0f} días de cobertura")

        # GMROI excelente
        if gmroi and gmroi > 3.0:
            scores["success"] += 1
            reasons.append(f"GMROI excelente: {gmroi:.1f}")

        # Determinar nivel de alerta
        if scores["danger"] >= 2:
            alert_level = "danger"
            action = "reducir" if coverage_days and coverage_days > DANGER_COVERAGE_DAYS else "eliminar"
            confidence = min(95, 60 + scores["danger"] * 15)
        elif scores["warning"] >= 2:
            alert_level = "warning"
            action = "revisar"
            confidence = min(85, 50 + scores["warning"] * 15)
        elif scores["success"] >= 2:
            alert_level = "success"
            action = "comprar"
            confidence = min(90, 55 + scores["success"] * 15)
        else:
            alert_level = "info"
            action = "mantener"
            confidence = 50
            if not reasons:
                reasons.append("Producto con métricas estables")

        return alert_level, action, confidence, reasons

    def _calculate_suggested_quantity(
        self,
        db: Session,
        pharmacy_id: UUID,
        product_code: str,
        target_days: int = 30,
    ) -> Optional[int]:
        """Calcular cantidad sugerida para próximo pedido."""
        try:
            # Venta diaria promedio (últimos 90 días)
            date_from = date.today() - timedelta(days=90)
            date_to = date.today()

            sales = (
                db.query(func.sum(SalesData.quantity))
                .filter(
                    SalesData.pharmacy_id == pharmacy_id,
                    SalesData.codigo_nacional == product_code,
                    SalesData.sale_date >= date_from,
                    SalesData.sale_date <= date_to,
                    SalesData.quantity > 0,
                )
                .scalar() or 0
            )

            daily_sales = sales / 90

            # Stock actual
            stock = (
                db.query(func.sum(InventorySnapshot.stock_quantity))
                .filter(
                    InventorySnapshot.pharmacy_id == pharmacy_id,
                    InventorySnapshot.product_code == product_code,
                )
                .scalar() or 0
            )

            # Calcular cantidad para target_days de cobertura
            target_units = daily_sales * target_days
            needed = max(0, int(target_units - stock))

            return needed if needed > 0 else None

        except Exception:
            return None
