﻿# backend/app/services/auth_service.py
"""
Authentication service for OAuth2 and traditional authentication
Handles user registration, login, and social authentication
"""

import logging
import secrets
from datetime import timedelta
from typing import Any, Dict, Optional
from uuid import uuid4

from sqlalchemy.orm import Session

from app.core.security import (
    create_access_token,
    create_refresh_token,
    get_password_hash,
    verify_password,
    verify_token,
)
from app.models.pharmacy import Pharmacy
from app.models.user import User
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)

# Temporary storage for OAuth tokens (in production, use Redis or database)
# Keys expire after 5 minutes
OAUTH_TOKEN_STORE = {}


def clean_expired_tokens():
    """Remove expired tokens from the store"""
    now = utc_now()
    expired = [k for k, v in OAUTH_TOKEN_STORE.items() if v["expires_at"] < now]
    for key in expired:
        del OAUTH_TOKEN_STORE[key]


class AuthService:
    """Service for handling authentication operations"""

    def authenticate_user(self, db: Session, email: str, password: str) -> Optional[User]:
        """
        Authenticate user with email and password

        Args:
            db: Database session
            email: User email
            password: Plain text password

        Returns:
            User if authentication successful, None otherwise
        """
        user = db.query(User).filter(User.email == email).first()

        if not user:
            return None

        if not verify_password(password, user.hashed_password):
            return None

        # Update last login
        user.last_login = utc_now()
        db.commit()

        return user

    def create_user_from_oauth(
        self,
        db: Session,
        email: str,
        full_name: str,
        provider: str,
        provider_user_id: str,
    ) -> User:
        """
        Create or get user from OAuth provider data

        Args:
            db: Database session
            email: User email from OAuth provider
            full_name: User full name from OAuth provider
            provider: OAuth provider name (google, microsoft)
            provider_user_id: User ID from OAuth provider

        Returns:
            User object
        """
        # Check if user already exists
        user = db.query(User).filter(User.email == email).first()

        if user:
            # Update last login
            user.last_login = utc_now()

            # Update OAuth provider info if not set
            if not user.oauth_provider:
                user.oauth_provider = provider
                user.oauth_provider_id = provider_user_id

            # Verificar que el usuario tiene farmacia asignada
            # Si no tiene, crear una automáticamente
            if not user.pharmacy_id:
                logger.info(f"Creating pharmacy for existing OAuth user without pharmacy: {email}")
                pharmacy = Pharmacy(
                    id=uuid4(),
                    name=f"Farmacia de {full_name}",
                    code=f"OAUTH-{uuid4().hex[:8].upper()}",  # Código único generado
                    email=email,
                )
                db.add(pharmacy)
                db.flush()  # Flush para obtener el ID
                user.pharmacy_id = pharmacy.id
                logger.info(f"Pharmacy created for user {email} with ID: {pharmacy.id}")

            db.commit()
            return user

        # Create new user
        username = email.split("@")[0]
        # Ensure username is unique
        base_username = username
        counter = 1
        while db.query(User).filter(User.username == username).first():
            username = f"{base_username}{counter}"
            counter += 1

        # IMPORTANTE: Crear farmacia automáticamente para OAuth users
        # Esto asegura que todos los usuarios tengan una farmacia (constraint NOT NULL)
        logger.info(f"Creating pharmacy for new OAuth user: {email}")
        pharmacy = Pharmacy(
            id=uuid4(),
            name=f"Farmacia de {full_name}",
            code=f"OAUTH-{uuid4().hex[:8].upper()}",  # Código único generado
            email=email,
        )
        db.add(pharmacy)
        db.flush()  # Flush para obtener el ID sin hacer commit todavía

        user = User(
            id=uuid4(),
            email=email,
            username=username,
            full_name=full_name,
            # OAuth users don't have a password
            hashed_password=None,
            oauth_provider=provider,
            oauth_provider_id=provider_user_id,
            pharmacy_id=pharmacy.id,  # Asignar farmacia creada
            is_active=True,
            is_verified=True,  # OAuth users are pre-verified
            email_verified_at=utc_now(),
            last_login=utc_now(),
            role="user",  # Default role
        )

        db.add(user)
        db.commit()
        db.refresh(user)

        logger.info(f"Created new OAuth user: {email} via {provider} with pharmacy: {pharmacy.id}")
        return user

    def register_user(
        self,
        db: Session,
        email: str,
        username: str,
        password: str,
        full_name: Optional[str] = None,
        phone: Optional[str] = None,
        pharmacy_id: Optional[str] = None,
        # Pharmacy creation fields
        pharmacy_name: Optional[str] = None,
        pharmacy_code: Optional[str] = None,
        pharmacy_email: Optional[str] = None,
        pharmacy_address: Optional[str] = None,
        pharmacy_city: Optional[str] = None,
        pharmacy_postal_code: Optional[str] = None,
        pharmacy_phone: Optional[str] = None,
    ) -> User:
        """
        Register a new user with traditional authentication

        Args:
            db: Database session
            email: User email
            username: Username
            password: Plain text password
            full_name: User full name
            phone: Phone number
            pharmacy_id: Associated pharmacy ID (DEPRECATED - se crea automáticamente)
            pharmacy_name: Nombre de la farmacia (REQUERIDO si no hay pharmacy_id)
            pharmacy_code: Código de la farmacia
            pharmacy_email: Email de la farmacia
            pharmacy_address: Dirección de la farmacia
            pharmacy_city: Ciudad de la farmacia
            pharmacy_postal_code: Código postal de la farmacia
            pharmacy_phone: Teléfono de la farmacia

        Returns:
            Created User object
        """
        # Check if user already exists
        if db.query(User).filter(User.email == email).first():
            raise ValueError("Email already registered")

        if db.query(User).filter(User.username == username).first():
            raise ValueError("Username already taken")

        # Crear farmacia automáticamente si no hay pharmacy_id
        if pharmacy_id is None:
            # Si se proporcionan datos de farmacia, usarlos
            if pharmacy_name:
                logger.info(f"Creating pharmacy for new user: {pharmacy_name}")
                pharmacy = Pharmacy(
                    id=uuid4(),
                    name=pharmacy_name,
                    code=pharmacy_code or username.upper(),
                    email=pharmacy_email or email,
                    address=pharmacy_address,
                    city=pharmacy_city,
                    postal_code=pharmacy_postal_code,
                    phone=pharmacy_phone or phone,
                )
            else:
                # Si no hay datos, crear una farmacia básica
                # Esto asegura que SIEMPRE haya una farmacia (constraint NOT NULL)
                logger.info(f"Creating default pharmacy for new user: {full_name or username}")
                pharmacy = Pharmacy(
                    id=uuid4(),
                    name=f"Farmacia de {full_name or username}",
                    code=username.upper(),
                    email=email,
                    phone=phone,
                )

            db.add(pharmacy)
            db.flush()  # Flush para obtener el ID sin hacer commit todavía
            pharmacy_id = str(pharmacy.id)

            logger.info(f"Pharmacy created with ID: {pharmacy_id}")

        user = User(
            id=uuid4(),
            email=email,
            username=username,
            hashed_password=get_password_hash(password),
            full_name=full_name,
            phone=phone,
            pharmacy_id=pharmacy_id,
            is_active=True,
            is_verified=False,  # Require email verification
            role="user",  # Default role
        )

        db.add(user)
        db.commit()
        db.refresh(user)

        logger.info(f"Registered new user: {email} with pharmacy: {pharmacy_id}")
        return user

    def create_tokens(self, user: User) -> Dict[str, str]:
        """
        Create access and refresh tokens for user

        Args:
            user: User object

        Returns:
            Dictionary with access_token and refresh_token
        """
        # Token payload
        token_data = {
            "sub": str(user.id),
            "email": user.email,
            "role": user.role,
        }

        access_token = create_access_token(token_data)
        refresh_token = create_refresh_token(token_data)

        return {
            "access_token": access_token,
            "refresh_token": refresh_token,
            "token_type": "bearer",
        }

    def refresh_access_token(self, db: Session, refresh_token: str) -> Dict[str, str]:
        """
        Create new access token from refresh token

        Args:
            db: Database session
            refresh_token: JWT refresh token

        Returns:
            Dictionary with new access_token
        """
        # Verify refresh token
        payload = verify_token(refresh_token, "refresh")
        user_id = payload.get("sub")

        # Get user
        user = db.query(User).filter(User.id == user_id).first()
        if not user or not user.is_active:
            raise ValueError("Invalid user")

        # Create new access token
        token_data = {
            "sub": str(user.id),
            "email": user.email,
            "role": user.role,
        }

        access_token = create_access_token(token_data)

        return {
            "access_token": access_token,
            "token_type": "bearer",
        }

    def verify_email(self, db: Session, user_id: str, token: str) -> bool:
        """
        Verify user email with verification token

        Args:
            db: Database session
            user_id: User ID
            token: Email verification token

        Returns:
            True if verification successful
        """
        # For now, simple implementation
        # In production, use signed tokens with expiration
        user = db.query(User).filter(User.id == user_id).first()

        if not user:
            return False

        user.is_verified = True
        user.email_verified_at = utc_now()
        db.commit()

        logger.info(f"Email verified for user: {user.email}")
        return True

    def initiate_password_reset(self, db: Session, email: str) -> Optional[str]:
        """
        Initiate password reset process

        Args:
            db: Database session
            email: User email

        Returns:
            Reset token if user exists
        """
        user = db.query(User).filter(User.email == email).first()

        if not user:
            # Don't reveal if email exists
            return None

        # Generate reset token
        # In production, store this token with expiration
        reset_token = str(uuid4())

        logger.info(f"Password reset initiated for: {email}")
        return reset_token

    def reset_password(self, db: Session, reset_token: str, new_password: str) -> bool:
        """
        Reset user password with reset token

        Args:
            db: Database session
            reset_token: Password reset token
            new_password: New password

        Returns:
            True if reset successful
        """
        # In production, validate token and expiration
        # For now, simplified implementation

        # This would need proper token storage and validation
        # user = get_user_by_reset_token(reset_token)

        # Placeholder - in production, implement proper token validation
        logger.warning("Password reset functionality is disabled temporarily")
        return False

    def store_oauth_tokens(self, tokens: Dict[str, str], user: Optional[Any] = None) -> str:
        """
        Store OAuth tokens temporarily and return a state key

        Args:
            tokens: Dictionary with access_token and refresh_token
            user: Optional user object to include in response (Issue #398)

        Returns:
            State key to retrieve tokens
        """
        # Clean expired tokens first
        clean_expired_tokens()

        # Generate secure state key
        state_key = secrets.token_urlsafe(32)

        # Store tokens with expiration (Issue #398: Include user info)
        OAUTH_TOKEN_STORE[state_key] = {
            "tokens": tokens,
            "user": user,
            "expires_at": utc_now() + timedelta(minutes=5)
        }

        return state_key

    def retrieve_oauth_tokens(self, state_key: str) -> Optional[Dict[str, Any]]:
        """
        Retrieve OAuth tokens using state key

        Args:
            state_key: State key from OAuth callback

        Returns:
            Dictionary with tokens and user info if found and valid, None otherwise
            Format: {"tokens": {...}, "user": {...}} (Issue #398)
        """
        # Clean expired tokens first
        clean_expired_tokens()

        if state_key not in OAUTH_TOKEN_STORE:
            return None

        entry = OAUTH_TOKEN_STORE.pop(state_key)  # Remove after retrieval

        if entry["expires_at"] < utc_now():
            return None

        # Return both tokens and user info (Issue #398)
        return {
            "tokens": entry["tokens"],
            "user": entry.get("user")  # May be None for legacy entries
        }


# Singleton instance
auth_service = AuthService()
