# backend/app/services/action_tracking_service.py
"""
Service para gestión de acciones en el sistema ROI Tracker.

Issue #514: Feedback Loop de Acciones - ROI Tracker.
CRUD y transiciones de estado para ActionTracking.
"""

import logging
from calendar import monthrange
from datetime import datetime, timezone
from typing import Dict, List, Optional
from uuid import UUID

from sqlalchemy import and_, desc, func
from sqlalchemy.orm import Session

from app.models.action_tracking import ActionStatus, ActionTracking, ActionType
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


class ActionTrackingService:
    """Service para gestión de acciones sugeridas por el sistema."""

    def __init__(self, db: Session):
        self.db = db

    # =========================================================================
    # CRUD OPERATIONS
    # =========================================================================

    def create_action(
        self,
        pharmacy_id: UUID,
        action_type: str,
        action_description: str,
        expected_impact_eur: float,
        affected_products: Optional[List[Dict]] = None,
        expected_impact_description: Optional[str] = None,
        insight_hash: Optional[str] = None,
    ) -> ActionTracking:
        """
        Crear una nueva acción sugerida.

        Args:
            pharmacy_id: ID de la farmacia
            action_type: Tipo de acción (liquidation, restock, pricing, diversify)
            action_description: Descripción legible
            expected_impact_eur: Impacto económico esperado
            affected_products: Lista de productos afectados
            expected_impact_description: Explicación del cálculo
            insight_hash: Hash del insight origen (opcional)

        Returns:
            ActionTracking creada

        Raises:
            ValueError: Si action_type no es válido
        """
        # Validate action_type
        valid_types = [t.value for t in ActionType]
        if action_type not in valid_types:
            raise ValueError(
                f"Invalid action_type '{action_type}'. "
                f"Must be one of: {valid_types}"
            )

        action = ActionTracking(
            pharmacy_id=pharmacy_id,
            action_type=action_type,
            action_description=action_description,
            expected_impact_eur=expected_impact_eur,
            affected_products=affected_products or [],
            expected_impact_description=expected_impact_description,
            insight_hash=insight_hash,
            status=ActionStatus.PENDING.value,
        )

        self.db.add(action)
        self.db.commit()
        self.db.refresh(action)

        logger.info(
            f"Created action {action.id} for pharmacy {pharmacy_id}: "
            f"{action_type} - expected {expected_impact_eur}€"
        )

        return action

    def get_action(self, action_id: UUID) -> Optional[ActionTracking]:
        """Obtener una acción por ID."""
        return self.db.query(ActionTracking).filter(
            ActionTracking.id == action_id
        ).first()

    def get_action_for_pharmacy(
        self,
        action_id: UUID,
        pharmacy_id: UUID,
    ) -> Optional[ActionTracking]:
        """Obtener una acción verificando que pertenece a la farmacia."""
        return self.db.query(ActionTracking).filter(
            and_(
                ActionTracking.id == action_id,
                ActionTracking.pharmacy_id == pharmacy_id,
            )
        ).first()

    def get_actions(
        self,
        pharmacy_id: UUID,
        status: Optional[str] = None,
        action_type: Optional[str] = None,
        limit: int = 50,
        offset: int = 0,
    ) -> List[ActionTracking]:
        """
        Listar acciones de una farmacia con filtros opcionales.

        Args:
            pharmacy_id: ID de la farmacia
            status: Filtrar por estado (pending, executed, discarded, postponed)
            action_type: Filtrar por tipo de acción
            limit: Máximo de resultados
            offset: Offset para paginación

        Returns:
            Lista de acciones
        """
        query = self.db.query(ActionTracking).filter(
            ActionTracking.pharmacy_id == pharmacy_id
        )

        if status:
            query = query.filter(ActionTracking.status == status)

        if action_type:
            query = query.filter(ActionTracking.action_type == action_type)

        return query.order_by(
            desc(ActionTracking.created_at)
        ).offset(offset).limit(limit).all()

    def count_actions(
        self,
        pharmacy_id: UUID,
        status: Optional[str] = None,
    ) -> int:
        """Contar acciones de una farmacia."""
        query = self.db.query(func.count(ActionTracking.id)).filter(
            ActionTracking.pharmacy_id == pharmacy_id
        )

        if status:
            query = query.filter(ActionTracking.status == status)

        return query.scalar() or 0

    # =========================================================================
    # STATUS TRANSITIONS
    # =========================================================================

    def mark_executed(
        self,
        action_id: UUID,
        pharmacy_id: UUID,
        executed_at: Optional[datetime] = None,
        notes: Optional[str] = None,
    ) -> Optional[ActionTracking]:
        """
        Marcar una acción como ejecutada.

        Args:
            action_id: ID de la acción
            pharmacy_id: ID de la farmacia (verificación)
            executed_at: Fecha de ejecución (default: ahora)
            notes: Notas opcionales

        Returns:
            ActionTracking actualizada o None si no existe
        """
        action = self.get_action_for_pharmacy(action_id, pharmacy_id)
        if not action:
            return None

        if action.status == ActionStatus.EXECUTED.value:
            logger.warning(f"Action {action_id} already executed")
            return action

        action.status = ActionStatus.EXECUTED.value
        action.executed_at = executed_at or utc_now()
        action.updated_at = utc_now()

        # Clear postpone if was postponed
        action.postponed_until = None

        self.db.commit()
        self.db.refresh(action)

        logger.info(
            f"Action {action_id} marked as executed at {action.executed_at}"
        )

        return action

    def mark_discarded(
        self,
        action_id: UUID,
        pharmacy_id: UUID,
        reason: Optional[str] = None,
    ) -> Optional[ActionTracking]:
        """
        Marcar una acción como descartada.

        Args:
            action_id: ID de la acción
            pharmacy_id: ID de la farmacia (verificación)
            reason: Razón del descarte

        Returns:
            ActionTracking actualizada o None si no existe
        """
        action = self.get_action_for_pharmacy(action_id, pharmacy_id)
        if not action:
            return None

        if action.status == ActionStatus.DISCARDED.value:
            logger.warning(f"Action {action_id} already discarded")
            return action

        action.status = ActionStatus.DISCARDED.value
        action.discarded_at = utc_now()
        action.discard_reason = reason
        action.updated_at = utc_now()

        # Clear postpone if was postponed
        action.postponed_until = None

        self.db.commit()
        self.db.refresh(action)

        logger.info(f"Action {action_id} marked as discarded: {reason}")

        return action

    def mark_postponed(
        self,
        action_id: UUID,
        pharmacy_id: UUID,
        postpone_until: datetime,
        notes: Optional[str] = None,
    ) -> Optional[ActionTracking]:
        """
        Posponer una acción.

        Args:
            action_id: ID de la acción
            pharmacy_id: ID de la farmacia (verificación)
            postpone_until: Fecha hasta la cual posponer
            notes: Notas opcionales

        Returns:
            ActionTracking actualizada o None si no existe
        """
        action = self.get_action_for_pharmacy(action_id, pharmacy_id)
        if not action:
            return None

        action.status = ActionStatus.POSTPONED.value
        action.postponed_until = postpone_until
        action.updated_at = utc_now()

        self.db.commit()
        self.db.refresh(action)

        logger.info(
            f"Action {action_id} postponed until {postpone_until}"
        )

        return action

    # =========================================================================
    # QUERY HELPERS
    # =========================================================================

    def get_pending_actions(
        self,
        pharmacy_id: UUID,
        limit: int = 50,
    ) -> List[ActionTracking]:
        """Obtener acciones pendientes de una farmacia."""
        return self.get_actions(
            pharmacy_id=pharmacy_id,
            status=ActionStatus.PENDING.value,
            limit=limit,
        )

    def get_executed_actions_for_month(
        self,
        pharmacy_id: UUID,
        year: int,
        month: int,
    ) -> List[ActionTracking]:
        """
        Obtener acciones ejecutadas en un mes específico.

        Args:
            pharmacy_id: ID de la farmacia
            year: Año
            month: Mes (1-12)

        Returns:
            Lista de acciones ejecutadas en el mes
        """
        start_date = datetime(year, month, 1, tzinfo=timezone.utc)
        _, last_day = monthrange(year, month)
        end_date = datetime(year, month, last_day, 23, 59, 59, tzinfo=timezone.utc)

        return self.db.query(ActionTracking).filter(
            and_(
                ActionTracking.pharmacy_id == pharmacy_id,
                ActionTracking.status == ActionStatus.EXECUTED.value,
                ActionTracking.executed_at >= start_date,
                ActionTracking.executed_at <= end_date,
            )
        ).order_by(ActionTracking.executed_at).all()

    def get_expired_postponed_actions(
        self,
        pharmacy_id: Optional[UUID] = None,
    ) -> List[ActionTracking]:
        """
        Obtener acciones pospuestas que ya expiraron.
        Útil para reactivarlas automáticamente.

        Args:
            pharmacy_id: Filtrar por farmacia (opcional)

        Returns:
            Lista de acciones con postpone expirado
        """
        now = utc_now()
        query = self.db.query(ActionTracking).filter(
            and_(
                ActionTracking.status == ActionStatus.POSTPONED.value,
                ActionTracking.postponed_until <= now,
            )
        )

        if pharmacy_id:
            query = query.filter(ActionTracking.pharmacy_id == pharmacy_id)

        return query.all()

    def reactivate_expired_postponed(
        self,
        pharmacy_id: Optional[UUID] = None,
    ) -> int:
        """
        Reactivar acciones pospuestas expiradas a estado pending.

        Args:
            pharmacy_id: Filtrar por farmacia (opcional)

        Returns:
            Número de acciones reactivadas
        """
        actions = self.get_expired_postponed_actions(pharmacy_id)
        count = 0

        for action in actions:
            action.status = ActionStatus.PENDING.value
            action.postponed_until = None
            action.updated_at = utc_now()
            count += 1

        if count > 0:
            self.db.commit()
            logger.info(f"Reactivated {count} expired postponed actions")

        return count

    def count_by_status(
        self,
        pharmacy_id: UUID,
        year: Optional[int] = None,
        month: Optional[int] = None,
    ) -> Dict[str, int]:
        """
        Contar acciones por estado para una farmacia.

        Args:
            pharmacy_id: ID de la farmacia
            year: Filtrar por año (opcional)
            month: Filtrar por mes (opcional)

        Returns:
            Dict con conteos por estado
        """
        query = self.db.query(
            ActionTracking.status,
            func.count(ActionTracking.id).label("count"),
        ).filter(
            ActionTracking.pharmacy_id == pharmacy_id
        )

        if year and month:
            start_date = datetime(year, month, 1, tzinfo=timezone.utc)
            _, last_day = monthrange(year, month)
            end_date = datetime(year, month, last_day, 23, 59, 59, tzinfo=timezone.utc)
            query = query.filter(
                and_(
                    ActionTracking.created_at >= start_date,
                    ActionTracking.created_at <= end_date,
                )
            )

        results = query.group_by(ActionTracking.status).all()

        counts = {
            ActionStatus.PENDING.value: 0,
            ActionStatus.EXECUTED.value: 0,
            ActionStatus.DISCARDED.value: 0,
            ActionStatus.POSTPONED.value: 0,
        }

        for status, count in results:
            counts[status] = count

        return counts

    def get_total_expected_impact(
        self,
        pharmacy_id: UUID,
        status: Optional[str] = None,
    ) -> float:
        """
        Calcular suma del impacto esperado de acciones.

        Args:
            pharmacy_id: ID de la farmacia
            status: Filtrar por estado (opcional)

        Returns:
            Suma de expected_impact_eur
        """
        query = self.db.query(
            func.sum(ActionTracking.expected_impact_eur)
        ).filter(
            ActionTracking.pharmacy_id == pharmacy_id
        )

        if status:
            query = query.filter(ActionTracking.status == status)

        return query.scalar() or 0.0
