# backend/app/services/action_impact_calculator.py
"""
Calculadora de impacto real para acciones ejecutadas.

Issue #514: Feedback Loop de Acciones - ROI Tracker.
Calcula el impacto económico real de cada tipo de acción tras su ejecución.
"""

import logging
from datetime import datetime, timedelta, timezone
from typing import Dict, Optional, Tuple
from uuid import UUID

from sqlalchemy import and_, func
from sqlalchemy.orm import Session

from app.models.action_tracking import ActionTracking, ActionType
from app.models.sales_data import SalesData
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


class ActionImpactCalculator:
    """
    Calcula el impacto real de acciones ejecutadas.

    Cada tipo de acción tiene una lógica de cálculo específica:
    - LIQUIDATION: Ventas de productos afectados tras ejecución
    - RESTOCK: Ventas no perdidas por evitar rotura de stock
    - PRICING: Delta de margen * unidades vendidas
    - DIVERSIFY: Ventas de nuevos productos añadidos
    """

    # ==== Configuration constants ====
    # Days to measure impact after action execution
    DEFAULT_MEASUREMENT_DAYS = 30
    # Default stockout days for restock impact calculation (conservative estimate)
    DEFAULT_STOCKOUT_DAYS = 7
    # Default margin improvement % for pricing actions (conservative estimate: 5%)
    DEFAULT_MARGIN_IMPROVEMENT_PCT = 0.05
    # Max products per query to avoid query plan issues with large IN clauses
    MAX_PRODUCTS_PER_QUERY = 1000

    def __init__(self, db: Session):
        self.db = db

    def calculate_impact(
        self,
        action: ActionTracking,
        measurement_days: Optional[int] = None,
    ) -> Tuple[float, str]:
        """
        Calcular el impacto real de una acción ejecutada.

        Args:
            action: Acción ejecutada
            measurement_days: Días para medir impacto (default: 30)

        Returns:
            Tuple (impacto_eur, descripción)
        """
        if not action.executed_at:
            logger.warning(f"Action {action.id} has no executed_at date")
            return 0.0, "Acción no ejecutada"

        days = measurement_days or self.DEFAULT_MEASUREMENT_DAYS

        calculators = {
            ActionType.LIQUIDATION.value: self._calculate_liquidation_impact,
            ActionType.RESTOCK.value: self._calculate_restock_impact,
            ActionType.PRICING.value: self._calculate_pricing_impact,
            ActionType.DIVERSIFY.value: self._calculate_diversification_impact,
        }

        calculator = calculators.get(action.action_type)
        if not calculator:
            logger.error(f"Unknown action type: {action.action_type}")
            return 0.0, f"Tipo de acción desconocido: {action.action_type}"

        try:
            impact, description = calculator(action, days)
            logger.info(
                f"Calculated impact for action {action.id}: "
                f"{impact:.2f}€ - {description}"
            )
            return impact, description
        except Exception as e:
            logger.error(f"Error calculating impact for action {action.id}: {e}")
            return 0.0, f"Error en cálculo: {str(e)}"

    def calculate_and_save(
        self,
        action: ActionTracking,
        measurement_days: Optional[int] = None,
    ) -> ActionTracking:
        """
        Calcular y guardar el impacto real en la acción.

        Args:
            action: Acción ejecutada
            measurement_days: Días para medir impacto

        Returns:
            ActionTracking actualizada
        """
        impact, description = self.calculate_impact(action, measurement_days)

        action.actual_impact_eur = impact
        action.actual_impact_description = description
        action.impact_calculated_at = utc_now()

        self.db.commit()
        self.db.refresh(action)

        return action

    # =========================================================================
    # CALCULATORS BY ACTION TYPE
    # =========================================================================

    def _calculate_liquidation_impact(
        self,
        action: ActionTracking,
        days: int,
    ) -> Tuple[float, str]:
        """
        Calcular impacto de liquidación.

        Lógica: Suma de ventas de productos afectados después de ejecución.
        El impacto es lo que se recuperó vendiendo stock que iba a caducar.

        Returns:
            Tuple (impacto_eur, descripción)
        """
        if not action.affected_products:
            return 0.0, "Sin productos afectados"

        # Extraer códigos nacionales
        product_codes = self._extract_product_codes(action.affected_products)
        if not product_codes:
            return 0.0, "Sin códigos de producto válidos"

        # Calcular periodo de medición
        start_date = action.executed_at
        end_date = start_date + timedelta(days=days)

        # Sumar ventas en el periodo
        total_sales = self._sum_sales_for_products(
            pharmacy_id=action.pharmacy_id,
            product_codes=product_codes,
            start_date=start_date,
            end_date=end_date,
        )

        products_count = len(product_codes)
        description = (
            f"Ventas recuperadas de {products_count} productos en {days} días: "
            f"{total_sales:.2f}€"
        )

        return total_sales, description

    def _calculate_restock_impact(
        self,
        action: ActionTracking,
        days: int,
    ) -> Tuple[float, str]:
        """
        Calcular impacto de reposición urgente.

        Lógica: Estimación de ventas NO perdidas por evitar rotura de stock.
        Se calcula como: (ventas diarias promedio) * (días que habría estado sin stock)

        Returns:
            Tuple (impacto_eur, descripción)
        """
        if not action.affected_products:
            return 0.0, "Sin productos afectados"

        product_codes = self._extract_product_codes(action.affected_products)
        if not product_codes:
            return 0.0, "Sin códigos de producto válidos"

        # Calcular ventas promedio diarias (30 días antes de ejecución)
        avg_period_start = action.executed_at - timedelta(days=30)
        avg_period_end = action.executed_at

        avg_daily = self._calculate_avg_daily_sales(
            pharmacy_id=action.pharmacy_id,
            product_codes=product_codes,
            start_date=avg_period_start,
            end_date=avg_period_end,
        )

        if avg_daily <= 0:
            return 0.0, "Sin historial de ventas para estimar"

        # Estimar días que habría estado sin stock
        # (usando información del affected_products si está disponible)
        estimated_stockout_days = self._estimate_stockout_days(action)

        impact = avg_daily * estimated_stockout_days

        description = (
            f"Ventas no perdidas: {avg_daily:.2f}€/día x {estimated_stockout_days} días "
            f"= {impact:.2f}€"
        )

        return impact, description

    def _calculate_pricing_impact(
        self,
        action: ActionTracking,
        days: int,
    ) -> Tuple[float, str]:
        """
        Calcular impacto de ajuste de precios.

        Lógica: Delta de margen * unidades vendidas después del cambio.

        Returns:
            Tuple (impacto_eur, descripción)
        """
        if not action.affected_products:
            return 0.0, "Sin productos afectados"

        product_codes = self._extract_product_codes(action.affected_products)
        if not product_codes:
            return 0.0, "Sin códigos de producto válidos"

        # Calcular ventas después de la ejecución
        start_date = action.executed_at
        end_date = start_date + timedelta(days=days)

        total_sales = self._sum_sales_for_products(
            pharmacy_id=action.pharmacy_id,
            product_codes=product_codes,
            start_date=start_date,
            end_date=end_date,
        )

        # Estimar mejora de margen (si está en affected_products)
        # Por defecto asumimos 5% de mejora de margen
        margin_improvement_pct = self._extract_margin_improvement(action) or self.DEFAULT_MARGIN_IMPROVEMENT_PCT
        impact = total_sales * margin_improvement_pct

        description = (
            f"Mejora de margen ({margin_improvement_pct*100:.1f}%) sobre "
            f"{total_sales:.2f}€ ventas = {impact:.2f}€"
        )

        return impact, description

    def _calculate_diversification_impact(
        self,
        action: ActionTracking,
        days: int,
    ) -> Tuple[float, str]:
        """
        Calcular impacto de diversificación (nuevos productos).

        Lógica: Ventas de los nuevos productos añadidos.

        Returns:
            Tuple (impacto_eur, descripción)
        """
        if not action.affected_products:
            return 0.0, "Sin productos afectados"

        product_codes = self._extract_product_codes(action.affected_products)
        if not product_codes:
            return 0.0, "Sin códigos de producto válidos"

        # Calcular ventas de nuevos productos
        start_date = action.executed_at
        end_date = start_date + timedelta(days=days)

        total_sales = self._sum_sales_for_products(
            pharmacy_id=action.pharmacy_id,
            product_codes=product_codes,
            start_date=start_date,
            end_date=end_date,
        )

        products_count = len(product_codes)
        description = (
            f"Ventas de {products_count} nuevos productos en {days} días: "
            f"{total_sales:.2f}€"
        )

        return total_sales, description

    # =========================================================================
    # HELPER METHODS
    # =========================================================================

    def _extract_product_codes(
        self,
        affected_products: list,
    ) -> list[str]:
        """Extraer códigos nacionales de lista de productos afectados."""
        codes = []
        for product in affected_products:
            if isinstance(product, dict):
                code = product.get("codigo_nacional") or product.get("product_code")
                if code:
                    codes.append(str(code))
        return codes

    def _sum_sales_for_products(
        self,
        pharmacy_id: UUID,
        product_codes: list[str],
        start_date: datetime,
        end_date: datetime,
    ) -> float:
        """
        Sumar ventas de productos en un periodo.

        Implements chunking for large product lists to avoid query plan issues
        with large IN clauses (PostgreSQL performs poorly with >1000 items).
        """
        if not product_codes:
            return 0.0

        # Chunk large product lists to avoid query plan issues
        if len(product_codes) > self.MAX_PRODUCTS_PER_QUERY:
            total = 0.0
            for i in range(0, len(product_codes), self.MAX_PRODUCTS_PER_QUERY):
                chunk = product_codes[i : i + self.MAX_PRODUCTS_PER_QUERY]
                total += self._sum_sales_for_products(
                    pharmacy_id, chunk, start_date, end_date
                )
            return total

        result = self.db.query(
            func.sum(SalesData.total_amount)
        ).filter(
            and_(
                SalesData.pharmacy_id == pharmacy_id,
                SalesData.codigo_nacional.in_(product_codes),
                SalesData.sale_date >= start_date,
                SalesData.sale_date <= end_date,
            )
        ).scalar()

        return float(result or 0.0)

    def _calculate_avg_daily_sales(
        self,
        pharmacy_id: UUID,
        product_codes: list[str],
        start_date: datetime,
        end_date: datetime,
    ) -> float:
        """Calcular promedio de ventas diarias."""
        total = self._sum_sales_for_products(
            pharmacy_id, product_codes, start_date, end_date
        )
        days = (end_date - start_date).days
        if days <= 0:
            return 0.0
        return total / days

    def _estimate_stockout_days(self, action: ActionTracking) -> int:
        """
        Estimar días que habría estado sin stock.

        Usa información de affected_products si está disponible,
        sino usa un valor por defecto conservador.
        """
        # Intentar extraer de affected_products
        for product in action.affected_products or []:
            if isinstance(product, dict):
                stockout_days = product.get("estimated_stockout_days")
                if stockout_days is not None:  # Explicit None check (0 is valid)
                    try:
                        return int(stockout_days)
                    except (ValueError, TypeError):
                        continue

        # Default conservador
        return self.DEFAULT_STOCKOUT_DAYS

    def _extract_margin_improvement(self, action: ActionTracking) -> Optional[float]:
        """
        Extraer porcentaje de mejora de margen de affected_products.

        Returns:
            Float entre 0 y 1 representando el % de mejora
        """
        for product in action.affected_products or []:
            if isinstance(product, dict):
                improvement = product.get("margin_improvement_pct")
                if improvement:
                    return float(improvement) / 100 if improvement > 1 else float(improvement)

        return None

    # =========================================================================
    # BATCH OPERATIONS
    # =========================================================================

    def calculate_pending_impacts(
        self,
        pharmacy_id: Optional[UUID] = None,
        min_days_since_execution: int = 7,
        batch_size: int = 100,
    ) -> int:
        """
        Calcular impactos pendientes para acciones ejecutadas.

        Procesa acciones que fueron ejecutadas hace al menos X días
        y aún no tienen impacto calculado.

        Args:
            pharmacy_id: Filtrar por farmacia (opcional)
            min_days_since_execution: Días mínimos desde ejecución
            batch_size: Tamaño del batch

        Returns:
            Número de acciones procesadas
        """
        from app.models.action_tracking import ActionStatus

        cutoff_date = utc_now() - timedelta(days=min_days_since_execution)

        query = self.db.query(ActionTracking).filter(
            and_(
                ActionTracking.status == ActionStatus.EXECUTED.value,
                ActionTracking.executed_at <= cutoff_date,
                ActionTracking.actual_impact_eur.is_(None),
            )
        )

        if pharmacy_id:
            query = query.filter(ActionTracking.pharmacy_id == pharmacy_id)

        actions = query.limit(batch_size).all()
        processed = 0

        # Process all actions before committing (batch transaction)
        for action in actions:
            try:
                impact, description = self.calculate_impact(action)
                action.actual_impact_eur = impact
                action.actual_impact_description = description
                action.impact_calculated_at = utc_now()
                processed += 1
            except Exception as e:
                logger.error(f"Error processing action {action.id}: {e}")
                # Continue processing others, don't fail entire batch

        # Single commit for all successful calculations
        if processed > 0:
            try:
                self.db.commit()
            except Exception as e:
                self.db.rollback()
                logger.error(f"Failed to commit batch: {e}")
                return 0

        logger.info(f"Calculated impact for {processed} actions")
        return processed
