#!/usr/bin/env python3
"""
Script de inicialización de usuario admin
Ejecutar desde el directorio backend: python -m app.scripts.init_admin

Requiere variables de entorno:
- ADMIN_EMAIL: Email del usuario administrador
- ADMIN_PASSWORD: Contraseña segura (mínimo 8 caracteres)
- ADMIN_FULL_NAME: Nombre completo (opcional, default: "Admin User")
"""
import asyncio
import os
import sys

from sqlalchemy.ext.asyncio import AsyncSession, create_async_engine
from sqlalchemy.orm import sessionmaker

from app.core.config import settings
from app.core.security import get_password_hash
from app.models.auth import User


def validate_email(email: str) -> bool:
    """Validación básica de formato de email"""
    import re

    pattern = r"^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$"
    return re.match(pattern, email) is not None


def validate_password(password: str) -> tuple[bool, str]:
    """Validar complejidad de password"""
    if len(password) < 8:
        return False, "Password debe tener al menos 8 caracteres"
    if not any(c.isupper() for c in password):
        return False, "Password debe contener al menos una mayúscula"
    if not any(c.islower() for c in password):
        return False, "Password debe contener al menos una minúscula"
    if not any(c.isdigit() for c in password):
        return False, "Password debe contener al menos un número"
    return True, "Password válido"


async def create_admin():
    """Crear usuario admin usando variables de entorno"""

    # Obtener credenciales de variables de entorno
    ADMIN_EMAIL = os.getenv("ADMIN_EMAIL")
    ADMIN_PASSWORD = os.getenv("ADMIN_PASSWORD")
    ADMIN_NAME = os.getenv("ADMIN_FULL_NAME", "Admin User")

    # Validación de variables requeridas
    if not ADMIN_EMAIL or not ADMIN_PASSWORD:
        print("❌ Error: ADMIN_EMAIL y ADMIN_PASSWORD son requeridos")
        print()
        print("   Configura las variables de entorno antes de ejecutar:")
        print("   export ADMIN_EMAIL='admin@example.com'")
        print("   export ADMIN_PASSWORD='YourSecurePassword123!'")
        print("   export ADMIN_FULL_NAME='Admin User'  # Opcional")
        print()
        sys.exit(1)

    # Validar email
    if not validate_email(ADMIN_EMAIL):
        print(f"❌ Email inválido: {ADMIN_EMAIL}")
        print("   Usa un formato válido: nombre@dominio.com")
        sys.exit(1)

    # Validar password
    is_valid, message = validate_password(ADMIN_PASSWORD)
    if not is_valid:
        print(f"❌ Password inválido: {message}")
        print()
        print("   Requisitos del password:")
        print("   - Mínimo 8 caracteres")
        print("   - Al menos una mayúscula")
        print("   - Al menos una minúscula")
        print("   - Al menos un número")
        print()
        sys.exit(1)

    # Crear engine async
    engine = create_async_engine(str(settings.SQLALCHEMY_DATABASE_URI))
    async_session = sessionmaker(engine, class_=AsyncSession, expire_on_commit=False)

    async with async_session() as session:
        try:
            # Verificar si existe
            from sqlalchemy import select

            result = await session.execute(select(User).where(User.email == ADMIN_EMAIL))
            existing_user = result.scalar_one_or_none()

            if existing_user:
                print("✅ Usuario ya existe, actualizando contraseña...")
                existing_user.hashed_password = get_password_hash(ADMIN_PASSWORD)
                existing_user.full_name = ADMIN_NAME
                await session.commit()
                print("✅ Contraseña actualizada")
            else:
                print("📝 Creando usuario admin...")
                admin = User(
                    email=ADMIN_EMAIL,
                    full_name=ADMIN_NAME,
                    hashed_password=get_password_hash(ADMIN_PASSWORD),
                    role="admin",
                    is_active=True,
                )
                session.add(admin)
                await session.commit()
                print("✅ Usuario admin creado")

            # Verificar
            result = await session.execute(select(User).where(User.email == ADMIN_EMAIL))
            user = result.scalar_one()
            print("\n📊 Usuario verificado:")
            print(f"   Email: {user.email}")
            print(f"   Nombre: {user.full_name}")
            print(f"   Rol: {user.role}")
            print(f"   Activo: {user.is_active}")
            print()

        except Exception as e:
            print(f"❌ Error: {e}")
            await session.rollback()
            raise
        finally:
            await engine.dispose()


if __name__ == "__main__":
    asyncio.run(create_admin())
