#!/usr/bin/env python3
"""
Script de limpieza nocturna para usuarios FREE (Issue #142)

Este script elimina todos los datos de ventas de usuarios con plan FREE:
- SalesData (con CASCADE a SalesEnrichment automático via FK)
- FileUpload asociados
- PharmacyPartner (laboratorios seleccionados)

EJECUCIÓN:
    python backend/scripts/cleanup_free_tier_data.py

SCHEDULER:
    Configurado en render.yaml para ejecutarse diariamente a las 9 PM UTC:
    - schedule: "0 21 * * *"  # 9 PM UTC (10 PM Madrid, 11 PM Madrid verano)

SEGURIDAD:
    - Solo afecta a usuarios con subscription_plan='free'
    - Respeta soft deletes (deleted_at IS NULL)
    - Solo usuarios activos (is_active=True)
    - Genera audit logs de cada limpieza
"""

import logging
import os
import sys
from pathlib import Path

# Agregar backend al path para imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from sqlalchemy.orm import Session

from app.database import SessionLocal
from app.models.file_upload import FileUpload
from app.models.pharmacy_partners import PharmacyPartner
from app.models.sales_data import SalesData
from app.models.user import User
from app.services.audit_service import AuditService
from app.utils.datetime_utils import utc_now

# Configurar logging
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(name)s: %(message)s",
    datefmt="%Y-%m-%d %H:%M:%S"
)
logger = logging.getLogger(__name__)


def cleanup_free_tier_sales_data(db: Session, dry_run: bool = False) -> dict:
    """
    Elimina datos de ventas de usuarios FREE con CASCADE completo.

    Args:
        db: Sesión de base de datos
        dry_run: Si True, solo reporta sin eliminar (default: False)

    Returns:
        dict: Estadísticas de limpieza {
            "users_processed": int,
            "total_sales_deleted": int,
            "total_uploads_deleted": int,
            "users_details": List[dict]
        }

    Note:
        REGLA #18 (Issue #142): Limpieza automática nocturna de datos FREE
        - Cascade: SalesData → SalesEnrichment (automático via FK CASCADE)
        - FileUploads: Eliminación manual explícita
    """
    logger.info(f"[CLEANUP_FREE_TIER] Iniciando limpieza {'(DRY RUN)' if dry_run else ''}")

    # Instanciar audit service
    audit_service = AuditService(db)

    stats = {
        "users_processed": 0,
        "total_sales_deleted": 0,
        "total_uploads_deleted": 0,
        "total_partners_deleted": 0,
        "users_details": [],
        "dry_run": dry_run,
        "timestamp": utc_now().isoformat()
    }

    try:
        # Obtener usuarios FREE activos
        free_users = db.query(User).filter(
            User.subscription_plan == "free",
            User.deleted_at.is_(None),  # No soft-deleted
            User.is_active == True  # Solo activos
        ).all()

        logger.info(f"[CLEANUP_FREE_TIER] Encontrados {len(free_users)} usuarios FREE activos")

        for user in free_users:
            if not user.pharmacy_id:
                logger.warning(
                    f"[CLEANUP_FREE_TIER] Usuario {user.email} (ID: {user.id}) sin pharmacy_id asignada - SKIP"
                )
                continue

            user_stats = {
                "user_id": str(user.id),
                "user_email": user.email,
                "pharmacy_id": str(user.pharmacy_id),
                "sales_deleted": 0,
                "uploads_deleted": 0,
                "partners_deleted": 0
            }

            # Contar datos antes de eliminar
            sales_count = db.query(SalesData).filter(
                SalesData.pharmacy_id == user.pharmacy_id
            ).count()

            uploads_count = db.query(FileUpload).filter(
                FileUpload.pharmacy_id == user.pharmacy_id
            ).count()

            partners_count = db.query(PharmacyPartner).filter(
                PharmacyPartner.pharmacy_id == user.pharmacy_id
            ).count()

            user_stats["sales_deleted"] = sales_count
            user_stats["uploads_deleted"] = uploads_count
            user_stats["partners_deleted"] = partners_count

            logger.info(
                f"[CLEANUP_FREE_TIER] Usuario {user.email}: "
                f"{sales_count} ventas, {uploads_count} uploads, {partners_count} partners"
            )

            if not dry_run:
                try:
                    # Eliminar en batches para evitar locks largos en BD
                    # Render tiene 512MB RAM límite, usar batch más pequeño
                    BATCH_SIZE = 100 if os.getenv("RENDER") else 1000
                    deleted_sales = 0

                    # Delete SalesData en batches (CASCADE automático a SalesEnrichment via FK)
                    while True:
                        batch = db.query(SalesData).filter(
                            SalesData.pharmacy_id == user.pharmacy_id
                        ).limit(BATCH_SIZE).all()

                        if not batch:
                            break

                        for sale in batch:
                            db.delete(sale)

                        deleted_sales += len(batch)
                        db.commit()  # Commit por batch

                        logger.debug(
                            f"[CLEANUP_FREE_TIER] Batch processed: {len(batch)} sales deleted "
                            f"(total: {deleted_sales})"
                        )

                    # Eliminar FileUploads asociados (menos crítico, sin batching)
                    deleted_uploads = db.query(FileUpload).filter(
                        FileUpload.pharmacy_id == user.pharmacy_id
                    ).delete(synchronize_session=False)

                    # Eliminar PharmacyPartner (laboratorios seleccionados)
                    deleted_partners = db.query(PharmacyPartner).filter(
                        PharmacyPartner.pharmacy_id == user.pharmacy_id
                    ).delete(synchronize_session=False)

                    # Commit final
                    db.commit()

                    logger.info(
                        f"[CLEANUP_FREE_TIER] ✓ Usuario {user.email}: "
                        f"Eliminados {deleted_sales} ventas, {deleted_uploads} uploads, {deleted_partners} partners"
                    )

                    # Generar audit log
                    try:
                        from app.models.audit_log import AuditAction
                        audit_service.log_action(
                            action=AuditAction.ADMIN_ACTION,
                            method="CRON",
                            endpoint="/scripts/cleanup_free_tier_data",
                            user=user,
                            details={
                                "sales_deleted": deleted_sales,
                                "uploads_deleted": deleted_uploads,
                                "partners_deleted": deleted_partners,
                                "pharmacy_id": str(user.pharmacy_id),
                                "cleanup_timestamp": utc_now().isoformat()
                            }
                        )
                    except Exception as audit_error:
                        logger.warning(f"No se pudo crear audit log: {audit_error}")

                    # Actualizar estadísticas verificadas
                    user_stats["sales_deleted"] = deleted_sales
                    user_stats["uploads_deleted"] = deleted_uploads
                    user_stats["partners_deleted"] = deleted_partners
                    stats["total_sales_deleted"] += deleted_sales
                    stats["total_uploads_deleted"] += deleted_uploads
                    stats["total_partners_deleted"] += deleted_partners

                except Exception as e:
                    db.rollback()
                    logger.error(
                        f"[CLEANUP_FREE_TIER] Error eliminando datos de {user.email}: {e}",
                        exc_info=True
                    )
                    # Continuar con siguiente usuario en vez de fallar todo el job
                    continue

            else:
                logger.info(
                    f"[CLEANUP_FREE_TIER] [DRY RUN] Usuario {user.email}: "
                    f"Se eliminarían {sales_count} ventas, {uploads_count} uploads, {partners_count} partners"
                )
                # En dry run, usar los counts estimados
                stats["total_sales_deleted"] += sales_count
                stats["total_uploads_deleted"] += uploads_count
                stats["total_partners_deleted"] += partners_count

            stats["users_details"].append(user_stats)
            stats["users_processed"] += 1

        if not dry_run:
            db.commit()

        logger.info(
            f"[CLEANUP_FREE_TIER] ✓ Limpieza completada: "
            f"{stats['users_processed']} usuarios procesados, "
            f"{stats['total_sales_deleted']} ventas eliminadas, "
            f"{stats['total_uploads_deleted']} uploads eliminados, "
            f"{stats['total_partners_deleted']} partners eliminados"
        )

        return stats

    except Exception as e:
        logger.error(f"[CLEANUP_FREE_TIER] ✗ Error durante limpieza: {str(e)}", exc_info=True)
        db.rollback()
        raise


def main():
    """
    Punto de entrada principal del script de limpieza.

    Soporta modo DRY RUN para testing:
        python backend/scripts/cleanup_free_tier_data.py --dry-run
    """
    import argparse

    parser = argparse.ArgumentParser(
        description="Limpieza nocturna de datos de usuarios FREE (Issue #142)"
    )
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Modo prueba: reporta sin eliminar datos"
    )
    args = parser.parse_args()

    logger.info("=" * 80)
    logger.info("Script de limpieza FREE tier - Issue #142")
    logger.info(f"Entorno: {os.getenv('ENVIRONMENT', 'development')}")
    logger.info(f"Modo: {'DRY RUN (sin eliminar)' if args.dry_run else 'PRODUCCIÓN (elimina datos)'}")
    logger.info("=" * 80)

    db = SessionLocal()
    try:
        stats = cleanup_free_tier_sales_data(db, dry_run=args.dry_run)

        logger.info("\n" + "=" * 80)
        logger.info("RESUMEN DE LIMPIEZA:")
        logger.info(f"  Usuarios procesados: {stats['users_processed']}")
        logger.info(f"  Ventas eliminadas: {stats['total_sales_deleted']}")
        logger.info(f"  Uploads eliminados: {stats['total_uploads_deleted']}")
        logger.info(f"  Partners eliminados: {stats['total_partners_deleted']}")
        logger.info("=" * 80 + "\n")

        # Exit code 0 = éxito
        sys.exit(0)

    except Exception as e:
        logger.error(f"✗ Error fatal en script de limpieza: {str(e)}", exc_info=True)
        # Exit code 1 = error
        sys.exit(1)

    finally:
        db.close()
        logger.info("Sesión de base de datos cerrada")


if __name__ == "__main__":
    main()
