# Scripts de Administración

Colección de scripts de utilidad para gestión del backend de xFarma.

## 📋 Scripts Disponibles

### init_admin.py

Script para crear o actualizar usuario administrador.

#### Requisitos

- Variables de entorno configuradas
- Base de datos inicializada y accesible
- Container backend corriendo

#### Variables de Entorno Requeridas

```bash
# Email del administrador (formato válido)
ADMIN_EMAIL="admin@example.com"

# Password seguro (requisitos mínimos)
ADMIN_PASSWORD="SecurePass123!"

# Nombre completo (opcional)
ADMIN_FULL_NAME="Admin User"
```

#### Requisitos de Password

- Mínimo 8 caracteres
- Al menos una letra mayúscula
- Al menos una letra minúscula
- Al menos un número

#### Uso - Desarrollo Local

```bash
# 1. Configurar variables de entorno
export ADMIN_EMAIL="admin@example.com"
export ADMIN_PASSWORD="SecurePass123!"
export ADMIN_FULL_NAME="Admin User"

# 2. Ejecutar desde container
docker-compose exec backend python -m app.scripts.init_admin
```

#### Uso - PowerShell (Windows)

```powershell
# 1. Configurar variables de entorno
$env:ADMIN_EMAIL = "admin@example.com"
$env:ADMIN_PASSWORD = "SecurePass123!"
$env:ADMIN_FULL_NAME = "Admin User"

# 2. Ejecutar desde container
docker-compose exec backend python -m app.scripts.init_admin
```

#### Uso - Producción (Render)

```bash
# 1. Configurar variables en Render Dashboard
# - Ve a: Settings > Environment > Add Environment Variable
# - Agregar: ADMIN_EMAIL
# - Agregar: ADMIN_PASSWORD
# - Agregar: ADMIN_FULL_NAME (opcional)

# 2. Ejecutar via Render Shell
python -m app.scripts.init_admin
```

#### Comportamiento

- **Si el usuario existe**: Actualiza la contraseña y nombre
- **Si no existe**: Crea nuevo usuario con rol "admin"
- **Siempre verifica**: Muestra información del usuario al finalizar

#### Mensajes de Salida

**Éxito**:
```
✅ Usuario ya existe, actualizando contraseña...
✅ Contraseña actualizada

📊 Usuario verificado:
   Email: admin@example.com
   Nombre: Admin User
   Rol: admin
   Activo: True
```

**Error - Variables faltantes**:
```
❌ Error: ADMIN_EMAIL y ADMIN_PASSWORD son requeridos

   Configura las variables de entorno antes de ejecutar:
   export ADMIN_EMAIL='admin@example.com'
   export ADMIN_PASSWORD='YourSecurePassword123!'
   export ADMIN_FULL_NAME='Admin User'  # Opcional
```

**Error - Email inválido**:
```
❌ Email inválido: invalid-email
   Usa un formato válido: nombre@dominio.com
```

**Error - Password débil**:
```
❌ Password inválido: Password debe tener al menos 8 caracteres

   Requisitos del password:
   - Mínimo 8 caracteres
   - Al menos una mayúscula
   - Al menos una minúscula
   - Al menos un número
```

## 🔒 Seguridad

### Buenas Prácticas

1. **NUNCA commitear credenciales en código**
   - ✅ Usar variables de entorno
   - ❌ Hardcodear en scripts

2. **Usar passwords seguros**
   - ✅ Mínimo 8 caracteres con complejidad
   - ✅ Usar generadores de passwords
   - ❌ Passwords simples o predecibles

3. **Proteger variables de entorno**
   - ✅ Usar .env para desarrollo (en .gitignore)
   - ✅ Configurar secrets en Render para producción
   - ❌ Exponer en logs o outputs

4. **Passwords de producción**
   - ✅ Usar passwords únicos y complejos
   - ✅ Rotarlos regularmente
   - ✅ Almacenarlos en gestor de passwords
   - ❌ Reutilizar passwords de desarrollo

### Ejemplo de Password Seguro

```bash
# Generar password seguro con OpenSSL
openssl rand -base64 32
# Ejemplo: Kj8mN2pL9xW3vB5nQ7rT1sY4uZ6wA8cE

# O usar generador online seguro
# https://passwordsgenerator.net/
```

## 🛡️ Validaciones Implementadas

### Validación de Email

- Formato: `usuario@dominio.extensión`
- Caracteres permitidos: letras, números, puntos, guiones, guión bajo
- Dominio válido requerido

### Validación de Password

- **Longitud**: Mínimo 8 caracteres
- **Mayúsculas**: Al menos 1
- **Minúsculas**: Al menos 1
- **Números**: Al menos 1
- **Recomendado**: Incluir caracteres especiales (@, #, $, etc.)

## 📊 Troubleshooting

### Problema: "ADMIN_EMAIL y ADMIN_PASSWORD son requeridos"

**Causa**: Variables de entorno no están configuradas

**Solución**:
```bash
# Verificar variables
echo $ADMIN_EMAIL
echo $ADMIN_PASSWORD

# Si están vacías, configurarlas
export ADMIN_EMAIL="tu-email@example.com"
export ADMIN_PASSWORD="TuPassword123!"
```

### Problema: "Email inválido"

**Causa**: Formato de email incorrecto

**Solución**:
```bash
# Verificar formato
# ✅ Correcto: admin@example.com
# ❌ Incorrecto: admin@, admin, @example.com

export ADMIN_EMAIL="admin@example.com"
```

### Problema: "Password debe tener al menos X"

**Causa**: Password no cumple requisitos de complejidad

**Solución**:
```bash
# Asegurar que incluye:
# - Mínimo 8 caracteres
# - Mayúsculas (A-Z)
# - Minúsculas (a-z)
# - Números (0-9)

export ADMIN_PASSWORD="SecurePass123!"
```

### Problema: "Error al conectar a base de datos"

**Causa**: Base de datos no está corriendo o no es accesible

**Solución**:
```bash
# Verificar que PostgreSQL está activo
docker-compose ps

# Si no está activo, iniciar servicios
docker-compose up -d

# Verificar conexión
docker-compose exec backend python -c "from app.database import engine; engine.connect()"
```

## 🔄 Workflow Completo

### Setup Inicial (Primera vez)

```bash
# 1. Copiar template de variables
cp .env.example .env

# 2. Editar .env con credenciales reales
# ADMIN_EMAIL=tu-email@example.com
# ADMIN_PASSWORD=TuPasswordSeguro123!
# ADMIN_FULL_NAME=Tu Nombre

# 3. Cargar variables (bash/zsh)
source .env

# 4. Iniciar servicios
docker-compose up -d

# 5. Crear usuario admin
docker-compose exec backend python -m app.scripts.init_admin
```

### Actualizar Password Existente

```bash
# 1. Configurar nuevas credenciales
export ADMIN_EMAIL="admin@example.com"  # Email existente
export ADMIN_PASSWORD="NewSecurePass456!"  # Nuevo password

# 2. Ejecutar script (detecta usuario existente y actualiza)
docker-compose exec backend python -m app.scripts.init_admin
```

### Verificar Usuario Creado

```bash
# Opción 1: Via API
curl http://localhost:8000/api/v1/auth/me \
  -H "Authorization: Bearer YOUR_JWT_TOKEN"

# Opción 2: Via shell Python
docker-compose exec backend python
>>> from app.database import SessionLocal
>>> from app.models.auth import User
>>> db = SessionLocal()
>>> admin = db.query(User).filter(User.email == "admin@example.com").first()
>>> print(f"Email: {admin.email}, Role: {admin.role}")
```

## 📝 Notas Importantes

1. **Idempotencia**: El script es idempotente - ejecutarlo múltiples veces es seguro
2. **Sin Bypass**: No hay atajos o bypasses de seguridad implementados
3. **Logging**: Todas las acciones se registran en logs del backend
4. **Rollback**: Si hay error, se hace rollback automático de cambios en DB
5. **Roles**: El script siempre crea usuarios con rol "admin"

## 🔗 Referencias

- [FastAPI Security](https://fastapi.tiangolo.com/tutorial/security/)
- [SQLAlchemy Async](https://docs.sqlalchemy.org/en/14/orm/extensions/asyncio.html)
- [Password Security Best Practices](https://cheatsheetseries.owasp.org/cheatsheets/Password_Storage_Cheat_Sheet.html)
- [Environment Variables Security](https://12factor.net/config)

---

> **Última actualización**: 2025-10-02
> **Mantenido por**: xFarma Team
> **Versión**: 1.0.0
