# backend/app/schemas/user_pharmacy.py
"""
Pydantic schemas para gestión de usuarios y farmacias.

Issue #348 FASE 2: Backend - Services & APIs
"""
from datetime import datetime
from typing import Optional, List
from uuid import UUID
import re
from enum import Enum

from pydantic import BaseModel, EmailStr, Field, field_validator

# Import SubscriptionPlan from canonical source (FIX 2)
from app.core.subscription_limits import SubscriptionPlan


class UserRole(str, Enum):
    """Roles de usuario disponibles"""
    USER = "user"
    ADMIN = "admin"


class PharmacyBase(BaseModel):
    """Base schema para farmacia"""
    name: str = Field(..., min_length=1, max_length=255)
    nif: str = Field(..., min_length=9, max_length=20)
    license_number: Optional[str] = Field(None, max_length=50)
    address: Optional[str] = Field(None, max_length=500)
    city: Optional[str] = Field(None, max_length=100)
    province: Optional[str] = Field(None, max_length=100)
    postal_code: Optional[str] = Field(None, max_length=10)
    # Issue #504: Códigos oficiales INE/ISO
    municipality_code: Optional[str] = Field(None, max_length=10, description="Código INE municipio")
    region_code: Optional[str] = Field(None, max_length=5, description="Código ISO 3166-2:ES")
    phone: Optional[str] = Field(None, max_length=20)
    email: Optional[EmailStr] = None
    erp_type: Optional[str] = Field(None, max_length=50)


class PharmacyCreate(PharmacyBase):
    """Schema para crear farmacia"""
    pass


class PharmacyUpdate(BaseModel):
    """Schema para actualizar farmacia"""
    name: Optional[str] = Field(None, min_length=1, max_length=255)
    license_number: Optional[str] = Field(None, max_length=50)
    address: Optional[str] = Field(None, max_length=500)
    city: Optional[str] = Field(None, max_length=100)
    province: Optional[str] = Field(None, max_length=100)
    postal_code: Optional[str] = Field(None, max_length=10)
    # Issue #504: Códigos oficiales INE/ISO
    municipality_code: Optional[str] = Field(None, max_length=10, description="Código INE municipio")
    region_code: Optional[str] = Field(None, max_length=5, description="Código ISO 3166-2:ES")
    phone: Optional[str] = Field(None, max_length=20)
    email: Optional[EmailStr] = None
    erp_type: Optional[str] = Field(None, max_length=50)
    subscription_plan: Optional[SubscriptionPlan] = None


class PharmacyResponse(PharmacyBase):
    """Response schema para farmacia"""
    id: UUID
    code: Optional[str]
    subscription_plan: Optional[str] = None  # Puede ser None en datos legacy
    is_active: Optional[bool] = True  # Default True si None
    created_at: Optional[datetime] = None  # Puede ser None en datos legacy
    updated_at: Optional[datetime] = None  # Puede ser None en datos legacy

    class Config:
        from_attributes = True


class UserBase(BaseModel):
    """Base schema para usuario"""
    email: EmailStr
    username: str = Field(..., min_length=3, max_length=100)
    full_name: Optional[str] = Field(None, min_length=1, max_length=255)  # Puede ser None en datos legacy
    phone: Optional[str] = Field(None, max_length=20)
    dni_nie: Optional[str] = Field(None, max_length=20)


class UserCreate(UserBase):
    """Schema para crear usuario con farmacia"""
    password: str = Field(..., min_length=8)

    @field_validator('password')
    @classmethod
    def validate_password(cls, v):
        """
        Validar fortaleza del password (FIX 4).

        Requisitos:
        - Mínimo 8 caracteres
        - Al menos 1 mayúscula
        - Al menos 1 minúscula
        - Al menos 1 dígito
        """
        if len(v) < 8:
            raise ValueError('Password must be at least 8 characters long')

        if not re.search(r'[A-Z]', v):
            raise ValueError('Password must contain at least one uppercase letter')

        if not re.search(r'[a-z]', v):
            raise ValueError('Password must contain at least one lowercase letter')

        if not re.search(r'[0-9]', v):
            raise ValueError('Password must contain at least one digit')

        return v


class UserUpdate(BaseModel):
    """Schema para actualizar usuario"""
    full_name: Optional[str] = Field(None, min_length=1, max_length=255)
    phone: Optional[str] = Field(None, max_length=20)
    password: Optional[str] = Field(None, min_length=8, max_length=255)  # Permitir cambio de contraseña
    role: Optional[UserRole] = None
    subscription_plan: Optional[SubscriptionPlan] = None
    is_active: Optional[bool] = None
    is_verified: Optional[bool] = None
    notify_uploads: Optional[bool] = None
    notify_errors: Optional[bool] = None
    notify_analysis: Optional[bool] = None

    @field_validator('password')
    @classmethod
    def validate_password(cls, v):
        """
        Validar fortaleza del password si se proporciona (opcional en updates).
        """
        if v is None:
            return v

        if len(v) < 8:
            raise ValueError('La contraseña debe tener al menos 8 caracteres')
        if not any(c.isupper() for c in v):
            raise ValueError('La contraseña debe contener al menos una mayúscula')
        if not any(c.islower() for c in v):
            raise ValueError('La contraseña debe contener al menos una minúscula')
        if not any(c.isdigit() for c in v):
            raise ValueError('La contraseña debe contener al menos un número')
        return v


class UserResponse(UserBase):
    """Response schema para usuario"""
    id: UUID
    pharmacy_id: UUID
    role: str
    role_id: Optional[UUID]
    subscription_plan: str
    is_active: bool
    is_verified: bool
    created_at: datetime
    last_login: Optional[datetime]
    deleted_at: Optional[datetime]
    pharmacy: Optional[PharmacyResponse] = None

    class Config:
        from_attributes = True


class UserWithPharmacyCreate(BaseModel):
    """Schema para crear usuario con farmacia (transacción atómica)"""
    user: UserCreate
    pharmacy: PharmacyCreate
    role: UserRole = UserRole.USER
    subscription_plan: SubscriptionPlan = SubscriptionPlan.FREE


class UserListResponse(BaseModel):
    """Response para listado paginado de usuarios"""
    users: List[UserResponse]
    total: int
    skip: int
    limit: int


class StorageUsageResponse(BaseModel):
    """Response para uso de almacenamiento (Admin)"""
    pharmacy_id: UUID
    pharmacy_name: str
    total_files: int
    total_size_mb: float
    sales_records_count: int
    enriched_records_count: int
    last_upload_date: Optional[datetime]
    subscription_plan: str

    class Config:
        from_attributes = True


class UserStorageUsageResponse(BaseModel):
    """
    Response para uso de almacenamiento del usuario actual.
    Issue #420: FREE tier experience - Subscription page.
    """
    total_used_mb: float = Field(..., description="Total MB usados")
    limit_mb: int = Field(..., description="Límite de almacenamiento en MB")
    percentage: float = Field(..., description="Porcentaje de uso (0-100)")
    plan: str = Field(..., description="Plan de suscripción actual")
    warning_threshold: int = Field(default=80, description="Umbral de advertencia (%)")
