"""
Symptom-Based Taxonomy for Pharmacy Products - Issue #456

This module defines a symptom-centric taxonomy that classifies products
by the PATIENT NEED they address, not their legal status.

Philosophy:
- "What symptom does this solve?" > "What type of product is this?"
- A pharmacist thinks: "Customer has headache → what solutions do I have?"
- Cross-selling becomes intuitive: Espidifen + Thermacare + Migrastick

Example cluster "dolor":
- Espidifen 600mg (Medicamento)
- Thermacare Parche (Producto Sanitario)
- Fisiocrem Gel (Parafarmacia)
- Magnesio Total (Suplemento)
- Migrastick (Aceite Esencial)

All solve the same problem: PAIN.
"""

from enum import Enum
from typing import Dict, List, Optional
from pydantic import BaseModel


class NecesidadPrincipal(str, Enum):
    """21 main symptom groups covering a complete pharmacy."""

    DOLOR_FIEBRE = "dolor_fiebre"
    RESPIRATORIO = "respiratorio"
    DIGESTIVO = "digestivo"
    PIEL = "piel"
    BUCAL = "bucal"
    OCULAR = "ocular"
    OIDOS = "oidos"               # New: Ear health
    CAPILAR = "capilar"
    BIENESTAR = "bienestar"
    CIRCULATORIO = "circulatorio"
    URINARIO = "urinario"         # New: Urinary system
    SEXUAL = "sexual"
    INFANTIL = "infantil"
    NUTRICION = "nutricion"
    MOVILIDAD = "movilidad"
    INCONTINENCIA_CAT = "incontinencia_cat"  # New: Incontinence
    DIABETES_CAT = "diabetes_cat"            # New: Diabetes management
    BOTIQUIN = "botiquin"
    # Issue #467: Separar categorías que no son botiquín
    VETERINARIA_CAT = "veterinaria_cat"      # Productos para mascotas
    SERVICIOS_CAT = "servicios_cat"          # Servicios de farmacia
    NO_SALUD = "no_salud"                    # Regalos, perfumería, joyería, dulcería


class NecesidadEspecifica(str, Enum):
    """
    70+ specific symptom categories.

    Naming convention: lowercase with underscores.
    Each belongs to one NecesidadPrincipal parent.
    """

    # === DOLOR Y FIEBRE ===
    DOLOR_MUSCULAR = "dolor_muscular"           # Contracturas, agujetas, lumbalgia
    DOLOR_CABEZA = "dolor_cabeza"               # Migrañas, cefaleas tensionales
    DOLOR_ARTICULAR = "dolor_articular"         # Artritis, artrosis, reuma
    DOLOR_MENSTRUAL = "dolor_menstrual"         # Dismenorrea, SPM
    DOLOR_DENTAL = "dolor_dental"               # Muelas, sensibilidad
    DOLOR_FIEBRE = "dolor_fiebre"               # Analgésicos generales
    DOLOR_GARGANTA = "dolor_garganta"           # Faringitis, amigdalitis
    HEMATOMAS = "hematomas"                     # Golpes, contusiones, moratones

    # === RESPIRATORIO ===
    TOS_SECA = "tos_seca"                       # Antitusivos
    MUCOSIDAD_RESPIRATORIA = "mucosidad_respiratoria"  # Mucolíticos (acetilcisteína), expectorantes
    CONGESTION_NASAL = "congestion_nasal"       # Descongestivos, sprays
    HIGIENE_NASAL = "higiene_nasal"             # Suero fisiológico, agua de mar, lavados nasales
    GRIPE_RESFRIADO = "gripe_resfriado"         # Antigripales combinados
    ALERGIAS = "alergias"                       # Antihistamínicos, rinitis
    ASMA = "asma"                               # Broncodilatadores OTC
    ASMA_BRONQUITIS = "asma_bronquitis"         # Asma, bronquitis crónica
    BALSAMOS_RESPIRATORIOS = "balsamos_respiratorios"  # Vaporub, bálsamos pectorales, aceites esenciales

    # === DIGESTIVO ===
    ACIDEZ_REFLUJO = "acidez_reflujo"           # Antiácidos, IBP
    DIARREA = "diarrea"                         # Antidiarreicos, suero oral
    ESTRENIMIENTO = "estrenimiento"             # Laxantes, fibra
    GASES_FLATULENCIA = "gases_flatulencia"     # Antiflatulentos
    NAUSEAS_VOMITOS = "nauseas_vomitos"         # Antieméticos, jengibre
    DIGESTION_PESADA = "digestion_pesada"       # Enzimas digestivas
    DIGESTIVO = "digestivo"                     # Salud digestiva general
    COLON_IRRITABLE = "colon_irritable"         # Síndrome colon irritable
    COLICOS = "colicos"                         # Cólicos abdominales
    PROBIOTICOS = "probioticos"                 # Flora intestinal
    PARASITOS = "parasitos"                     # Antiparasitarios

    # === PIEL ===
    HIDRATACION_CORPORAL = "hidratacion_corporal"   # Cremas corporales
    HIGIENE_CORPORAL = "higiene_corporal"           # Geles, jabones corporales
    DESODORANTE = "desodorante"                     # Desodorantes, antitranspirantes
    HIDRATACION_FACIAL = "hidratacion_facial"       # Cremas faciales
    PROTECCION_SOLAR = "proteccion_solar"           # SPF, after sun
    ACNE = "acne"                                   # Antiacné
    DERMATITIS = "dermatitis"                       # Dermatitis general
    DERMATITIS_ECCEMA = "dermatitis_eccema"         # Atopia, eccema
    PIEL_ATOPICA = "piel_atopica"                   # Atópica bebés/adultos
    PSORIASIS = "psoriasis"                         # Psoriasis
    # Issue #457: Consolidación categorías heridas
    DESINFECCION_HERIDAS = "desinfeccion_heridas"   # Antisépticos (betadine, alcohol, clorhexidina)
    CICATRIZACION = "cicatrizacion"                 # Cicatrizantes (Bio-Oil, Bepanthol, rosa mosqueta)
    # APOSITOS_CURAS definido en BOTIQUIN
    PICADURAS = "picaduras"                         # Antihistamínicos tópicos
    QUEMADURAS = "quemaduras"                       # Quemaduras leves
    HONGOS_PIEL = "hongos_piel"                     # Antifúngicos tópicos
    HERPES = "herpes"                               # Herpes labial
    SARNA = "sarna"                                 # Escabiosis
    VERRUGAS_CALLOS = "verrugas_callos"             # Queratolíticos
    MANCHAS = "manchas"                             # Despigmentantes
    ARRUGAS_ANTIEDAD = "arrugas_antiedad"           # Antienvejecimiento
    CONTORNO_OJOS = "contorno_ojos"                 # Ojeras, bolsas
    LIMPIEZA_FACIAL = "limpieza_facial"             # Desmaquillantes
    EXFOLIACION = "exfoliacion"                     # Peelings, scrubs
    CELULITIS = "celulitis"                         # Anticelulíticos
    ESTRIAS = "estrias"                             # Antiestrías
    BRONCEADO = "bronceado"                         # Autobronceadores
    CUIDADO_LABIAL = "cuidado_labial"               # Bálsamos, protectores labiales
    ACCESORIOS_BANO = "accesorios_bano"             # Esponjas, cepillos cuerpo, guantes exfoliantes

    # === BUCAL ===
    # Subcategorías desglosadas de higiene_dental (Issue #457)
    CEPILLO_MANUAL = "cepillo_manual"           # Cepillos de dientes manuales
    CEPILLO_ELECTRICO = "cepillo_electrico"     # Cepillos eléctricos, recambios
    PASTA_DENTAL = "pasta_dental"               # Dentífricos, pastas de dientes
    ENJUAGUE_BUCAL = "enjuague_bucal"           # Colutorios, enjuagues
    SEDA_DENTAL = "seda_dental"                 # Seda, hilo dental
    INTERDENTAL = "interdental"                 # Cepillos interdentales (Interprox, TePe)
    IRRIGADOR_DENTAL = "irrigador_dental"       # Irrigadores bucales
    HIGIENE_BUCAL_INFANTIL = "higiene_bucal_infantil"  # Cepillos niños, pastas sabores
    HIGIENE_DENTAL = "higiene_dental"           # Fallback: otros productos higiene dental
    ENCIAS = "encias"                           # Gingivitis, sangrado
    AFTAS_LLAGAS = "aftas_llagas"               # Úlceras bucales
    BLANQUEAMIENTO = "blanqueamiento"           # Blanqueamiento dental
    PROTESIS_DENTAL = "protesis_dental"         # Adhesivos, limpiadores
    HALITOSIS = "halitosis"                     # Mal aliento
    SENSIBILIDAD_DENTAL = "sensibilidad_dental" # Dientes sensibles

    # === OCULAR ===
    OJO_SECO = "ojo_seco"                       # Lágrimas artificiales
    CONJUNTIVITIS = "conjuntivitis"             # Colirios
    ALERGIA_OCULAR = "alergia_ocular"           # Alergia ocular
    FATIGA_VISUAL = "fatiga_visual"             # Pantallas, cansancio
    LENTILLAS = "lentillas"                     # Soluciones, gotas
    HIGIENE_OCULAR = "higiene_ocular"           # Toallitas, sueros
    OPTICA = "optica"                           # Gafas presbicia, lupas, cordones, fundas

    # === OIDOS ===
    HIGIENE_OIDOS = "higiene_oidos"             # Limpieza oídos
    CERUMEN = "cerumen"                         # Tapones cerumen
    DOLOR_OIDOS = "dolor_oidos"                 # Dolor de oídos
    OTITIS = "otitis"                           # Otitis externa

    # === CAPILAR ===
    CAIDA_CABELLO = "caida_cabello"             # Minoxidil, biotina, Triphasic, Quinina
    CASPA = "caspa"                             # Anticaspa seca
    CABELLO_GRASO = "cabello_graso"             # Seborrea, raíz grasa
    PIOJOS = "piojos"                           # Pediculicidas
    # Nuevas categorías capilares (Issue #457 - Klorane, René Furterer, Lazartigue)
    CAPILAR_CUERO_SENSIBLE = "capilar_cuero_sensible"       # Astera, Peonía - picor, irritación
    CAPILAR_HIDRATACION = "capilar_hidratacion"             # Karité, Cupuaçu, Mango - pelo seco/dañado
    CAPILAR_COLOR = "capilar_color"                         # Okara - protección color
    CAPILAR_DENSIDAD = "capilar_densidad"                   # Tonucia - pelo fino/envejecido
    CAPILAR_FRECUENTE = "capilar_frecuente"                 # Naturia, Avena - uso diario
    CANAS = "canas"                                         # Tintes, camuflaje canas

    # === BIENESTAR / SUEÑO / ÁNIMO ===
    SUENO_INSOMNIO = "sueno_insomnio"           # Melatonina, valeriana
    MELATONINA = "melatonina"                   # Issue #457: Melatonina específica (Melamil, ZzzQuil)
    ESTRES_ANSIEDAD = "estres_ansiedad"         # Adaptógenos, magnesio
    ENERGIA_VITALIDAD = "energia_vitalidad"     # Estimulantes naturales
    CONCENTRACION_MEMORIA = "concentracion_memoria"  # Nootrópicos
    ESTADO_ANIMO = "estado_animo"               # Hipérico, omega-3
    MAREO_VIAJE = "mareo_viaje"                 # Issue #457: Cinetosis, Biodramina

    # === CIRCULATORIO ===
    PIERNAS_CANSADAS = "piernas_cansadas"       # Venotónicos
    VARICES = "varices"                         # Compresión, geles fríos
    HEMORROIDES = "hemorroides"                 # Antihemorroides
    TENSION_ARTERIAL = "tension_arterial"       # Tensiómetros, suplementos
    COLESTEROL = "colesterol"                   # Omega-3, levadura roja
    RETENCION_LIQUIDOS = "retencion_liquidos"   # Drenantes

    # === SEXUAL Y REPRODUCTIVO ===
    SALUD_SEXUAL = "salud_sexual"               # Preservativos, lubricantes
    FERTILIDAD = "fertilidad"                   # Tests ovulación, suplementos
    MENOPAUSIA = "menopausia"                   # Isoflavonas, lubricantes
    HIGIENE_INTIMA = "higiene_intima"           # Geles íntimos
    FLORA_VAGINAL = "flora_vaginal"             # Probióticos vaginales
    CANDIDIASIS_VAGINAL = "candidiasis_vaginal" # Antifúngicos vaginales
    VAGINOSIS = "vaginosis"                     # Vaginosis bacteriana
    CICLO_MENSTRUAL = "ciclo_menstrual"         # SPM, dismenorrea
    EMBARAZO_LACTANCIA = "embarazo_lactancia"   # Ácido fólico, vitaminas
    DISFUNCION_ERECTIL = "disfuncion_erectil"   # OTC permitidos

    # === INFANTIL ===
    ALIMENTACION_BEBE = "alimentacion_bebe"     # Leches, papillas
    HIGIENE_BEBE = "higiene_bebe"               # Pañales, toallitas
    COLICOS_BEBE = "colicos_bebe"               # Antiflatulentos bebé
    DENTICION = "denticion"                     # Mordedores, geles
    DERMATITIS_PANAL = "dermatitis_panal"       # Cremas culito
    FIEBRE_BEBE = "fiebre_bebe"                 # Paracetamol/ibuprofeno pediátrico
    BEBES_INFANTIL = "bebes_infantil"           # Issue #457: Accesorios bebé (biberones, limas, chupetes)

    # === NUTRICIÓN Y SUPLEMENTOS ===
    VITAMINAS_MINERALES = "vitaminas_minerales" # Multivitamínicos
    VITAMINAS_GENERAL = "vitaminas_general"     # Issue #457: Vitaminas sin minerales (B, C, D, etc.)
    VITAMINAS_INFANTIL = "vitaminas_infantil"   # Issue #457: Vitaminas para niños
    OMEGA3 = "omega3"                           # Aceites de pescado
    HIERRO = "hierro"                           # Anemia
    CALCIO_VITAMINA_D = "calcio_vitamina_d"     # Calcio + Vitamina D combinados (huesos)
    VITAMINA_D = "vitamina_d"                   # Issue #457: Vitamina D sola (inmunidad, bienestar, déficit)
    HUESOS = "huesos"                           # Salud ósea general
    MAGNESIO = "magnesio"                       # Muscular, nervioso
    PROTEINAS_DEPORTE = "proteinas_deporte"     # Proteínas, BCAA
    NUTRICION_DEPORTIVA = "nutricion_deportiva" # Issue #457: Barritas proteicas, suplementos deporte
    CONTROL_PESO = "control_peso"               # Quemagrasas, saciantes
    DETOX = "detox"                             # Depurativos
    COLAGENO = "colageno"                       # Articulaciones, piel
    NUTRICION_CLINICA = "nutricion_clinica"     # Nutrición enteral
    HIDRATACION = "hidratacion"                 # Hidratación deportiva
    INFUSIONES = "infusiones"                   # Issue #457: Infusiones/tés medicinales

    # === SISTEMA URINARIO ===
    INFECCION_URINARIA = "infeccion_urinaria"   # Cistitis, ITU
    PROSTATA = "prostata"                       # Salud prostática

    # === INCONTINENCIA ===
    INCONTINENCIA = "incontinencia"             # Absorbentes adultos

    # === DIABETES ===
    DIABETES = "diabetes"                       # Glucómetros, lancetas

    # === MOVILIDAD Y ORTOPEDIA ===
    MATERIAL_ORTOPEDICO = "material_ortopedico" # Muñequeras, rodilleras
    ORTOPEDIA_PIE = "ortopedia_pie"             # Plantillas, juanetes
    REHABILITACION = "rehabilitacion"           # Bandas, pelotas
    COMPRESION_VENDAJES = "compresion_vendajes" # Vendas, medias
    AYUDAS_TECNICAS = "ayudas_tecnicas"         # Andadores, sillas

    # === BOTIQUÍN ===
    APOSITOS_CURAS = "apositos_curas"           # Tiritas, gasas, vendas, apósitos (Issue #457)
    TERMOMETROS = "termometros"                 # Digitales, infrarrojos
    # Tests diagnóstico (Issue #457: desglose por tipo)
    TEST_EMBARAZO = "test_embarazo"             # Clearblue, Predictor, tests embarazo
    TEST_OVULACION = "test_ovulacion"           # Tests ovulación, fertilidad
    TEST_COVID_GRIPE = "test_covid_gripe"       # Tests COVID, gripe, antígenos
    TEST_DIAGNOSTICO = "test_diagnostico"       # Otros: glucosa, ITU, HbA1c, vitamina D
    MATERIAL_SANITARIO = "material_sanitario"   # Jeringuillas, guantes, mascarillas, kits, pinzas

    # === SERVICIOS ===
    SERVICIOS_FARMACIA = "servicios_farmacia"   # Tensión, tests, formulación, alquileres

    # === NO SALUD ===
    REGALOS_BAZAR = "regalos_bazar"             # Juguetes, adornos, regalos, merchandising
    DULCERIA = "dulceria"                       # Caramelos sin propiedades medicinales (Issue #457)
    PERFUMERIA = "perfumeria"                   # Issue #457: Perfumes, colonias
    JOYERIA = "joyeria"                         # Issue #457: Pendientes, joyas

    # === CUIDADO DEPENDIENTES (Issue #457) ===
    CUIDADO_DEPENDIENTES = "cuidado_dependientes"  # Esponjas enjabonadas, toallitas adultos

    # === BELLEZA Y CUIDADO PERSONAL (Issue #457) ===
    UNAS = "unas"                               # Quitaesmalte, manicura, cuidado uñas
    TATUAJES = "tatuajes"                       # Aftercare tatuajes, cremas cicatrizantes tattoo
    CUIDADO_CABELLO = "cuidado_cabello"         # Champú en seco, styling general

    # === VETERINARIA (Issue #457) ===
    VETERINARIA = "veterinaria"                 # Productos para mascotas (Fortekor, antiparasitarios pet)

    # === ALIASES/SHORTCUTS (Issue #457) ===
    # Categorías que el usuario usa como shortcut pero mapean a otras
    DOLOR = "dolor"                             # Alias → dolor_fiebre (analgésicos genéricos OTC)
    ALERGIA = "alergia"                         # Alias → alergias (antihistamínicos)
    FLORA_INTESTINAL = "flora_intestinal"       # Alias → probioticos
    HONGOS_PIES = "hongos_pies"                 # Especificación de hongos_piel para pies
    CIRCULACION_PIERNAS = "circulacion_piernas" # Alias → piernas_cansadas

    # === FALLBACK ===
    OTROS = "otros"                             # Sin clasificar


# Mapping: Subcategory -> Parent Category
TAXONOMY_HIERARCHY: Dict[NecesidadEspecifica, NecesidadPrincipal] = {
    # DOLOR_FIEBRE
    NecesidadEspecifica.DOLOR_MUSCULAR: NecesidadPrincipal.DOLOR_FIEBRE,
    NecesidadEspecifica.DOLOR_CABEZA: NecesidadPrincipal.DOLOR_FIEBRE,
    NecesidadEspecifica.DOLOR_ARTICULAR: NecesidadPrincipal.DOLOR_FIEBRE,
    NecesidadEspecifica.DOLOR_MENSTRUAL: NecesidadPrincipal.DOLOR_FIEBRE,
    NecesidadEspecifica.DOLOR_DENTAL: NecesidadPrincipal.DOLOR_FIEBRE,
    NecesidadEspecifica.DOLOR_FIEBRE: NecesidadPrincipal.DOLOR_FIEBRE,
    NecesidadEspecifica.DOLOR_GARGANTA: NecesidadPrincipal.DOLOR_FIEBRE,
    NecesidadEspecifica.HEMATOMAS: NecesidadPrincipal.DOLOR_FIEBRE,

    # RESPIRATORIO
    NecesidadEspecifica.TOS_SECA: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.MUCOSIDAD_RESPIRATORIA: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.CONGESTION_NASAL: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.HIGIENE_NASAL: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.GRIPE_RESFRIADO: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.ALERGIAS: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.ASMA: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.ASMA_BRONQUITIS: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.BALSAMOS_RESPIRATORIOS: NecesidadPrincipal.RESPIRATORIO,

    # DIGESTIVO
    NecesidadEspecifica.ACIDEZ_REFLUJO: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.DIARREA: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.ESTRENIMIENTO: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.GASES_FLATULENCIA: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.NAUSEAS_VOMITOS: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.DIGESTION_PESADA: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.DIGESTIVO: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.COLON_IRRITABLE: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.COLICOS: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.PROBIOTICOS: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.PARASITOS: NecesidadPrincipal.DIGESTIVO,

    # PIEL
    NecesidadEspecifica.HIDRATACION_CORPORAL: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.HIGIENE_CORPORAL: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.DESODORANTE: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.HIDRATACION_FACIAL: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.PROTECCION_SOLAR: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.ACNE: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.DERMATITIS: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.DERMATITIS_ECCEMA: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.PIEL_ATOPICA: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.PSORIASIS: NecesidadPrincipal.PIEL,
    # Issue #457: Consolidación categorías heridas
    NecesidadEspecifica.DESINFECCION_HERIDAS: NecesidadPrincipal.BOTIQUIN,
    NecesidadEspecifica.CICATRIZACION: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.PICADURAS: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.QUEMADURAS: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.HONGOS_PIEL: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.HERPES: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.SARNA: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.VERRUGAS_CALLOS: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.MANCHAS: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.ARRUGAS_ANTIEDAD: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.CONTORNO_OJOS: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.LIMPIEZA_FACIAL: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.EXFOLIACION: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.CELULITIS: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.ESTRIAS: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.BRONCEADO: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.CUIDADO_LABIAL: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.ACCESORIOS_BANO: NecesidadPrincipal.PIEL,

    # BUCAL (Issue #457: subcategorías desglosadas)
    NecesidadEspecifica.CEPILLO_MANUAL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.CEPILLO_ELECTRICO: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.PASTA_DENTAL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.ENJUAGUE_BUCAL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.SEDA_DENTAL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.INTERDENTAL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.IRRIGADOR_DENTAL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.HIGIENE_BUCAL_INFANTIL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.HIGIENE_DENTAL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.ENCIAS: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.AFTAS_LLAGAS: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.BLANQUEAMIENTO: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.PROTESIS_DENTAL: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.HALITOSIS: NecesidadPrincipal.BUCAL,
    NecesidadEspecifica.SENSIBILIDAD_DENTAL: NecesidadPrincipal.BUCAL,

    # OCULAR
    NecesidadEspecifica.OJO_SECO: NecesidadPrincipal.OCULAR,
    NecesidadEspecifica.CONJUNTIVITIS: NecesidadPrincipal.OCULAR,
    NecesidadEspecifica.ALERGIA_OCULAR: NecesidadPrincipal.OCULAR,
    NecesidadEspecifica.FATIGA_VISUAL: NecesidadPrincipal.OCULAR,
    NecesidadEspecifica.LENTILLAS: NecesidadPrincipal.OCULAR,
    NecesidadEspecifica.HIGIENE_OCULAR: NecesidadPrincipal.OCULAR,
    NecesidadEspecifica.OPTICA: NecesidadPrincipal.OCULAR,

    # OIDOS
    NecesidadEspecifica.HIGIENE_OIDOS: NecesidadPrincipal.OIDOS,
    NecesidadEspecifica.CERUMEN: NecesidadPrincipal.OIDOS,
    NecesidadEspecifica.DOLOR_OIDOS: NecesidadPrincipal.OIDOS,
    NecesidadEspecifica.OTITIS: NecesidadPrincipal.OIDOS,

    # CAPILAR
    NecesidadEspecifica.CAIDA_CABELLO: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.CASPA: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.CABELLO_GRASO: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.PIOJOS: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.CAPILAR_CUERO_SENSIBLE: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.CAPILAR_HIDRATACION: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.CAPILAR_COLOR: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.CAPILAR_DENSIDAD: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.CAPILAR_FRECUENTE: NecesidadPrincipal.CAPILAR,
    NecesidadEspecifica.CANAS: NecesidadPrincipal.CAPILAR,

    # BIENESTAR → NUTRICION (Issue #467: fusionado en Suplementos)
    NecesidadEspecifica.SUENO_INSOMNIO: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.ESTRES_ANSIEDAD: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.ENERGIA_VITALIDAD: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.CONCENTRACION_MEMORIA: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.ESTADO_ANIMO: NecesidadPrincipal.NUTRICION,

    # CIRCULATORIO
    NecesidadEspecifica.PIERNAS_CANSADAS: NecesidadPrincipal.CIRCULATORIO,
    NecesidadEspecifica.VARICES: NecesidadPrincipal.CIRCULATORIO,
    NecesidadEspecifica.HEMORROIDES: NecesidadPrincipal.CIRCULATORIO,
    NecesidadEspecifica.TENSION_ARTERIAL: NecesidadPrincipal.CIRCULATORIO,
    NecesidadEspecifica.COLESTEROL: NecesidadPrincipal.CIRCULATORIO,
    NecesidadEspecifica.RETENCION_LIQUIDOS: NecesidadPrincipal.CIRCULATORIO,

    # SEXUAL
    NecesidadEspecifica.SALUD_SEXUAL: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.FERTILIDAD: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.MENOPAUSIA: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.HIGIENE_INTIMA: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.FLORA_VAGINAL: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.CANDIDIASIS_VAGINAL: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.VAGINOSIS: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.CICLO_MENSTRUAL: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.EMBARAZO_LACTANCIA: NecesidadPrincipal.SEXUAL,
    NecesidadEspecifica.DISFUNCION_ERECTIL: NecesidadPrincipal.SEXUAL,

    # INFANTIL
    NecesidadEspecifica.ALIMENTACION_BEBE: NecesidadPrincipal.INFANTIL,
    NecesidadEspecifica.HIGIENE_BEBE: NecesidadPrincipal.INFANTIL,
    NecesidadEspecifica.COLICOS_BEBE: NecesidadPrincipal.INFANTIL,
    NecesidadEspecifica.DENTICION: NecesidadPrincipal.INFANTIL,
    NecesidadEspecifica.DERMATITIS_PANAL: NecesidadPrincipal.INFANTIL,
    NecesidadEspecifica.FIEBRE_BEBE: NecesidadPrincipal.INFANTIL,
    NecesidadEspecifica.BEBES_INFANTIL: NecesidadPrincipal.INFANTIL,  # Issue #457

    # NUTRICION
    NecesidadEspecifica.VITAMINAS_MINERALES: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.OMEGA3: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.HIERRO: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.CALCIO_VITAMINA_D: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.VITAMINA_D: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.HUESOS: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.MAGNESIO: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.PROTEINAS_DEPORTE: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.CONTROL_PESO: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.DETOX: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.COLAGENO: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.NUTRICION_CLINICA: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.HIDRATACION: NecesidadPrincipal.NUTRICION,

    # URINARIO
    NecesidadEspecifica.INFECCION_URINARIA: NecesidadPrincipal.URINARIO,
    NecesidadEspecifica.PROSTATA: NecesidadPrincipal.URINARIO,

    # INCONTINENCIA
    NecesidadEspecifica.INCONTINENCIA: NecesidadPrincipal.INCONTINENCIA_CAT,

    # DIABETES
    NecesidadEspecifica.DIABETES: NecesidadPrincipal.DIABETES_CAT,

    # MOVILIDAD
    NecesidadEspecifica.MATERIAL_ORTOPEDICO: NecesidadPrincipal.MOVILIDAD,
    NecesidadEspecifica.ORTOPEDIA_PIE: NecesidadPrincipal.MOVILIDAD,
    NecesidadEspecifica.REHABILITACION: NecesidadPrincipal.MOVILIDAD,
    NecesidadEspecifica.COMPRESION_VENDAJES: NecesidadPrincipal.MOVILIDAD,
    NecesidadEspecifica.AYUDAS_TECNICAS: NecesidadPrincipal.MOVILIDAD,

    # BOTIQUIN
    NecesidadEspecifica.APOSITOS_CURAS: NecesidadPrincipal.BOTIQUIN,
    NecesidadEspecifica.TERMOMETROS: NecesidadPrincipal.BOTIQUIN,
    NecesidadEspecifica.TEST_EMBARAZO: NecesidadPrincipal.BOTIQUIN,
    NecesidadEspecifica.TEST_OVULACION: NecesidadPrincipal.BOTIQUIN,
    NecesidadEspecifica.TEST_COVID_GRIPE: NecesidadPrincipal.BOTIQUIN,
    NecesidadEspecifica.TEST_DIAGNOSTICO: NecesidadPrincipal.BOTIQUIN,
    NecesidadEspecifica.MATERIAL_SANITARIO: NecesidadPrincipal.BOTIQUIN,

    # SERVICIOS (Issue #467: separado de Botiquín)
    NecesidadEspecifica.SERVICIOS_FARMACIA: NecesidadPrincipal.SERVICIOS_CAT,

    # NO SALUD (Issue #467: separado de Botiquín)
    NecesidadEspecifica.REGALOS_BAZAR: NecesidadPrincipal.NO_SALUD,
    NecesidadEspecifica.DULCERIA: NecesidadPrincipal.NO_SALUD,

    # OTROS (fallback)
    NecesidadEspecifica.OTROS: NecesidadPrincipal.BOTIQUIN,

    # Issue #457: Nuevas categorías
    NecesidadEspecifica.CUIDADO_DEPENDIENTES: NecesidadPrincipal.INCONTINENCIA_CAT,  # Junto con incontinencia
    NecesidadEspecifica.UNAS: NecesidadPrincipal.PIEL,  # Belleza/cuidado personal
    NecesidadEspecifica.TATUAJES: NecesidadPrincipal.PIEL,  # Aftercare = piel
    NecesidadEspecifica.CUIDADO_CABELLO: NecesidadPrincipal.CAPILAR,  # Capilar genérico

    # Issue #457: Nuevas categorías de nutrición
    NecesidadEspecifica.VITAMINAS_GENERAL: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.VITAMINAS_INFANTIL: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.NUTRICION_DEPORTIVA: NecesidadPrincipal.NUTRICION,
    NecesidadEspecifica.INFUSIONES: NecesidadPrincipal.NUTRICION,

    # Issue #457: Bienestar nuevas
    NecesidadEspecifica.MELATONINA: NecesidadPrincipal.NUTRICION,  # Issue #467: → Suplementos
    NecesidadEspecifica.MAREO_VIAJE: NecesidadPrincipal.NUTRICION,  # Issue #467: → Suplementos

    # Issue #457/#467: No salud (separado de Botiquín)
    NecesidadEspecifica.PERFUMERIA: NecesidadPrincipal.NO_SALUD,
    NecesidadEspecifica.JOYERIA: NecesidadPrincipal.NO_SALUD,

    # VETERINARIA (Issue #467: categoría principal propia)
    NecesidadEspecifica.VETERINARIA: NecesidadPrincipal.VETERINARIA_CAT,

    # Issue #457: Aliases (mapean a misma necesidad que su padre)
    NecesidadEspecifica.DOLOR: NecesidadPrincipal.DOLOR_FIEBRE,
    NecesidadEspecifica.ALERGIA: NecesidadPrincipal.RESPIRATORIO,
    NecesidadEspecifica.FLORA_INTESTINAL: NecesidadPrincipal.DIGESTIVO,
    NecesidadEspecifica.HONGOS_PIES: NecesidadPrincipal.PIEL,
    NecesidadEspecifica.CIRCULACION_PIERNAS: NecesidadPrincipal.CIRCULATORIO,
}


# Priority tiers for the Arbiter
class RulePriority(str, Enum):
    """
    Priority tiers for the Sandwich Strategy Arbiter.

    TIER_1: Rule ALWAYS wins (specific medical/safety categories)
    TIER_2: Rule wins if confidence > 0.8
    TIER_3: LLM can override (generic categories benefit from LLM nuance)
    """
    TIER_1 = "tier_1"  # Rule ALWAYS wins
    TIER_2 = "tier_2"  # Rule wins if confidence > 0.8
    TIER_3 = "tier_3"  # LLM can override


# Which categories are in which tier
CATEGORY_PRIORITY: Dict[str, RulePriority] = {
    # TIER 1: Rule ALWAYS wins (specific, safety-critical)
    "omega3": RulePriority.TIER_1,
    "hematomas": RulePriority.TIER_1,
    "salud_sexual": RulePriority.TIER_1,
    "probioticos": RulePriority.TIER_1,
    "hierro": RulePriority.TIER_1,
    "calcio_vitamina_d": RulePriority.TIER_1,
    "magnesio": RulePriority.TIER_1,
    "piojos": RulePriority.TIER_1,
    "parasitos": RulePriority.TIER_1,
    "disfuncion_erectil": RulePriority.TIER_1,
    "fertilidad": RulePriority.TIER_1,
    "embarazo_lactancia": RulePriority.TIER_1,
    "test_diagnostico": RulePriority.TIER_1,
    "fiebre_bebe": RulePriority.TIER_1,

    # TIER 2: Rule wins if confidence > 0.8
    # Bucal (Issue #457: subcategorías desglosadas)
    "cepillo_manual": RulePriority.TIER_2,
    "cepillo_electrico": RulePriority.TIER_2,
    "pasta_dental": RulePriority.TIER_2,
    "enjuague_bucal": RulePriority.TIER_2,
    "seda_dental": RulePriority.TIER_2,
    "interdental": RulePriority.TIER_2,
    "irrigador_dental": RulePriority.TIER_2,
    "higiene_bucal_infantil": RulePriority.TIER_2,
    "higiene_dental": RulePriority.TIER_2,
    "proteccion_solar": RulePriority.TIER_2,
    "caida_cabello": RulePriority.TIER_2,
    "dolor_muscular": RulePriority.TIER_2,
    "dolor_articular": RulePriority.TIER_2,
    "gripe_resfriado": RulePriority.TIER_2,
    "acidez_reflujo": RulePriority.TIER_2,
    "estrenimiento": RulePriority.TIER_2,
    "diarrea": RulePriority.TIER_2,
    "ojo_seco": RulePriority.TIER_2,
    "protesis_dental": RulePriority.TIER_2,
    "alimentacion_bebe": RulePriority.TIER_2,
    "hemorroides": RulePriority.TIER_2,
    "varices": RulePriority.TIER_2,
    "sueno_insomnio": RulePriority.TIER_2,

    # TIER 3: LLM can override (generic categories)
    "arrugas_antiedad": RulePriority.TIER_3,
    "hidratacion_facial": RulePriority.TIER_3,
    "hidratacion_corporal": RulePriority.TIER_3,
    "limpieza_facial": RulePriority.TIER_3,
    "vitaminas_minerales": RulePriority.TIER_3,
    "energia_vitalidad": RulePriority.TIER_3,
    "control_peso": RulePriority.TIER_3,
    "otros": RulePriority.TIER_3,
}


def get_priority(category: str) -> RulePriority:
    """Get the priority tier for a category. Default to TIER_2."""
    return CATEGORY_PRIORITY.get(category, RulePriority.TIER_2)


def get_parent_category(subcategory: str) -> Optional[str]:
    """Get the parent category for a subcategory."""
    try:
        sub = NecesidadEspecifica(subcategory)
        parent = TAXONOMY_HIERARCHY.get(sub)
        return parent.value if parent else None
    except ValueError:
        return None


def get_all_subcategories(parent: str) -> List[str]:
    """Get all subcategories for a parent category."""
    try:
        parent_enum = NecesidadPrincipal(parent)
        return [
            sub.value
            for sub, par in TAXONOMY_HIERARCHY.items()
            if par == parent_enum
        ]
    except ValueError:
        return []


# Pydantic models for API responses
class TaxonomyNode(BaseModel):
    """A node in the taxonomy tree."""
    id: str
    name: str
    parent_id: Optional[str] = None
    children: List["TaxonomyNode"] = []
    product_count: int = 0


class TaxonomyTree(BaseModel):
    """The complete taxonomy tree."""
    version: str = "2.0.0"
    total_categories: int = len(NecesidadEspecifica)
    total_groups: int = len(NecesidadPrincipal)
    roots: List[TaxonomyNode] = []


def build_taxonomy_tree() -> TaxonomyTree:
    """Build the complete taxonomy tree."""
    roots = []

    for parent in NecesidadPrincipal:
        subcategories = get_all_subcategories(parent.value)
        children = [
            TaxonomyNode(id=sub, name=sub.replace("_", " ").title(), parent_id=parent.value)
            for sub in subcategories
        ]

        root = TaxonomyNode(
            id=parent.value,
            name=parent.value.replace("_", " ").title(),
            children=children
        )
        roots.append(root)

    return TaxonomyTree(roots=roots)


# Human-readable category names (Spanish)
CATEGORY_DISPLAY_NAMES: Dict[str, str] = {
    # Main groups (18)
    "dolor_fiebre": "Dolor y Fiebre",
    "respiratorio": "Respiratorio",
    "digestivo": "Digestivo",
    "piel": "Dermocosmética",
    "bucal": "Salud Bucal",
    "ocular": "Salud Ocular",
    "oidos": "Salud Oídos",
    "capilar": "Capilar",
    "bienestar": "Suplementos",  # Issue #467: Fusionado con nutricion
    "circulatorio": "Circulatorio",
    "urinario": "Sistema Urinario",
    "sexual": "Salud Sexual",
    "infantil": "Infantil",
    "nutricion": "Suplementos",
    "movilidad": "Movilidad",
    "incontinencia_cat": "Incontinencia",
    "diabetes_cat": "Diabetes",
    "botiquin": "Botiquín",
    # Issue #467: Nuevas categorías principales
    "veterinaria_cat": "Veterinaria",
    "servicios_cat": "Servicios",
    "no_salud": "No Salud",

    # Subcategories (selected)
    "dolor_muscular": "Dolor Muscular",
    "dolor_cabeza": "Dolor de Cabeza",
    "proteccion_solar": "Protección Solar",
    # Bucal (Issue #457: subcategorías desglosadas)
    "cepillo_manual": "Cepillos Manuales",
    "cepillo_electrico": "Cepillos Eléctricos",
    "pasta_dental": "Pastas Dentales",
    "enjuague_bucal": "Enjuagues Bucales",
    "seda_dental": "Seda Dental",
    "interdental": "Interdental",
    "irrigador_dental": "Irrigadores Dentales",
    "higiene_bucal_infantil": "Higiene Bucal Infantil",
    "higiene_dental": "Higiene Dental",
    # Heridas (Issue #457: consolidación)
    "desinfeccion_heridas": "Desinfección Heridas",
    "cicatrizacion": "Cicatrización",
    "apositos_curas": "Apósitos y Curas",
    "caida_cabello": "Caída del Cabello",
    "arrugas_antiedad": "Antienvejecimiento",
    "higiene_oidos": "Higiene Oídos",
    "flora_vaginal": "Flora Vaginal",
    "candidiasis_vaginal": "Candidiasis Vaginal",
    "nutricion_clinica": "Nutrición Clínica",
    "ortopedia_pie": "Ortopedia Pie",
    "incontinencia": "Incontinencia",
    "diabetes": "Diabetes",
    "dulceria": "Dulcería",  # Issue #457
    # Tests diagnóstico (Issue #457)
    "test_embarazo": "Test Embarazo",
    "test_ovulacion": "Test Ovulación",
    "test_covid_gripe": "Test COVID/Gripe",
    "test_diagnostico": "Test Diagnóstico",
    # Issue #457: Nuevas categorías
    "bebes_infantil": "Accesorios Bebé",
    "cuidado_dependientes": "Cuidado Dependientes",
    "unas": "Uñas y Manicura",
    "tatuajes": "Cuidado Tatuajes",
    "cuidado_cabello": "Cuidado Cabello",
    # Issue #457: Categorías adicionales de correcciones manuales
    "vitaminas_general": "Vitaminas",
    "vitaminas_infantil": "Vitaminas Infantiles",
    "nutricion_deportiva": "Nutrición Deportiva",
    "infusiones": "Infusiones",
    "melatonina": "Melatonina",
    "mareo_viaje": "Mareo por Viaje",
    "perfumeria": "Perfumería",
    "joyeria": "Joyería",
    "veterinaria": "Veterinaria",
    # Issue #467: Subcategorías de nuevas categorías principales
    "servicios_farmacia": "Servicios Farmacia",
    "regalos_bazar": "Regalos y Bazar",
    "dolor": "Dolor General",
    "alergia": "Alergia",
    "flora_intestinal": "Flora Intestinal",
    "hongos_pies": "Hongos en Pies",
    "circulacion_piernas": "Circulación Piernas",
    "vitamina_d": "Vitamina D",
}


def get_display_name(category: str) -> str:
    """Get human-readable name for a category."""
    return CATEGORY_DISPLAY_NAMES.get(category, category.replace("_", " ").title())


# =============================================================================
# LEVEL 2 SUBCATEGORIES (Issue #505)
# =============================================================================
# Sub-categorías para análisis granular en 3 categorías L1 prioritarias:
# - DERMOCOSMÉTICA (7 L2): Segmentación por comportamiento de compra
# - SUPLEMENTOS (6 L2): Segmentación por necesidad/momento
# - HIGIENE BUCAL (5 L2): Segmentación por tipo de cuidado
#
# Principio: "No clasificamos por lo que el producto ES, sino por CÓMO se vende"
# =============================================================================


class NecesidadL2(str, Enum):
    """
    18 subcategorías Level 2 para análisis granular.

    Segmentadas por comportamiento de compra, no por tipo de producto.
    Solo aplican a: dermocosmetica, suplementos, higiene_bucal
    """

    # === DERMOCOSMÉTICA (7 L2) ===
    # Arquetipo: 🪝 Gancho - Productos de entrada, alta rotación
    HIGIENE_LIMPIEZA = "higiene_limpieza"
    # Arquetipo: 💎 Joyas - Alto margen, ticket elevado
    TRATAMIENTO_AVANZADO = "tratamiento_avanzado"
    # Arquetipo: 🔄 Mantenimiento - Recompra frecuente
    HIDRATACION_BASICA = "hidratacion_basica"
    # Arquetipo: 🎯 Estratégico - Consejo farmacéutico
    SOLAR_FACIAL = "solar_facial"
    # Arquetipo: 📅 Estacional - Picos verano
    SOLAR_CORPORAL = "solar_corporal"
    # Arquetipo: 🔧 Específico - Tratamiento capilar
    CAPILAR_TRATANTE = "capilar_tratante"
    # Arquetipo: 🔒 Fidelización - Venta cruzada acné
    ACNE_PIEL_GRASA = "acne_piel_grasa"

    # === SUPLEMENTOS (6 L2) ===
    # Arquetipo: 🔄 Recurrencia - Compra mensual
    DESCANSO_ESTRES = "descanso_estres"
    # Arquetipo: 📅 Estacional - Picos primavera/otoño
    ENERGIA_VITALIDAD = "energia_vitalidad"
    # Arquetipo: 👴 Cronicidad - Cliente fidelizado
    ARTICULAR_OSEA = "articular_osea"
    # Arquetipo: ❄️ Coyuntural - Picos invierno
    DEFENSAS_INMUNIDAD = "defensas_inmunidad"
    # Arquetipo: 📈 Tendencia - Crecimiento sostenido
    DIGESTIVO_PROBIOTICOS = "digestivo_probioticos"
    # Arquetipo: 🏃 Zafarrancho - Picos enero/mayo
    CONTROL_PESO_L2 = "control_peso_l2"  # Suffix _l2 to avoid conflict with NecesidadEspecifica.CONTROL_PESO

    # === HIGIENE BUCAL (5 L2) ===
    # Arquetipo: 🛒 Commodity - Precio competitivo
    HIGIENE_DIARIA_BASICA = "higiene_diaria_basica"
    # Arquetipo: 🏥 Core Farmacia - Consejo profesional
    SENSIBILIDAD_ENCIAS = "sensibilidad_encias"
    # Arquetipo: ✨ Deseo - Compra impulsiva
    ESTETICA_BLANQUEAMIENTO = "estetica_blanqueamiento"
    # Arquetipo: 💊 Nicho - Especialización
    TRATAMIENTO_BUCAL = "tratamiento_bucal"
    # Arquetipo: ➕ Complementarios - Venta cruzada
    ORTODONCIA_INTERDENTALES = "ortodoncia_interdentales"

    # === SEXUAL (3 L2) - ADR-004: Added Q1 2026 ===
    # Arquetipo: 💰 Alto Margen - Compra discreta
    PRESERVATIVOS_LUBRICANTES = "preservativos_lubricantes"
    # Arquetipo: 🔬 Diagnóstico - Venta consultada
    TEST_FERTILIDAD_EMBARAZO = "test_fertilidad_embarazo"
    # Arquetipo: 🏥 Consejo Farmacéutico - Fidelización
    SALUD_INTIMA_FEMENINA = "salud_intima_femenina"

    # === CONTROL PESO (3 L2) - ADR-004: Added Q1 2026 ===
    # Arquetipo: 🥗 Sustitución - Compra recurrente
    SUSTITUTIVOS_COMIDA = "sustitutivos_comida"
    # Arquetipo: 🔥 Acción Rápida - Picos enero/mayo
    QUEMAGRASAS_DRENANTES = "quemagrasas_drenantes"
    # Arquetipo: 🎯 Complementario - Venta cruzada
    CONTROL_APETITO = "control_apetito"


# Categories that have L2 subcategories (ADR-004: Expanded Q1 2026)
L1_WITH_L2 = {"dermocosmetica", "suplementos", "higiene_bucal", "infantil", "sexual", "control_peso"}


class L1WithL2Category(str, Enum):
    """Enum for L1 categories that have L2 subcategories (Issue #505, ADR-004)."""

    DERMOCOSMETICA = "dermocosmetica"
    SUPLEMENTOS = "suplementos"
    HIGIENE_BUCAL = "higiene_bucal"
    INFANTIL = "infantil"  # ADR-004: Added Q1 2026
    SEXUAL = "sexual"  # ADR-004: Added Q1 2026
    CONTROL_PESO = "control_peso"  # ADR-004: Added Q1 2026


# Mapping from L1 category to its L2 subcategories
L1_TO_L2_CATEGORIES: Dict[str, List[str]] = {
    "dermocosmetica": [
        "higiene_limpieza",
        "tratamiento_avanzado",
        "hidratacion_basica",
        "solar_facial",
        "solar_corporal",
        "capilar_tratante",
        "acne_piel_grasa",
    ],
    "suplementos": [
        "descanso_estres",
        "energia_vitalidad",
        "articular_osea",
        "defensas_inmunidad",
        "digestivo_probioticos",
        # NOTE: control_peso_l2 here is for weight-control SUPPLEMENTS (e.g., Supradyn Diet)
        # Different from L1 "control_peso" which has its own L2s (sustitutivos, quemagrasas)
        "control_peso_l2",
    ],
    "higiene_bucal": [
        "higiene_diaria_basica",
        "sensibilidad_encias",
        "estetica_blanqueamiento",
        "tratamiento_bucal",
        "ortodoncia_interdentales",
    ],
    # ADR-004: Infantil added Q1 2026
    "infantil": [
        "nutricion_infantil",      # Leches, potitos, papillas
        "higiene_bebe",            # Pañales, toallitas, geles
        "cuidados_especificos",    # Cremas pañal, atopia
        "biberoneria_accesorios",  # Biberones, chupetes, extractores
    ],
    # ADR-004: Sexual added Q1 2026
    "sexual": [
        "preservativos_lubricantes",  # Durex, Control, Play, lubricantes
        "test_fertilidad_embarazo",   # Clearblue, Predictor, tests ovulación
        "salud_intima_femenina",      # Vaginesil, Lactacyd íntimo, probióticos vaginales
    ],
    # ADR-004: Control Peso added Q1 2026
    # NOTE: This is L1 "control_peso" category with specialized L2s for diet products.
    # Distinct from "control_peso_l2" in suplementos which is for weight-control supplements.
    "control_peso": [
        "sustitutivos_comida",     # Bimanán, Siken, barritas, batidos
        "quemagrasas_drenantes",   # XLS Medical, drenantes, termogénicos
        "control_apetito",         # Chitosán, saciantes, bloqueadores
    ],
}


# Display names for L2 subcategories (Spanish)
L2_DISPLAY_NAMES: Dict[str, str] = {
    # DERMOCOSMÉTICA
    "higiene_limpieza": "Higiene y Limpieza",
    "tratamiento_avanzado": "Tratamiento Avanzado",
    "hidratacion_basica": "Hidratación Básica",
    "solar_facial": "Solar Facial",
    "solar_corporal": "Solar Corporal",
    "capilar_tratante": "Capilar Tratante",
    "acne_piel_grasa": "Acné y Piel Grasa",
    # SUPLEMENTOS
    "descanso_estres": "Descanso y Estrés",
    "energia_vitalidad": "Energía y Vitalidad",
    "articular_osea": "Articular y Ósea",
    "defensas_inmunidad": "Defensas e Inmunidad",
    "digestivo_probioticos": "Digestivo y Probióticos",
    "control_peso_l2": "Control de Peso",
    # HIGIENE BUCAL
    "higiene_diaria_basica": "Higiene Diaria Básica",
    "sensibilidad_encias": "Sensibilidad y Encías",
    "estetica_blanqueamiento": "Estética y Blanqueamiento",
    "tratamiento_bucal": "Tratamiento Bucal",
    "ortodoncia_interdentales": "Ortodoncia e Interdentales",
    # INFANTIL (ADR-004)
    "nutricion_infantil": "Nutrición Infantil",
    "higiene_bebe": "Higiene Bebé",
    "cuidados_especificos": "Cuidados Específicos",
    "biberoneria_accesorios": "Biberonería y Accesorios",
    # SEXUAL (ADR-004)
    "preservativos_lubricantes": "Preservativos y Lubricantes",
    "test_fertilidad_embarazo": "Test Fertilidad/Embarazo",
    "salud_intima_femenina": "Salud Íntima Femenina",
    # CONTROL PESO (ADR-004)
    "sustitutivos_comida": "Sustitutivos de Comida",
    "quemagrasas_drenantes": "Quemagrasas y Drenantes",
    "control_apetito": "Control de Apetito",
}


# Business archetypes for L2 categories (for dashboards)
L2_ARCHETYPES: Dict[str, str] = {
    # DERMOCOSMÉTICA
    "higiene_limpieza": "🪝 Gancho",
    "tratamiento_avanzado": "💎 Joyas",
    "hidratacion_basica": "🔄 Mantenimiento",
    "solar_facial": "🎯 Estratégico",
    "solar_corporal": "📅 Estacional",
    "capilar_tratante": "🔧 Específico",
    "acne_piel_grasa": "🔒 Fidelización",
    # SUPLEMENTOS
    "descanso_estres": "🔄 Recurrencia",
    "energia_vitalidad": "📅 Estacional",
    "articular_osea": "👴 Cronicidad",
    "defensas_inmunidad": "❄️ Coyuntural",
    "digestivo_probioticos": "📈 Tendencia",
    "control_peso_l2": "🏃 Zafarrancho",
    # HIGIENE BUCAL
    "higiene_diaria_basica": "🛒 Commodity",
    "sensibilidad_encias": "🏥 Core Farmacia",
    "estetica_blanqueamiento": "✨ Deseo",
    "tratamiento_bucal": "💊 Nicho",
    "ortodoncia_interdentales": "➕ Complementarios",
    # INFANTIL (ADR-004)
    "nutricion_infantil": "🍼 Tráfico",
    "higiene_bebe": "🔄 Recurrencia",
    "cuidados_especificos": "🏥 Consejo",
    "biberoneria_accesorios": "➕ Complementarios",
    # SEXUAL (ADR-004)
    "preservativos_lubricantes": "💰 Alto Margen",
    "test_fertilidad_embarazo": "🔬 Diagnóstico",
    "salud_intima_femenina": "🏥 Consejo",
    # CONTROL PESO (ADR-004)
    "sustitutivos_comida": "🥗 Sustitución",
    "quemagrasas_drenantes": "🔥 Acción Rápida",
    "control_apetito": "🎯 Complementario",
}


def get_l2_display_name(l2_category: str) -> str:
    """Get human-readable name for an L2 subcategory."""
    return L2_DISPLAY_NAMES.get(l2_category, l2_category.replace("_", " ").title())


def get_l2_archetype(l2_category: str) -> str:
    """Get business archetype for an L2 subcategory."""
    return L2_ARCHETYPES.get(l2_category, "")


def get_l2_categories_for_l1(l1_category: str) -> List[str]:
    """Get all L2 subcategories for a given L1 category."""
    return L1_TO_L2_CATEGORIES.get(l1_category, [])


def has_l2_subcategories(l1_category: str) -> bool:
    """Check if an L1 category has L2 subcategories."""
    return l1_category in L1_WITH_L2


def get_l1_for_l2(l2_category: str) -> Optional[str]:
    """Get the parent L1 category for an L2 subcategory."""
    for l1, l2_list in L1_TO_L2_CATEGORIES.items():
        if l2_category in l2_list:
            return l1
    return None
