"""
Schemas Pydantic para Análisis de Tendencias y Estacionalidad.

Issue #489: Tab 3 "Tendencias y Estacionalidad" en /prescription.

Estructura:
- Heatmap por día de semana y hora
- Índice de estacionalidad mensual
- (Futuro) Descomposición STL y forecast
"""
from datetime import datetime
from decimal import Decimal
from typing import Any, Dict, List, Optional
from uuid import UUID

from pydantic import BaseModel, Field


# ============================================================================
# HEATMAP DÍA × HORA
# ============================================================================


class HeatmapCell(BaseModel):
    """
    Celda individual del heatmap día × hora.

    Representa ventas/unidades para una combinación específica de
    día de semana (1=Lunes, 7=Domingo) y hora (0-23).
    """

    weekday: int = Field(
        ...,
        ge=1,
        le=7,
        description="Día de la semana (1=Lunes, 7=Domingo)",
    )
    weekday_name: str = Field(
        ...,
        description="Nombre del día en español (Lunes, Martes, ...)",
    )
    hour: int = Field(
        ...,
        ge=0,
        le=23,
        description="Hora del día (0-23)",
    )
    sales: Decimal = Field(
        ...,
        description="Total de ventas (€) en este slot",
    )
    units: int = Field(
        ...,
        description="Total de unidades vendidas en este slot",
    )
    transaction_count: int = Field(
        ...,
        description="Número de transacciones en este slot",
    )

    class Config:
        from_attributes = True
        json_encoders = {Decimal: lambda v: float(v) if v is not None else None}


class HourlyHeatmapResponse(BaseModel):
    """
    Respuesta del endpoint de heatmap día × hora.

    Usado para visualizar patrones de venta por día y hora,
    útil para gestión de turnos y personal.
    """

    heatmap_data: List[HeatmapCell] = Field(
        ...,
        description="Lista de celdas del heatmap (7 días × 24 horas max)",
    )
    summary: Dict[str, Any] = Field(
        default_factory=dict,
        description="Resumen: peak_hour, peak_day, total_sales, avg_per_slot",
    )
    period: Dict[str, str] = Field(
        default_factory=dict,
        description="Período analizado: date_from, date_to",
    )

    class Config:
        from_attributes = True
        json_encoders = {Decimal: lambda v: float(v) if v is not None else None}


# ============================================================================
# ÍNDICE DE ESTACIONALIDAD MENSUAL
# ============================================================================


class MonthlyIndexPoint(BaseModel):
    """
    Punto individual del índice de estacionalidad mensual.

    El índice representa la intensidad relativa de ventas para un mes:
    - 1.0 = promedio anual
    - >1.0 = mes con ventas superiores al promedio
    - <1.0 = mes con ventas inferiores al promedio

    Ejemplo:
        - Enero: 1.4 (antigripales)
        - Julio: 0.6 (menos ventas)
    """

    month: int = Field(
        ...,
        ge=1,
        le=12,
        description="Mes del año (1=Enero, 12=Diciembre)",
    )
    month_name: str = Field(
        ...,
        description="Nombre del mes en español (Enero, Febrero, ...)",
    )
    index: float = Field(
        ...,
        description="Índice de estacionalidad (1.0 = promedio)",
    )
    avg_sales: Decimal = Field(
        ...,
        description="Ventas promedio en este mes (€)",
    )
    avg_units: int = Field(
        ...,
        description="Unidades promedio vendidas en este mes",
    )
    years_included: int = Field(
        ...,
        description="Número de años incluidos en el cálculo",
    )

    class Config:
        from_attributes = True
        json_encoders = {Decimal: lambda v: float(v) if v is not None else None}


class MonthlyIndexResponse(BaseModel):
    """
    Respuesta del endpoint de índice de estacionalidad mensual.

    Proporciona el patrón estacional de ventas a lo largo del año,
    útil para planificación de compras y previsión de demanda.
    """

    monthly_index: List[MonthlyIndexPoint] = Field(
        ...,
        description="Lista de 12 puntos (uno por mes)",
    )
    category: Optional[str] = Field(
        None,
        description="Categoría filtrada (None = todas las categorías)",
    )
    summary: Dict[str, Any] = Field(
        default_factory=dict,
        description="Resumen: peak_month, low_month, avg_annual_sales",
    )
    period: Dict[str, str] = Field(
        default_factory=dict,
        description="Período analizado: date_from, date_to, years_range",
    )
    data_quality: Dict[str, Any] = Field(
        default_factory=dict,
        description="Calidad de datos: months_with_data, min_months_recommended",
    )

    class Config:
        from_attributes = True
        json_encoders = {Decimal: lambda v: float(v) if v is not None else None}


# ============================================================================
# KPIS DE ESTACIONALIDAD (Tab 3 Header)
# ============================================================================


class SeasonalityKPIs(BaseModel):
    """
    KPIs principales para el header del Tab 3.

    Muestra métricas clave de tendencias y estacionalidad.
    """

    trend_yoy: float = Field(
        ...,
        description="Tendencia año sobre año (% cambio)",
    )
    trend_direction: str = Field(
        ...,
        description="Dirección de tendencia: 'up', 'down', 'stable'",
    )
    next_peak_category: Optional[str] = Field(
        None,
        description="Próxima categoría con pico estacional",
    )
    next_peak_months: Optional[str] = Field(
        None,
        description="Meses del próximo pico (ej: 'Nov-Feb')",
    )
    days_until_peak: Optional[int] = Field(
        None,
        description="Días hasta el próximo pico estacional",
    )
    anomalies_count: int = Field(
        default=0,
        description="Número de anomalías detectadas en últimos 30 días",
    )

    class Config:
        from_attributes = True


# ============================================================================
# STL DECOMPOSITION (Issue #498)
# ============================================================================


class DecompositionPoint(BaseModel):
    """
    Punto individual de la descomposición STL.

    Representa los componentes de la serie temporal para una fecha.
    """

    date: str = Field(
        ...,
        description="Fecha en formato ISO (YYYY-MM-DD)",
    )
    observed: float = Field(
        ...,
        description="Valor observado (ventas reales)",
    )
    trend: float = Field(
        ...,
        description="Componente de tendencia (serie suavizada)",
    )
    seasonal: float = Field(
        ...,
        description="Componente estacional (patrón repetitivo)",
    )
    residual: float = Field(
        ...,
        description="Residuo (ruido, anomalías)",
    )


class TrendDecompositionResponse(BaseModel):
    """
    Respuesta del endpoint de descomposición STL.

    STL (Seasonal-Trend decomposition using LOESS) separa la serie temporal en:
    - Tendencia: dirección general de largo plazo
    - Estacionalidad: patrón que se repite cada período
    - Residuo: variación no explicada (ruido o anomalías)
    """

    decomposition: List[DecompositionPoint] = Field(
        ...,
        description="Lista de puntos con componentes STL",
    )
    period: Dict[str, str] = Field(
        default_factory=dict,
        description="Período analizado: date_from, date_to",
    )
    summary: Dict[str, Any] = Field(
        default_factory=dict,
        description="Resumen: trend_direction, seasonal_strength, data_points",
    )
    data_quality: Dict[str, Any] = Field(
        default_factory=dict,
        description="Calidad: sufficient_data, min_periods_required",
    )

    class Config:
        from_attributes = True


# ============================================================================
# FORECAST HOLT-WINTERS (Issue #498)
# ============================================================================


class ForecastPoint(BaseModel):
    """
    Punto individual del forecast.

    Incluye valor predicho e intervalo de confianza.
    """

    date: str = Field(
        ...,
        description="Fecha predicha en formato ISO (YYYY-MM-DD)",
    )
    month_name: str = Field(
        ...,
        description="Nombre del mes en español",
    )
    forecast: float = Field(
        ...,
        description="Valor predicho (€)",
    )
    lower_bound: float = Field(
        ...,
        description="Límite inferior del intervalo de confianza",
    )
    upper_bound: float = Field(
        ...,
        description="Límite superior del intervalo de confianza",
    )


class ForecastResponse(BaseModel):
    """
    Respuesta del endpoint de forecast con Holt-Winters.

    Proporciona predicciones para los próximos N meses con
    intervalo de confianza del 95%.
    """

    forecast: List[ForecastPoint] = Field(
        ...,
        description="Lista de puntos predichos",
    )
    historical: List[Dict[str, Any]] = Field(
        default_factory=list,
        description="Datos históricos recientes para contexto",
    )
    confidence_level: float = Field(
        default=0.95,
        description="Nivel de confianza (0.95 = 95%)",
    )
    periods_ahead: int = Field(
        ...,
        description="Número de períodos predichos",
    )
    model_info: Dict[str, Any] = Field(
        default_factory=dict,
        description="Info del modelo: type, seasonal_periods, trend_type",
    )
    summary: Dict[str, Any] = Field(
        default_factory=dict,
        description="Resumen: total_forecast, avg_monthly, trend_direction",
    )

    class Config:
        from_attributes = True


# ============================================================================
# CATEGORY PATTERNS (Issue #499)
# ============================================================================


class CategoryMonthlyIndex(BaseModel):
    """
    Índice de estacionalidad para un mes específico.
    """

    month: int = Field(
        ...,
        ge=1,
        le=12,
        description="Mes del año (1=Enero, 12=Diciembre)",
    )
    month_name: str = Field(
        ...,
        description="Nombre del mes en español",
    )
    index: float = Field(
        ...,
        description="Índice de estacionalidad (1.0 = promedio)",
    )


class CategoryPattern(BaseModel):
    """
    Patrón de estacionalidad para una categoría de prescripción.

    Incluye índices mensuales y resumen del comportamiento estacional.
    """

    category: str = Field(
        ...,
        description="Nombre interno de la categoría",
    )
    category_display: str = Field(
        ...,
        description="Nombre formateado para mostrar en UI",
    )
    total_sales: float = Field(
        ...,
        description="Ventas totales de la categoría en el período (€)",
    )
    monthly_indices: List[CategoryMonthlyIndex] = Field(
        ...,
        description="Lista de 12 índices mensuales",
    )
    peak_month: Optional[int] = Field(
        None,
        description="Mes con mayor índice (1-12)",
    )
    peak_month_name: str = Field(
        ...,
        description="Nombre del mes pico en español",
    )
    peak_index: float = Field(
        ...,
        description="Valor del índice en el mes pico",
    )
    low_month: Optional[int] = Field(
        None,
        description="Mes con menor índice (1-12)",
    )
    low_month_name: str = Field(
        ...,
        description="Nombre del mes valle en español",
    )
    low_index: float = Field(
        ...,
        description="Valor del índice en el mes valle",
    )
    peak_period: str = Field(
        ...,
        description="Período de pico (ej: 'Nov-Feb')",
    )
    seasonality_strength: float = Field(
        ...,
        description="Fuerza de estacionalidad (peak_index - low_index)",
    )

    class Config:
        from_attributes = True


class CategoryPatternsResponse(BaseModel):
    """
    Respuesta del endpoint de patrones de estacionalidad por categoría.

    Issue #499: Fase 3 - Estacionalidad por Categoría ATC.
    """

    category_patterns: List[CategoryPattern] = Field(
        ...,
        description="Lista de patrones por categoría (ordenados por fuerza estacional)",
    )
    summary: Dict[str, Any] = Field(
        default_factory=dict,
        description="Resumen: categories_analyzed, most_seasonal_category, least_seasonal_category",
    )
    period: Dict[str, str] = Field(
        default_factory=dict,
        description="Período analizado: date_from, date_to",
    )

    class Config:
        from_attributes = True


# ============================================================================
# STOCK-OUT RISK MATRIX (Issue #500)
# ============================================================================


class RiskLevel(str):
    """Niveles de riesgo para matriz de stock-out."""

    CRITICAL = "CRÍTICO"
    HIGH = "ALTO"
    OK = "OK"
    MEDIUM = "MEDIO"
    EXCESS = "EXCESO"


class StockoutRiskItem(BaseModel):
    """
    Item individual de la matriz de riesgo de stock-out.

    Combina datos de inventario con índice estacional para calcular
    el riesgo de rotura de stock ajustado por estacionalidad.
    """

    product_code: Optional[str] = Field(
        None,
        description="Código Nacional del producto",
    )
    product_name: str = Field(
        ...,
        description="Nombre del producto",
    )
    category: Optional[str] = Field(
        None,
        description="Categoría de prescripción",
    )
    current_stock: int = Field(
        ...,
        description="Unidades en stock actual",
    )
    avg_daily_sales: float = Field(
        ...,
        description="Ventas promedio diarias (últimos 30 días)",
    )
    seasonality_index: float = Field(
        ...,
        description="Índice de estacionalidad del mes actual",
    )
    adjusted_daily_sales: float = Field(
        ...,
        description="Ventas diarias ajustadas (avg_daily_sales * seasonality_index)",
    )
    days_of_coverage: float = Field(
        ...,
        description="Días de cobertura con stock actual",
    )
    coverage_ratio: float = Field(
        ...,
        description="Ratio de cobertura vs. reposición (7 días)",
    )
    risk_level: str = Field(
        ...,
        description="Nivel de riesgo: CRÍTICO, ALTO, OK, MEDIO, EXCESO",
    )
    risk_color: str = Field(
        ...,
        description="Color del semáforo: danger, warning, success, info, secondary",
    )
    recommended_order: int = Field(
        0,
        description="Unidades recomendadas a pedir",
    )

    class Config:
        from_attributes = True


class StockoutRiskResponse(BaseModel):
    """
    Respuesta del endpoint de matriz de riesgo de stock-out.

    Issue #500: Fase 3.5 - Cruza estacionalidad con inventario actual.
    """

    risk_items: List[StockoutRiskItem] = Field(
        ...,
        description="Lista de productos con análisis de riesgo",
    )
    summary: Dict[str, Any] = Field(
        default_factory=dict,
        description="Resumen: critical_count, high_count, ok_count, excess_count",
    )
    period: Dict[str, str] = Field(
        default_factory=dict,
        description="Período analizado para ventas promedio",
    )
    inventory_date: Optional[str] = Field(
        None,
        description="Fecha del snapshot de inventario utilizado",
    )

    class Config:
        from_attributes = True


# ============================================================================
# ANOMALY DETECTION (Issue #501)
# ============================================================================


class AnomalyType(str):
    """Tipos de anomalía detectada."""

    SPIKE = "PICO"  # Ventas muy superiores a lo esperado
    DROP = "CAÍDA"  # Ventas muy inferiores a lo esperado


class AnomalyItem(BaseModel):
    """
    Anomalía individual detectada en el patrón de ventas.

    Una anomalía es un día con ventas estadísticamente inusuales
    que no se explica por estacionalidad, festivos o roturas de stock.
    """

    date: str = Field(
        ...,
        description="Fecha de la anomalía en formato ISO (YYYY-MM-DD)",
    )
    weekday_name: str = Field(
        ...,
        description="Nombre del día de la semana",
    )
    anomaly_type: str = Field(
        ...,
        description="Tipo de anomalía: PICO o CAÍDA",
    )
    observed_sales: float = Field(
        ...,
        description="Ventas observadas ese día (€)",
    )
    expected_sales: float = Field(
        ...,
        description="Ventas esperadas según el modelo (€)",
    )
    deviation_pct: float = Field(
        ...,
        description="Desviación porcentual respecto a lo esperado",
    )
    z_score: float = Field(
        ...,
        description="Z-score (desviaciones estándar del promedio)",
    )
    severity: str = Field(
        ...,
        description="Severidad: ALTA (|z|>3), MEDIA (|z|>2.5), BAJA (resto)",
    )
    severity_color: str = Field(
        ...,
        description="Color Bootstrap: danger, warning, info",
    )
    possible_causes: List[str] = Field(
        default_factory=list,
        description="Posibles causas de la anomalía",
    )

    class Config:
        from_attributes = True


class AnomaliesResponse(BaseModel):
    """
    Respuesta del endpoint de detección de anomalías.

    Issue #501: Fase 4 - Detector de Anomalías + Alertas.

    Incluye filtros anti-falsos positivos:
    - Excluye festivos nacionales españoles
    - Excluye días con inventario cero (roturas de stock)
    - Excluye ventas en lote (>30% de venta diaria = institucional)
    """

    anomalies: List[AnomalyItem] = Field(
        ...,
        description="Lista de anomalías detectadas (ordenadas por fecha desc)",
    )
    summary: Dict[str, Any] = Field(
        default_factory=dict,
        description="Resumen: total_anomalies, spikes, drops, high_severity_count",
    )
    period: Dict[str, str] = Field(
        default_factory=dict,
        description="Período analizado: date_from, date_to",
    )
    filters_applied: Dict[str, Any] = Field(
        default_factory=dict,
        description="Filtros aplicados: sensitivity, exclude_holidays, exclude_zero_stock, exclude_bulk_sales",
    )
    data_quality: Dict[str, Any] = Field(
        default_factory=dict,
        description="Calidad: days_analyzed, days_excluded_holidays, days_excluded_stock",
    )

    class Config:
        from_attributes = True


# ============================================================================
# EXPORT CSV (Issue #502)
# ============================================================================


class ExportForecastRow(BaseModel):
    """
    Fila individual del CSV exportable.

    Formato listo para importar en Excel.
    """

    category: str = Field(
        ...,
        description="Categoría de prescripción",
    )
    month: str = Field(
        ...,
        description="Mes en formato texto (Enero, Febrero, ...)",
    )
    month_num: int = Field(
        ...,
        ge=1,
        le=12,
        description="Mes numérico (1-12)",
    )
    forecast: float = Field(
        ...,
        description="Previsión de ventas (€)",
    )
    lower_bound: float = Field(
        ...,
        description="Límite inferior del intervalo de confianza (€)",
    )
    upper_bound: float = Field(
        ...,
        description="Límite superior del intervalo de confianza (€)",
    )
    seasonality_index: float = Field(
        ...,
        description="Índice de estacionalidad del mes (1.0 = promedio)",
    )


class ExportForecastResponse(BaseModel):
    """
    Respuesta del endpoint de exportación de forecast.

    Issue #502: Fase 5 - Exportación CSV + Badges.

    Proporciona datos listos para exportar a CSV o Excel.
    """

    data: List[ExportForecastRow] = Field(
        ...,
        description="Lista de filas exportables (categoría × mes)",
    )
    summary: Dict[str, Any] = Field(
        default_factory=dict,
        description="Resumen: total_categories, total_forecast, period",
    )
    export_info: Dict[str, Any] = Field(
        default_factory=dict,
        description="Info de exportación: generated_at, pharmacy_name, periods",
    )

    class Config:
        from_attributes = True


class SeasonalityBadges(BaseModel):
    """
    Badges para el header del tab de estacionalidad.

    Issue #502: Fase 5 - Badges visuales en tab header.

    Muestra información rápida sobre estado actual.
    """

    anomalies_count: int = Field(
        0,
        description="Número de anomalías en últimos 30 días",
    )
    anomalies_badge_color: str = Field(
        "secondary",
        description="Color del badge de anomalías (danger, warning, secondary)",
    )
    next_peak_label: Optional[str] = Field(
        None,
        description="Etiqueta del próximo pico (ej: 'Antigripales en 45d')",
    )
    next_peak_badge_color: str = Field(
        "info",
        description="Color del badge de próximo pico",
    )
    stockout_alerts: int = Field(
        0,
        description="Productos con riesgo crítico/alto de stockout",
    )
    stockout_badge_color: str = Field(
        "secondary",
        description="Color del badge de alertas de stock",
    )

    class Config:
        from_attributes = True


# ============================================================================
# SEASONALITY INDEX FOR DECISION BOX (Issue #507)
# ============================================================================


class SeasonalityIndexBase(BaseModel):
    """Schema base para indices de estacionalidad (Issue #507)."""

    category: str = Field(..., description="Categoria NECESIDAD L1")
    month: int = Field(..., ge=1, le=12, description="Mes (1-12)")
    index_value: float = Field(
        default=1.0,
        ge=0.1,
        le=3.0,
        description="Indice estacional (1.0 = promedio)",
    )


class SeasonalityIndexCreate(SeasonalityIndexBase):
    """Schema para crear indice estacional."""

    pharmacy_id: Optional[UUID] = Field(
        default=None,
        description="ID farmacia o NULL para default sector",
    )
    source: str = Field(default="manual", description="Origen del indice")


class SeasonalityIndexResponse(SeasonalityIndexBase):
    """Schema de respuesta para indice estacional."""

    id: UUID
    pharmacy_id: Optional[UUID] = None
    source: str
    created_at: Optional[datetime] = None
    updated_at: Optional[datetime] = None

    class Config:
        from_attributes = True


class MomentumSeasonalResult(BaseModel):
    """
    Resultado del calculo de momentum estacional para Decision Box.

    Issue #507: Elimina falsos positivos ajustando por expectativa estacional.
    Ejemplo: Protector solar -45% en octubre = momentum +10% vs esperado.
    """

    momentum_pct: Optional[float] = Field(
        default=None,
        description="Momentum en porcentaje vs expectativa estacional",
    )
    seasonality_index: float = Field(
        default=1.0,
        description="Indice estacional aplicado (1.0 = promedio)",
    )
    context_message: Optional[str] = Field(
        default=None,
        description="Mensaje de contexto para UI (ej: 'Temporada baja (0.5x)')",
    )
    is_low_season: bool = Field(
        default=False,
        description="True si index < 0.8",
    )
    is_high_season: bool = Field(
        default=False,
        description="True si index > 1.2",
    )
    current_sales: Optional[float] = Field(
        default=None,
        description="Ventas periodo actual (ultimos 30 dias)",
    )
    expected_sales: Optional[float] = Field(
        default=None,
        description="Ventas esperadas ajustadas por estacionalidad",
    )

    class Config:
        from_attributes = True
