# backend/app/schemas/roi_tracker.py
"""
Pydantic schemas para ROI Tracker API.

Issue #514: Feedback Loop de Acciones - ROI Tracker.
Schemas para request/response de acciones y resúmenes de ROI.
"""

from datetime import datetime
from typing import Any, Dict, List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field, field_validator


# =============================================================================
# AFFECTED PRODUCT SCHEMA
# =============================================================================

class AffectedProduct(BaseModel):
    """Schema para un producto afectado por una acción."""

    codigo_nacional: str = Field(
        ...,
        min_length=1,
        max_length=20,
        description="Código nacional del producto (SIEMPRE string)",
    )
    nombre: str = Field(..., description="Nombre del producto")
    cantidad: Optional[int] = Field(None, ge=0, description="Cantidad afectada")
    pvp: Optional[float] = Field(None, ge=0, description="PVP unitario en EUR")
    stock_actual: Optional[int] = Field(None, ge=0, description="Stock actual")

    model_config = ConfigDict(
        json_schema_extra={
            "example": {
                "codigo_nacional": "712345",
                "nombre": "IBUPROFENO 600MG 40 COMP",
                "cantidad": 50,
                "pvp": 4.95,
                "stock_actual": 12,
            }
        }
    )


# =============================================================================
# ACTION CREATE/UPDATE SCHEMAS
# =============================================================================

class ActionCreate(BaseModel):
    """Schema para crear una nueva acción."""

    action_type: str = Field(
        ...,
        pattern="^(liquidation|restock|pricing|diversify)$",
        description="Tipo de acción: liquidation, restock, pricing, diversify",
    )
    action_description: str = Field(
        ...,
        min_length=10,
        max_length=1000,
        description="Descripción legible de la acción sugerida",
    )
    affected_products: List[AffectedProduct] = Field(
        default_factory=list,
        description="Lista de productos afectados",
    )
    expected_impact_eur: float = Field(
        ...,
        ge=0,
        description="Impacto económico esperado en EUR",
    )
    expected_impact_description: Optional[str] = Field(
        None,
        max_length=500,
        description="Explicación del cálculo del impacto esperado",
    )
    insight_hash: Optional[str] = Field(
        None,
        max_length=64,
        description="Hash del insight que generó esta acción",
    )

    @field_validator("action_description")
    @classmethod
    def validate_description(cls, v: str) -> str:
        """Limpiar espacios en blanco."""
        return v.strip()

    model_config = ConfigDict(
        json_schema_extra={
            "example": {
                "action_type": "liquidation",
                "action_description": "Liquidar stock de crema solar temporada anterior con 30% descuento",
                "affected_products": [
                    {"codigo_nacional": "712345", "nombre": "CREMA SOLAR VICHY SPF50", "cantidad": 15, "pvp": 18.50}
                ],
                "expected_impact_eur": 185.0,
                "expected_impact_description": "Recuperar 15 uds x 18.50€ x 0.7 = 194.25€ aprox",
                "insight_hash": "abc123def456",
            }
        }
    )


class ActionExecuteRequest(BaseModel):
    """Schema para marcar una acción como ejecutada."""

    executed_at: Optional[datetime] = Field(
        None,
        description="Fecha/hora de ejecución (default: ahora)",
    )
    notes: Optional[str] = Field(
        None,
        max_length=500,
        description="Notas opcionales sobre la ejecución",
    )


class ActionDiscardRequest(BaseModel):
    """Schema para descartar una acción."""

    reason: Optional[str] = Field(
        None,
        max_length=500,
        description="Razón por la que se descarta la acción",
    )


class ActionPostponeRequest(BaseModel):
    """Schema para posponer una acción."""

    postpone_until: datetime = Field(
        ...,
        description="Fecha hasta la cual se pospone la acción",
    )
    notes: Optional[str] = Field(
        None,
        max_length=500,
        description="Notas opcionales",
    )

    @field_validator("postpone_until")
    @classmethod
    def validate_future_date(cls, v: datetime) -> datetime:
        """Ensure postpone date is in the future."""
        from datetime import timezone

        from app.utils.datetime_utils import utc_now

        if v.tzinfo is None:
            v = v.replace(tzinfo=timezone.utc)
        if v <= utc_now():
            raise ValueError("La fecha de postpone debe ser en el futuro")
        return v


# =============================================================================
# ACTION RESPONSE SCHEMAS
# =============================================================================

class ActionResponse(BaseModel):
    """Schema para respuesta de una acción individual."""

    id: UUID
    pharmacy_id: UUID
    action_type: str
    action_description: str
    affected_products: List[Dict[str, Any]]
    expected_impact_eur: float
    expected_impact_description: Optional[str] = None
    status: str
    executed_at: Optional[datetime] = None
    discarded_at: Optional[datetime] = None
    discard_reason: Optional[str] = None
    postponed_until: Optional[datetime] = None
    actual_impact_eur: Optional[float] = None
    actual_impact_description: Optional[str] = None
    impact_calculated_at: Optional[datetime] = None
    insight_hash: Optional[str] = None
    created_at: datetime
    updated_at: Optional[datetime] = None

    # Computed properties
    is_pending: bool = False
    is_executed: bool = False
    has_impact_calculated: bool = False
    impact_delta: Optional[float] = None
    impact_accuracy_pct: Optional[float] = None

    model_config = ConfigDict(from_attributes=True)


class ActionListResponse(BaseModel):
    """Schema para respuesta de lista de acciones."""

    actions: List[ActionResponse] = Field(
        default_factory=list,
        description="Lista de acciones",
    )
    total: int = Field(0, description="Total de acciones (para paginación)")
    pending_count: int = Field(0, description="Acciones pendientes")
    executed_count: int = Field(0, description="Acciones ejecutadas")
    discarded_count: int = Field(0, description="Acciones descartadas")
    postponed_count: int = Field(0, description="Acciones pospuestas")


# =============================================================================
# ROI SUMMARY SCHEMAS
# =============================================================================

class ROISummaryResponse(BaseModel):
    """Schema para respuesta de resumen ROI mensual."""

    id: UUID
    pharmacy_id: UUID
    month: str = Field(..., description="Mes en formato YYYY-MM")
    actions_suggested: int = Field(0, description="Total acciones sugeridas")
    actions_executed: int = Field(0, description="Acciones ejecutadas")
    actions_discarded: int = Field(0, description="Acciones descartadas")
    actions_postponed: int = Field(0, description="Acciones pospuestas")
    execution_rate: float = Field(0.0, description="Tasa de ejecución (%)")
    total_expected_impact: float = Field(0.0, description="Impacto esperado total (EUR)")
    total_actual_impact: float = Field(0.0, description="Impacto real total (EUR)")
    impact_by_type: Dict[str, float] = Field(
        default_factory=dict,
        description="Impacto desglosado por tipo de acción",
    )
    subscription_cost: float = Field(99.0, description="Coste de suscripción mensual")
    roi_percentage: Optional[float] = Field(None, description="ROI % = (actual/cost)*100")
    net_roi: float = Field(0.0, description="ROI neto = actual - cost")
    calculated_at: Optional[datetime] = None

    model_config = ConfigDict(
        from_attributes=True,
        json_schema_extra={
            "example": {
                "id": "123e4567-e89b-12d3-a456-426614174000",
                "pharmacy_id": "123e4567-e89b-12d3-a456-426614174001",
                "month": "2026-01",
                "actions_suggested": 12,
                "actions_executed": 8,
                "actions_discarded": 2,
                "actions_postponed": 2,
                "execution_rate": 66.7,
                "total_expected_impact": 850.0,
                "total_actual_impact": 920.0,
                "impact_by_type": {
                    "liquidation": 420.0,
                    "restock": 380.0,
                    "pricing": 120.0,
                },
                "subscription_cost": 99.0,
                "roi_percentage": 929.3,
                "net_roi": 821.0,
                "calculated_at": "2026-01-31T23:59:59Z",
            }
        },
    )


class ROIHistoryResponse(BaseModel):
    """Schema para respuesta de histórico de ROI."""

    summaries: List[ROISummaryResponse] = Field(
        default_factory=list,
        description="Lista de resúmenes mensuales ordenados por fecha DESC",
    )
    total_months: int = Field(0, description="Total de meses con datos")
    avg_roi_percentage: Optional[float] = Field(
        None,
        description="ROI promedio de todos los meses",
    )
    total_accumulated_impact: float = Field(
        0.0,
        description="Impacto acumulado total (EUR)",
    )
    total_subscription_cost: float = Field(
        0.0,
        description="Coste total de suscripción acumulado",
    )


# =============================================================================
# DASHBOARD WIDGET SCHEMA
# =============================================================================

class ROIDashboardWidget(BaseModel):
    """Schema para el widget de ROI en el dashboard."""

    # Current month summary
    current_month: str = Field(..., description="Mes actual YYYY-MM")
    roi_percentage: Optional[float] = Field(None, description="ROI del mes actual")
    net_roi_eur: float = Field(0.0, description="ROI neto en EUR")
    execution_rate: float = Field(0.0, description="Tasa de ejecución %")

    # Quick stats
    pending_actions: int = Field(0, description="Acciones pendientes de ejecutar")
    total_opportunity_eur: float = Field(
        0.0,
        description="Suma de expected_impact de acciones pendientes",
    )

    # Trend (vs previous month)
    roi_trend: Optional[float] = Field(
        None,
        description="Cambio ROI vs mes anterior (positivo = mejora)",
    )
    trend_direction: Optional[str] = Field(
        None,
        pattern="^(up|down|stable)$",
        description="Dirección del trend: up, down, stable",
    )

    # Top action types this month
    top_action_types: List[Dict[str, Any]] = Field(
        default_factory=list,
        description="Top tipos de acción por impacto [{type, impact, count}]",
    )

    model_config = ConfigDict(
        json_schema_extra={
            "example": {
                "current_month": "2026-01",
                "roi_percentage": 859.0,
                "net_roi_eur": 851.0,
                "execution_rate": 66.7,
                "pending_actions": 4,
                "total_opportunity_eur": 320.0,
                "roi_trend": 12.5,
                "trend_direction": "up",
                "top_action_types": [
                    {"type": "liquidation", "impact": 420.0, "count": 3},
                    {"type": "restock", "impact": 380.0, "count": 4},
                ],
            }
        },
    )
