# backend/app/schemas/reports.py
"""
Pydantic schemas para Sistema de Reportes (Issue #511).

Define estructuras de datos para:
- PDF One-Pager (Dirección): KPIs ejecutivos + alertas
- Excel Pick-Lists (Suelo): Listas operativas de inventario
"""

from datetime import date
from typing import Optional

from pydantic import BaseModel, Field


# =============================================================================
# Schemas comunes
# =============================================================================


class CategoryGrowth(BaseModel):
    """Categoría con métricas de crecimiento YoY."""

    category: str = Field(..., description="Nombre de la categoría NECESIDAD")
    display_name: str = Field(..., description="Nombre para mostrar en UI")
    sales: float = Field(..., ge=0, description="Ventas en período actual (€)")
    yoy_change: float = Field(..., description="Cambio YoY en % (puede ser negativo)")
    product_count: int = Field(..., ge=0, description="Número de SKUs")


class InsightSummary(BaseModel):
    """Resumen de insight para incluir en reportes."""

    rule_code: str = Field(..., description="Código de regla (ej: STOCK_001)")
    severity: str = Field(..., description="high, medium, low")
    title: str = Field(..., description="Título corto")
    economic_value: float = Field(..., description="Valor económico en €")
    action_label: str = Field(..., description="Acción sugerida")


class ProductPickItem(BaseModel):
    """Producto individual para pick-list de Suelo."""

    product_name: str = Field(..., description="Nombre del producto")
    codigo_nacional: str = Field(..., description="Código nacional (7 dígitos)")
    ean: Optional[str] = Field(None, description="EAN-13 si disponible")
    location_hint: Optional[str] = Field("N/A", description="Ubicación sugerida")
    value_eur: float = Field(..., ge=0, description="Valor en € (stock o ventas)")
    quantity: int = Field(..., ge=0, description="Unidades en stock")
    priority: int = Field(..., ge=1, le=3, description="Prioridad: 1=Alta, 2=Media, 3=Baja")
    reason: str = Field(..., description="Razón para incluir en lista")


# =============================================================================
# Schema para PDF Dirección
# =============================================================================


class DireccionReportData(BaseModel):
    """
    Datos para generar PDF One-Pager de Dirección.

    Estructura:
    - Header: Farmacia + período
    - KPIs: Ventas, Margen%, Stock, SKUs
    - YoY: Cambios respecto año anterior
    - Top Categories: 3 mejores/peores categorías
    - Alerts: Top 5 insights ordenados por economic_value
    - Recommendation: Acción principal sugerida
    """

    # Header
    pharmacy_name: str = Field(..., description="Nombre de la farmacia")
    period_start: date = Field(..., description="Inicio del período")
    period_end: date = Field(..., description="Fin del período")
    generated_at: str = Field(..., description="Timestamp de generación")

    # KPIs principales
    total_sales: float = Field(..., ge=0, description="Ventas totales venta libre (€)")
    margin_percent: float = Field(..., description="Margen bruto promedio (%)")
    stock_value: float = Field(..., ge=0, description="Valor del stock actual (€)")
    sku_count: int = Field(..., ge=0, description="Número de SKUs activos")

    # Cambios YoY
    sales_yoy_change: Optional[float] = Field(
        None, description="Cambio ventas YoY (%). None si no hay datos previos"
    )
    margin_yoy_change: Optional[float] = Field(
        None, description="Cambio margen YoY (puntos porcentuales)"
    )
    stock_yoy_change: Optional[float] = Field(
        None, description="Cambio stock YoY (%)"
    )

    # Top categorías (ordenadas por growth)
    top_categories: list[CategoryGrowth] = Field(
        default_factory=list, description="Top 3 categorías por crecimiento"
    )

    # Alertas del InsightEngine
    alerts: list[InsightSummary] = Field(
        default_factory=list, description="Top 5 insights por economic_value"
    )

    # Recomendación principal
    recommendation: str = Field(
        "", description="Texto de recomendación basado en insight principal"
    )

    class Config:
        json_schema_extra = {
            "example": {
                "pharmacy_name": "Farmacia Central",
                "period_start": "2025-12-01",
                "period_end": "2025-12-31",
                "generated_at": "2026-01-05T10:30:00",
                "total_sales": 45200.0,
                "margin_percent": 24.5,
                "stock_value": 32000.0,
                "sku_count": 1247,
                "sales_yoy_change": 4.2,
                "margin_yoy_change": -2.0,
                "stock_yoy_change": 5.1,
                "top_categories": [
                    {
                        "category": "suplementos",
                        "display_name": "Suplementos",
                        "sales": 8500.0,
                        "yoy_change": 23.0,
                        "product_count": 145,
                    }
                ],
                "alerts": [
                    {
                        "rule_code": "STOCK_001",
                        "severity": "high",
                        "title": "Stock crítico: 12 productos",
                        "economic_value": 1500.0,
                        "action_label": "Ver productos",
                    }
                ],
                "recommendation": "Diversificar suplementos de sueño para reducir dependencia de marca",
            }
        }


# =============================================================================
# Schema para Excel Suelo
# =============================================================================


class SueloReportData(BaseModel):
    """
    Datos para generar Excel Pick-Lists de Suelo.

    3 sheets:
    1. Liquidar: Stock muerto (>180 días sin venta)
    2. Reponer: Stock crítico (<7 días cobertura)
    3. Promocionar: Alto margen, baja rotación
    """

    # Metadata
    pharmacy_name: str = Field(..., description="Nombre de la farmacia")
    generated_at: str = Field(..., description="Timestamp de generación")

    # Pick-lists
    liquidation_items: list[ProductPickItem] = Field(
        default_factory=list, description="Productos a liquidar (dead stock)"
    )
    restock_items: list[ProductPickItem] = Field(
        default_factory=list, description="Productos a reponer (stock crítico)"
    )
    promotion_items: list[ProductPickItem] = Field(
        default_factory=list, description="Productos a promocionar"
    )

    # Resúmenes por lista
    liquidation_total_value: float = Field(
        0.0, description="Valor total de stock a liquidar (€)"
    )
    restock_urgency_count: int = Field(
        0, description="Productos con prioridad 1 (urgente)"
    )
    promotion_potential_margin: float = Field(
        0.0, description="Margen potencial si se activan promociones (€)"
    )

    class Config:
        json_schema_extra = {
            "example": {
                "pharmacy_name": "Farmacia Central",
                "generated_at": "2026-01-05T10:30:00",
                "liquidation_items": [
                    {
                        "product_name": "CREMA SOLAR SPF50 2023",
                        "codigo_nacional": "1234567",
                        "ean": "8412345678901",
                        "location_hint": "Lineal 3",
                        "value_eur": 120.0,
                        "quantity": 5,
                        "priority": 1,
                        "reason": "Sin ventas hace 210 días",
                    }
                ],
                "restock_items": [
                    {
                        "product_name": "COLGATE TOTAL 75ML",
                        "codigo_nacional": "7654321",
                        "ean": "8400000000001",
                        "location_hint": "Lineal 1",
                        "value_eur": 0.0,
                        "quantity": 2,
                        "priority": 1,
                        "reason": "Cobertura: 3 días",
                    }
                ],
                "promotion_items": [],
                "liquidation_total_value": 1500.0,
                "restock_urgency_count": 8,
                "promotion_potential_margin": 450.0,
            }
        }
