# backend/app/schemas/product_analysis.py
"""
Schemas Pydantic para Product Analysis API (Issue #494)

Define los modelos de request/response para:
- Búsqueda de productos con autocompletado
- Ficha de producto con KPIs
- Productos alternativos (mismo NECESIDAD, diferente marca)
- Productos complementarios (Market Basket Analysis)
- Análisis competitivo (scatter plot)
- Recomendación algorítmica (Decision Box)
"""

from typing import List, Optional

from pydantic import BaseModel, Field


# ========== PRODUCT SEARCH ==========


class ProductSearchResult(BaseModel):
    """Resultado individual de búsqueda de producto."""

    codigo_nacional: str = Field(
        ...,
        description="Código Nacional del producto (6 dígitos)",
        json_schema_extra={"example": "654321"},
    )
    product_name: str = Field(
        ...,
        description="Nombre del producto",
        json_schema_extra={"example": "ISDIN FOTOPROTECTOR SPF50 50ML"},
    )
    detected_brand: Optional[str] = Field(
        None,
        description="Marca detectada (puede ser None si no se detectó)",
        json_schema_extra={"example": "isdin"},
    )
    ml_category: Optional[str] = Field(
        None,
        description="Categoría NECESIDAD clasificada por ML",
        json_schema_extra={"example": "proteccion_solar"},
    )
    total_sales: float = Field(
        ...,
        description="Ventas totales en euros (período filtrado)",
        json_schema_extra={"example": 2450.50},
    )


class ProductSearchResponse(BaseModel):
    """Respuesta de búsqueda de productos."""

    results: List[ProductSearchResult] = Field(
        ...,
        description="Lista de productos encontrados",
    )
    total_count: int = Field(
        ...,
        description="Total de resultados (puede ser mayor que len(results) por paginación)",
        json_schema_extra={"example": 150},
    )
    query: str = Field(
        ...,
        description="Query de búsqueda utilizada",
        json_schema_extra={"example": "isdin foto"},
    )


# ========== PRODUCT KPIs ==========


class ProductKPIs(BaseModel):
    """KPIs del producto para ficha técnica."""

    mat: float = Field(
        ...,
        description="Moving Annual Total - Ventas últimos 12 meses en euros",
        json_schema_extra={"example": 2450.50},
    )
    mat_yoy_change: Optional[float] = Field(
        None,
        description="Variación MAT vs año anterior en porcentaje",
        json_schema_extra={"example": 8.5},
    )
    gmroi: float = Field(
        ...,
        description="Gross Margin Return on Investment (ratio)",
        json_schema_extra={"example": 2.3},
    )
    gmroi_interpretation: str = Field(
        ...,
        description="Interpretación del GMROI",
        json_schema_extra={"example": "ROI excelente"},
    )
    coverage_days: Optional[float] = Field(
        None,
        description="Días de cobertura con stock actual",
        json_schema_extra={"example": 45.0},
    )
    coverage_alert_level: str = Field(
        ...,
        description="Nivel de alerta de cobertura: danger (<7), warning (<14), success (<30), info (>30)",
        json_schema_extra={"example": "success"},
    )
    margin_vs_category: float = Field(
        ...,
        description="Diferencia de margen vs promedio de categoría en puntos porcentuales",
        json_schema_extra={"example": 3.5},
    )


class ProductFichaResponse(BaseModel):
    """Respuesta de ficha de producto completa."""

    product_code: str = Field(
        ...,
        description="Código Nacional del producto",
    )
    product_name: str = Field(
        ...,
        description="Nombre del producto",
    )
    detected_brand: Optional[str] = Field(
        None,
        description="Marca detectada",
    )
    brand_line: Optional[str] = Field(
        None,
        description="Línea de producto dentro de la marca",
        json_schema_extra={"example": "fotoprotector"},
    )
    ml_category: Optional[str] = Field(
        None,
        description="Categoría NECESIDAD (nivel 1)",
    )
    ml_subcategory: Optional[str] = Field(
        None,
        description="Subcategoría NECESIDAD (nivel 2)",
    )
    kpis: ProductKPIs = Field(
        ...,
        description="KPIs del producto",
    )


# ========== ALTERNATIVES (SWITCHING) ==========


class AlternativeProduct(BaseModel):
    """Producto alternativo (mismo NECESIDAD, diferente marca)."""

    codigo_nacional: str = Field(
        ...,
        description="Código Nacional del producto alternativo",
    )
    product_name: str = Field(
        ...,
        description="Nombre del producto alternativo",
    )
    detected_brand: Optional[str] = Field(
        None,
        description="Marca del producto alternativo",
    )
    total_sales: float = Field(
        ...,
        description="Ventas totales en euros",
        json_schema_extra={"example": 1800.00},
    )
    margin_percent: float = Field(
        ...,
        description="Margen porcentual del producto",
        json_schema_extra={"example": 25.0},
    )
    margin_diff_vs_selected: float = Field(
        ...,
        description="Diferencial de margen vs producto seleccionado (puntos porcentuales)",
        json_schema_extra={"example": 3.0},
    )
    gmroi: Optional[float] = Field(
        None,
        description="GMROI del producto alternativo",
        json_schema_extra={"example": 2.1},
    )


class AlternativesResponse(BaseModel):
    """Respuesta de productos alternativos."""

    alternatives: List[AlternativeProduct] = Field(
        ...,
        description="Lista de productos alternativos ordenados por ventas o margen",
    )
    total_count: int = Field(
        ...,
        description="Total de alternativas encontradas",
    )
    category: str = Field(
        ...,
        description="Categoría NECESIDAD común",
        json_schema_extra={"example": "proteccion_solar"},
    )
    base_product_margin: float = Field(
        ...,
        description="Margen del producto base para comparación",
        json_schema_extra={"example": 22.0},
    )


# ========== COMPLEMENTARY (BASKET ANALYSIS) ==========


class ComplementaryProduct(BaseModel):
    """Producto complementario basado en Market Basket Analysis."""

    codigo_nacional: str = Field(
        ...,
        description="Código Nacional del producto complementario",
    )
    product_name: str = Field(
        ...,
        description="Nombre del producto complementario",
    )
    detected_brand: Optional[str] = Field(
        None,
        description="Marca del producto complementario",
    )
    ml_category: Optional[str] = Field(
        None,
        description="Categoría NECESIDAD del producto complementario",
    )
    support: float = Field(
        ...,
        description="% de cestas que contienen este producto (con el producto base)",
        json_schema_extra={"example": 0.45},
    )
    support_label: str = Field(
        ...,
        description="Etiqueta humana del support",
        json_schema_extra={"example": "45% cestas"},
    )
    confidence: float = Field(
        ...,
        description="P(complementario | base) - Probabilidad de comprar este si compra el base",
        json_schema_extra={"example": 0.67},
    )
    confidence_label: str = Field(
        ...,
        description="Etiqueta humana de confidence",
        json_schema_extra={"example": "67% probabilidad"},
    )
    lift: float = Field(
        ...,
        description="Fuerza de asociación (>1 = asociación real, <1 = asociación negativa)",
        json_schema_extra={"example": 2.3},
    )
    lift_level: str = Field(
        ...,
        description="Nivel de lift: low (<1), medium (1-1.5), high (>1.5)",
        json_schema_extra={"example": "high"},
    )
    co_occurrence_count: int = Field(
        ...,
        description="Número de veces que se compraron juntos",
        json_schema_extra={"example": 45},
    )


class ComplementaryResponse(BaseModel):
    """Respuesta de productos complementarios (MBA)."""

    complementary: List[ComplementaryProduct] = Field(
        ...,
        description="Lista de productos complementarios ordenados por Lift descendente",
    )
    base_product_code: str = Field(
        ...,
        description="Código del producto base analizado",
    )
    base_product_name: str = Field(
        ...,
        description="Nombre del producto base",
    )
    base_product_basket_count: int = Field(
        ...,
        description="Número de cestas que contienen el producto base",
        json_schema_extra={"example": 100},
    )
    analysis_period_days: int = Field(
        ...,
        description="Período de análisis en días",
        json_schema_extra={"example": 90},
    )
    sampling_applied: bool = Field(
        False,
        description="Si se aplicó sampling para performance",
    )
    sample_rate: Optional[float] = Field(
        None,
        description="Tasa de sampling aplicada (si aplica)",
        json_schema_extra={"example": 0.2},
    )


# ========== COMPETITIVE ANALYSIS (SCATTER) ==========


class ScatterDataPoint(BaseModel):
    """Punto de datos para scatter plot competitivo."""

    codigo_nacional: str = Field(
        ...,
        description="Código Nacional del producto",
    )
    product_name: str = Field(
        ...,
        description="Nombre del producto (para tooltip)",
    )
    detected_brand: Optional[str] = Field(
        None,
        description="Marca (para tooltip)",
    )
    x: float = Field(
        ...,
        description="Eje X: Ventas totales en euros",
        json_schema_extra={"example": 2450.50},
    )
    y: float = Field(
        ...,
        description="Eje Y: Margen porcentual",
        json_schema_extra={"example": 25.0},
    )
    size: Optional[float] = Field(
        None,
        description="Tamaño del punto (opcional: stock actual)",
        json_schema_extra={"example": 50.0},
    )
    stock_qty: Optional[int] = Field(
        None,
        description="Stock actual para tooltip",
        json_schema_extra={"example": 50},
    )
    is_selected: bool = Field(
        False,
        description="True si es el producto seleccionado (para destacar)",
    )


class CompetitiveAnalysisResponse(BaseModel):
    """Respuesta de análisis competitivo para scatter plot."""

    data_points: List[ScatterDataPoint] = Field(
        ...,
        description="Puntos de datos para el scatter plot",
    )
    category: str = Field(
        ...,
        description="Categoría NECESIDAD analizada",
    )
    category_avg_sales: float = Field(
        ...,
        description="Ventas promedio de la categoría",
        json_schema_extra={"example": 1500.00},
    )
    category_avg_margin: float = Field(
        ...,
        description="Margen promedio de la categoría",
        json_schema_extra={"example": 22.0},
    )
    total_products_in_category: int = Field(
        ...,
        description="Total de productos en la categoría",
        json_schema_extra={"example": 45},
    )


# ========== RECOMMENDATION (DECISION BOX) ==========


class Recommendation(BaseModel):
    """Recomendación algorítmica para el Decision Box."""

    action: str = Field(
        ...,
        description="Acción recomendada: comprar, mantener, reducir, eliminar",
        json_schema_extra={"example": "reducir"},
    )
    alert_level: str = Field(
        ...,
        description="Nivel de alerta semáforo: success (verde), warning (amarillo), danger (rojo)",
        json_schema_extra={"example": "warning"},
    )
    confidence: float = Field(
        ...,
        description="Confianza en la recomendación (0-100)",
        json_schema_extra={"example": 85.0},
    )
    reasons: List[str] = Field(
        ...,
        description="Lista de razones que justifican la recomendación",
        json_schema_extra={
            "example": [
                "Cobertura de 85 días (sobrestock)",
                "Existe alternativa con 5% más margen",
            ]
        },
    )
    suggested_quantity: Optional[int] = Field(
        None,
        description="Cantidad sugerida para próximo pedido (si aplica)",
        json_schema_extra={"example": 15},
    )


class RecommendationFactors(BaseModel):
    """Factores considerados para la recomendación."""

    coverage_days: Optional[float] = Field(
        None,
        description="Días de cobertura actuales",
    )
    trend_mat: Optional[float] = Field(
        None,
        description="Tendencia MAT YoY en porcentaje",
    )
    margin_vs_category: Optional[float] = Field(
        None,
        description="Diferencial de margen vs categoría",
    )
    sales_volume_rank: Optional[str] = Field(
        None,
        description="Ranking de volumen de ventas: alto, medio, bajo",
    )
    gmroi: Optional[float] = Field(
        None,
        description="GMROI actual",
    )
    has_better_alternatives: bool = Field(
        False,
        description="Si existen alternativas con mejor margen/rotación",
    )


class RecommendationResponse(BaseModel):
    """Respuesta completa del Decision Box."""

    recommendation: Recommendation = Field(
        ...,
        description="Recomendación principal",
    )
    factors: RecommendationFactors = Field(
        ...,
        description="Factores considerados en el análisis",
    )
    product_code: str = Field(
        ...,
        description="Código del producto analizado",
    )
    product_name: str = Field(
        ...,
        description="Nombre del producto",
    )


# ========== FULL ANALYSIS (Batch endpoint - Issue #494 optimization) ==========


class FullProductAnalysisResponse(BaseModel):
    """
    Respuesta completa con todos los datos del producto en una sola llamada.
    
    Combina: ficha, alternativas, complementarios, análisis competitivo y recomendación.
    Reduce 5 API calls a 1, mejorando latencia de ~2-5s a ~0.5-1s.
    """

    ficha: Optional["ProductFichaResponse"] = Field(
        None,
        description="Ficha del producto con KPIs",
    )
    alternatives: Optional["AlternativesResponse"] = Field(
        None,
        description="Productos alternativos (switching)",
    )
    complementary: Optional["ComplementaryResponse"] = Field(
        None,
        description="Productos complementarios (MBA)",
    )
    competitive: Optional["CompetitiveAnalysisResponse"] = Field(
        None,
        description="Análisis competitivo (scatter)",
    )
    recommendation: Optional["RecommendationResponse"] = Field(
        None,
        description="Recomendación algorítmica (Decision Box)",
    )

    model_config = {"from_attributes": True}
