"""
Schemas Pydantic para Dashboard de Análisis de Ventas de Prescripción.

Issue #400 Sprint 1: Backend APIs para análisis terapéutico usando códigos ATC.

Estructura:
- KPIs principales de prescripción
- Time series para evolución temporal (formato long/flat para Plotly)
- Distribución por categorías de prescripción
- Distribución jerárquica por códigos ATC (5 niveles)
- Análisis waterfall para comparaciones YoY/MoM/QoQ
"""
from decimal import Decimal
from typing import Any, Dict, List, Optional
from datetime import date
from pydantic import BaseModel, Field


# ============================================================================
# KPIs PRINCIPALES
# ============================================================================

class PrescriptionKPIs(BaseModel):
    """
    KPIs principales del dashboard de prescripción.

    Incluye métricas agregadas de ventas, unidades y porcentajes clave.
    """
    total_sales: Decimal = Field(
        ...,
        description="Total de ventas de productos de prescripción en el período"
    )
    total_units: int = Field(
        ...,
        description="Total de unidades vendidas de productos de prescripción"
    )
    prescription_percentage: float = Field(
        ...,
        ge=0.0,
        le=1.0,
        description="Fracción del total de ventas de prescripción (0.1974 = 19.74%)"
    )
    avg_ticket: Decimal = Field(
        ...,
        description="Ticket promedio de productos de prescripción"
    )
    atc_coverage: float = Field(
        ...,
        ge=0.0,
        le=1.0,
        description="Fracción de ventas con código ATC asignado (0.1835 = 18.35%)"
    )

    class Config:
        from_attributes = True
        json_encoders = {
            Decimal: lambda v: float(v) if v is not None else None
        }


# ============================================================================
# TIME SERIES (Formato long/flat para Plotly)
# ============================================================================

class TimeSeriesPoint(BaseModel):
    """
    Punto individual en serie temporal (formato long/flat).

    Cada punto representa un mes + categoría específica.
    Formato optimizado para gráficos Plotly (un dataframe largo).

    Ejemplo:
        [
            {"period": "2025-01", "category": "ANTIBIOTICOS", "sales": 5000, "units": 150},
            {"period": "2025-01", "category": "CARDIACOS", "sales": 3000, "units": 100},
            {"period": "2025-02", "category": "ANTIBIOTICOS", "sales": 5200, "units": 155},
            ...
        ]
    """
    period: str = Field(
        ...,
        description="Período en formato YYYY-MM (mes)",
        pattern=r'^\d{4}-(0[1-9]|1[0-2])$'  # Pydantic 2.x usa 'pattern' en lugar de 'regex'
    )
    category: str = Field(
        ...,
        description="Categoría de prescripción (xfarma_prescription_category)"
    )
    sales: Decimal = Field(
        ...,
        description="Total de ventas para esta categoría en este mes"
    )
    units: int = Field(
        ...,
        description="Total de unidades vendidas para esta categoría en este mes"
    )

    class Config:
        from_attributes = True
        json_encoders = {
            Decimal: lambda v: float(v) if v is not None else None
        }


# ============================================================================
# RESUMEN POR CATEGORÍA
# ============================================================================

class CategorySummary(BaseModel):
    """
    Resumen agregado por categoría de prescripción.

    Usado para rankings y distribuciones generales.
    """
    category: str = Field(
        ...,
        description="Categoría de prescripción (xfarma_prescription_category)"
    )
    total_sales: Decimal = Field(
        ...,
        description="Total de ventas acumuladas en el período"
    )
    total_units: int = Field(
        ...,
        description="Total de unidades vendidas en el período"
    )
    percentage: float = Field(
        ...,
        ge=0.0,
        le=100.0,
        description="Porcentaje del total de ventas de prescripción"
    )

    class Config:
        from_attributes = True
        json_encoders = {
            Decimal: lambda v: float(v) if v is not None else None
        }


# ============================================================================
# RESPONSE OVERVIEW
# ============================================================================

class PrescriptionOverviewResponse(BaseModel):
    """
    Response completo del endpoint /overview.

    Combina KPIs, series temporales y resumen por categoría en una sola response.
    """
    kpis: PrescriptionKPIs = Field(
        ...,
        description="KPIs principales del período"
    )
    time_series: List[TimeSeriesPoint] = Field(
        ...,
        description="Serie temporal mensual en formato long/flat (mes × categoría)"
    )
    category_summary: List[CategorySummary] = Field(
        ...,
        description="Resumen agregado por categoría de prescripción"
    )

    class Config:
        from_attributes = True


# ============================================================================
# DISTRIBUCIÓN ATC (Jerarquía 5 niveles)
# ============================================================================

class ATCNode(BaseModel):
    """
    Nodo individual en jerarquía ATC (árbol de 5 niveles).

    Niveles ATC:
    - Nivel 1: Primer carácter (A-V) - Grupo anatómico principal
    - Nivel 2: Primeros 3 caracteres (A02) - Grupo terapéutico
    - Nivel 3: Primeros 4 caracteres (A02B) - Subgrupo farmacológico
    - Nivel 4: Primeros 5 caracteres (A02BC) - Subgrupo químico
    - Nivel 5: Completo 7 caracteres (A02BC01) - Principio activo

    Estructura recursiva para treemap/sunburst charts.
    """
    atc_code: str = Field(
        ...,
        description="Código ATC del nodo (puede ser parcial según nivel)"
    )
    atc_name: str = Field(
        ...,
        description="Nombre descriptivo del grupo/subgrupo/principio activo"
    )
    sales: Decimal = Field(
        ...,
        description="Total de ventas agregadas en este nodo"
    )
    units: int = Field(
        ...,
        description="Total de unidades vendidas en este nodo"
    )
    percentage: float = Field(
        ...,
        ge=0.0,
        le=100.0,
        description="Porcentaje del total de ventas (relativo al nivel superior)"
    )
    level: int = Field(
        ...,
        ge=1,
        le=5,
        description="Nivel jerárquico en árbol ATC (1=anatómico, 5=principio activo)"
    )
    children: Optional[List['ATCNode']] = Field(
        default=None,
        description="Nodos hijos en jerarquía (None si es nodo terminal)"
    )

    class Config:
        from_attributes = True
        json_encoders = {
            Decimal: lambda v: float(v) if v is not None else None
        }


# Actualizar forward reference para estructura recursiva
ATCNode.model_rebuild()


class ATCDistributionResponse(BaseModel):
    """
    Response del endpoint /distribution-by-atc.

    Incluye distribución jerárquica y productos sin código ATC.
    """
    atc_distribution: List[ATCNode] = Field(
        ...,
        description="Distribución jerárquica por códigos ATC (árbol de niveles)"
    )
    uncategorized: Optional[Dict[str, Any]] = Field(
        default=None,
        description="Productos de prescripción sin código ATC (sales, units, percentage)"
    )

    class Config:
        from_attributes = True


# ============================================================================
# ANÁLISIS WATERFALL (YoY/MoM/QoQ)
# ============================================================================

class CategoryChange(BaseModel):
    """
    Cambio en ventas de una categoría entre dos períodos.

    Usado para gráficos waterfall que muestran contribución de cada categoría
    al crecimiento/decrecimiento total.
    """
    category: str = Field(
        ...,
        description="Categoría de prescripción analizada"
    )
    current_sales: Decimal = Field(
        ...,
        description="Ventas en período actual"
    )
    previous_sales: Decimal = Field(
        ...,
        description="Ventas en período anterior (YoY/MoM/QoQ)"
    )
    absolute_change: Decimal = Field(
        ...,
        description="Cambio absoluto (current - previous)"
    )
    percentage_change: float = Field(
        ...,
        description="Cambio porcentual ((current - previous) / previous * 100)"
    )
    contribution_to_growth: float = Field(
        ...,
        description="Contribución al crecimiento total (% del cambio total)"
    )

    class Config:
        from_attributes = True
        json_encoders = {
            Decimal: lambda v: float(v) if v is not None else None
        }


class WaterfallAnalysisResponse(BaseModel):
    """
    Response del endpoint /waterfall-analysis.

    Análisis de crecimiento/decrecimiento por categoría con comparación temporal.
    """
    comparison: Dict[str, Any] = Field(
        ...,
        description="Metadata de comparación (type: yoy/mom/qoq, current_period, previous_period)"
    )
    waterfall_data: List[CategoryChange] = Field(
        ...,
        description="Cambios por categoría ordenados por contribución"
    )
    summary: Dict[str, Decimal] = Field(
        ...,
        description="Resumen agregado (total_current, total_previous, total_growth)"
    )

    class Config:
        from_attributes = True
        json_encoders = {
            Decimal: lambda v: float(v) if v is not None else None
        }


# ============================================================================
# TOP CONTRIBUTORS (Issue #436 Fase 2)
# ============================================================================

class TopContributor(BaseModel):
    """
    Elemento que contribuye al cambio en ventas de prescripcion.

    Issue #449: Agrupacion adaptativa jerarquica:
    - homogeneous_group: Nombre del grupo homogeneo (si existe)
    - active_ingredient: Principio activo (si no hay grupo homogeneo)
    - product: Nombre del producto (para categorias sin ATC)

    Usado para la tabla de Top Contributors en el dashboard de prescripcion.
    Muestra los elementos que mas impactan en las variaciones de ventas.
    """
    active_ingredient: str = Field(
        ...,
        description="Nombre del grupo de analisis (grupo homogeneo, principio activo o producto)"
    )
    category: str = Field(
        ...,
        description="Categoria de prescripcion (xfarma_prescription_category)"
    )
    grouping_type: str = Field(
        default="active_ingredient",
        description="Tipo de agrupacion usada: 'homogeneous_group', 'active_ingredient' o 'product'"
    )
    current_sales: Decimal = Field(
        ...,
        description="Ventas en el periodo actual"
    )
    previous_sales: Decimal = Field(
        ...,
        description="Ventas en el periodo anterior (YoY/MoM/QoQ)"
    )
    variation_euros: Decimal = Field(
        ...,
        description="Variacion en euros (current - previous)"
    )
    variation_percent: float = Field(
        ...,
        description="Variacion porcentual ((current - previous) / previous * 100)"
    )
    impact_percent: float = Field(
        ...,
        description="Contribucion al cambio total (% del cambio total de prescripcion)"
    )
    # Issue #457: Añadir unidades para productos individuales
    current_units: Optional[int] = Field(
        default=None,
        description="Unidades vendidas en el periodo actual (solo para grouping_type='product')"
    )
    previous_units: Optional[int] = Field(
        default=None,
        description="Unidades vendidas en el periodo anterior (solo para grouping_type='product')"
    )
    variation_units: Optional[int] = Field(
        default=None,
        description="Variacion en unidades (current_units - previous_units)"
    )

    class Config:
        from_attributes = True
        json_encoders = {
            Decimal: lambda v: float(v) if v is not None else None
        }


class TopContributorsResponse(BaseModel):
    """
    Response del endpoint /top-contributors.

    Lista de principios activos ordenados por impacto en el cambio de ventas.
    """
    contributors: List[TopContributor] = Field(
        ...,
        description="Lista de top contributors ordenados por impacto absoluto"
    )
    total_change: Decimal = Field(
        ...,
        description="Cambio total en ventas de prescripcion (periodo actual - anterior)"
    )
    comparison_period: str = Field(
        ...,
        description="Tipo de comparacion: yoy, mom, qoq"
    )
    filters_applied: Dict[str, Any] = Field(
        ...,
        description="Filtros aplicados (limit, direction, date_range)"
    )

    class Config:
        from_attributes = True
        json_encoders = {
            Decimal: lambda v: float(v) if v is not None else None
        }


# ============================================================================
# VALIDATORS
# ============================================================================
# Nota: La validación de formato YYYY-MM ya está implementada en el Field
# con pattern=r'^\d{4}-(0[1-9]|1[0-2])$' en TimeSeriesPoint.period
