"""
Schemas Pydantic para endpoints de clasificación de prescripción.

Issue #16 Fase 2: APIs admin para clasificación masiva de productos.
"""
from typing import Any, Dict, List, Optional
from datetime import datetime
from pydantic import BaseModel, Field
from app.models.enums import PrescriptionCategory


# ============================================================================
# REQUEST SCHEMAS
# ============================================================================

class ClassifyProductRequest(BaseModel):
    """Request para clasificar un producto individual"""
    product_id: str = Field(..., description="UUID del producto en product_catalog")
    force_reclassify: bool = Field(
        default=False,
        description="Si True, reclasifica aunque ya tenga categoría"
    )


class BulkClassifyRequest(BaseModel):
    """Request para clasificación batch de productos"""
    filters: Optional[Dict[str, Any]] = Field(
        default=None,
        description="Filtros opcionales para seleccionar productos"
    )
    dry_run: bool = Field(
        default=True,
        description="Si True, solo retorna estadísticas sin persistir cambios"
    )
    limit: Optional[int] = Field(
        default=None,
        ge=1,
        le=50000,
        description="Límite de productos a clasificar (max 50,000)"
    )


# ============================================================================
# RESPONSE SCHEMAS
# ============================================================================

class ClassificationResult(BaseModel):
    """Resultado de clasificación de un producto individual"""
    product_id: str
    national_code: str
    product_name: Optional[str] = None
    category: Optional[PrescriptionCategory] = None
    is_prescription: bool
    classified_at: datetime

    class Config:
        from_attributes = True


class CategoryBreakdown(BaseModel):
    """Desglose de productos por categoría"""
    category: PrescriptionCategory
    count: int
    percentage: float = Field(..., description="Porcentaje del total clasificado")


class BulkClassifyResponse(BaseModel):
    """Respuesta de clasificación batch"""
    total_products: int = Field(..., description="Total de productos evaluados")
    classified_count: int = Field(..., description="Productos clasificados")
    skipped_otc_count: int = Field(..., description="Productos OTC/parafarmacia (no de prescripción)")
    already_classified_count: int = Field(
        default=0,
        description="Productos que ya tenían categoría"
    )

    category_breakdown: List[CategoryBreakdown] = Field(
        default_factory=list,
        description="Distribución de productos por categoría"
    )

    dry_run: bool = Field(..., description="Si True, cambios NO fueron persistidos")
    execution_time_seconds: float = Field(
        ...,
        description="Tiempo de ejecución de la clasificación"
    )


class ClassificationStats(BaseModel):
    """Estadísticas globales de clasificación del catálogo"""
    total_products: int = Field(..., description="Total de productos en catálogo")

    # Prescripción vs OTC
    prescription_products: int = Field(
        ...,
        description="Productos de prescripción (nomenclator o requieren receta)"
    )
    otc_parapharmacy_products: int = Field(
        ...,
        description="Productos OTC/parafarmacia"
    )

    # Estado de clasificación
    classified_count: int = Field(
        ...,
        description="Productos de prescripción con categoría asignada"
    )
    unclassified_count: int = Field(
        ...,
        description="Productos de prescripción sin categoría"
    )
    classification_rate: float = Field(
        ...,
        ge=0.0,
        le=100.0,
        description="Porcentaje de productos de prescripción clasificados"
    )

    # Distribución por categoría
    category_distribution: List[CategoryBreakdown] = Field(
        default_factory=list,
        description="Distribución de productos clasificados por categoría"
    )

    # Metadata
    last_classification_at: Optional[datetime] = Field(
        default=None,
        description="Fecha de última clasificación (si existe)"
    )


class TopUnclassifiedProduct(BaseModel):
    """Producto sin clasificar con métricas de ventas"""
    national_code: str
    product_name: str
    suggested_category: Optional[PrescriptionCategory] = None
    total_sales_last_month: float = Field(
        default=0.0,
        description="Ventas totales del último mes"
    )


class UnclassifiedSummary(BaseModel):
    """Resumen de productos sin clasificar"""
    with_nomenclator: int = Field(
        ...,
        description="Productos con código homogéneo pero sin categoría"
    )
    only_cima: int = Field(
        ...,
        description="Productos solo con requiere_receta en CIMA"
    )
    top_unclassified_by_sales: List[TopUnclassifiedProduct] = Field(
        default_factory=list,
        description="Top 10 productos sin clasificar por ventas"
    )


class DetailedClassificationStats(ClassificationStats):
    """Estadísticas detalladas con productos sin clasificar prioritarios"""
    unclassified_summary: UnclassifiedSummary


# ============================================================================
# ERROR SCHEMAS
# ============================================================================

class ClassificationError(BaseModel):
    """Error durante clasificación"""
    product_id: Optional[str] = None
    national_code: Optional[str] = None
    error_type: str
    error_message: str
    timestamp: datetime = Field(default_factory=datetime.utcnow)


# ============================================================================
# ADMIN TAB SCHEMAS (Issue #408)
# ============================================================================

class ProductWithCategory(BaseModel):
    """
    Producto con categoría de prescripción para admin tab.

    Issue #408: Schema para endpoint GET /api/v1/admin/prescription/products
    """
    national_code: str = Field(..., description="Código nacional del producto")
    product_name: Optional[str] = Field(None, description="Nombre del producto")
    category: Optional[PrescriptionCategory] = Field(
        None,
        description="Categoría de prescripción asignada (None si no clasificado)"
    )
    laboratory: Optional[str] = Field(None, description="Laboratorio fabricante")
    suggested_category: Optional[PrescriptionCategory] = Field(
        None,
        description="Categoría sugerida por clasificación automática"
    )

    class Config:
        from_attributes = True


# ============================================================================
# BULK UPDATE SCHEMAS (Issue #408)
# ============================================================================

class ClassificationItem(BaseModel):
    """Item individual para actualización de clasificación."""
    codigo_nacional: str = Field(..., description="Código nacional del producto")
    categoria: str = Field(..., description="Categoría de prescripción a asignar")


class BulkUpdateRequest(BaseModel):
    """Request para actualización masiva de clasificaciones desde Excel."""
    classifications: List[ClassificationItem] = Field(
        ...,
        description="Lista de clasificaciones a actualizar",
        min_length=1,
        max_length=10000
    )


class BulkUpdateResponse(BaseModel):
    """Respuesta de actualización masiva de clasificaciones."""
    updated_count: int = Field(..., description="Número de productos actualizados")
    failed_count: int = Field(..., description="Número de productos con error")
    errors: List[str] = Field(
        default_factory=list,
        description="Lista de errores encontrados"
    )


# ============================================================================
# REFERENCE LIST UPLOAD SCHEMAS (Issue #445)
# ============================================================================

class ReferenceListUploadResponse(BaseModel):
    """
    Respuesta de carga de listados de referencia de prescripcion.

    Issue #445: Endpoint para cargar Excel oficial via API (no script manual).
    """
    total_loaded: int = Field(..., description="Total de productos cargados")
    dietas_count: int = Field(default=0, description="Productos DIETOTERAPICOS")
    tiras_count: int = Field(default=0, description="Productos TIRAS_REACTIVAS_GLUCOSA")
    incontinencia_count: int = Field(default=0, description="Productos INCONTINENCIA_FINANCIADA")
    ortopedia_count: int = Field(default=0, description="Productos ORTOPEDIA_FINANCIADA")
    efectos_count: int = Field(default=0, description="Productos EFECTOS_FINANCIADOS")
    special_codes_count: int = Field(default=2, description="Codigos especiales (formulas, vacunas)")
    catalog_entries_created: int = Field(
        default=0,
        description="Entradas creadas en ProductCatalog para joins con SalesEnrichment"
    )
    truncated_before: bool = Field(
        default=False,
        description="Si se trunco la tabla antes de cargar"
    )
    previous_count: int = Field(
        default=0,
        description="Registros existentes antes de la carga"
    )
    execution_time_seconds: float = Field(..., description="Tiempo de ejecucion")
    warnings: List[str] = Field(
        default_factory=list,
        description="Advertencias durante la carga"
    )


class ReferenceListStats(BaseModel):
    """
    Estadisticas de listados de referencia cargados.

    Issue #445: Endpoint para monitorear estado de listados.
    """
    total_entries: int = Field(..., description="Total de entradas en prescription_reference_list")
    category_breakdown: Dict[str, int] = Field(
        ...,
        description="Desglose por categoria"
    )
    source_breakdown: Dict[str, int] = Field(
        ...,
        description="Desglose por reference_source (hoja del Excel)"
    )
    last_loaded_at: Optional[datetime] = Field(
        None,
        description="Fecha de ultima carga"
    )
    catalog_coverage: int = Field(
        default=0,
        description="Productos de reference_list que existen en ProductCatalog"
    )
