# backend/app/schemas/pharmacy_targets.py
"""
Pydantic schemas para gestión de targets/benchmarks por farmacia.

Issue #510: Benchmarks y Líneas de Referencia en Gráficos
"""

import re
from datetime import datetime
from decimal import Decimal
from enum import Enum
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, Field, field_validator

# Patrón de color hex compilado a nivel de módulo para mejor rendimiento
HEX_COLOR_PATTERN = re.compile(r"^#[0-9A-Fa-f]{6}$")


class TargetType(str, Enum):
    """Tipos de target disponibles."""

    MARGIN_MIN = "margin_min"
    MARGIN_TARGET = "margin_target"
    MARGIN_EXCELLENT = "margin_excellent"
    HHI_MAX = "hhi_max"
    SALES_TARGET = "sales_target"
    PRICE_AVG = "price_avg"


class LineStyle(str, Enum):
    """Estilos de línea para gráficos Plotly."""

    SOLID = "solid"
    DASH = "dash"
    DOT = "dot"


class TargetBase(BaseModel):
    """Base schema para target."""

    target_type: TargetType = Field(..., description="Tipo de target")
    category: Optional[str] = Field(
        None,
        max_length=100,
        description="Categoría NECESIDAD específica o null para global",
    )
    value: Decimal = Field(..., ge=0, description="Valor numérico del target")
    color: str = Field(
        default="#28a745",
        max_length=20,
        description="Color hex de la línea (#RRGGBB)",
    )
    line_style: LineStyle = Field(
        default=LineStyle.DASH,
        description="Estilo de línea: solid, dash, dot",
    )
    label: Optional[str] = Field(
        None,
        max_length=100,
        description="Etiqueta mostrada junto a la línea",
    )

    @field_validator("color")
    @classmethod
    def validate_hex_color(cls, v: str) -> str:
        """Valida que el color sea un código hex válido."""
        if not HEX_COLOR_PATTERN.match(v):
            raise ValueError("Color debe ser formato hex (#RRGGBB)")
        return v.upper()


class TargetCreate(TargetBase):
    """Schema para crear un target."""

    pass


class TargetUpdate(BaseModel):
    """Schema para actualizar un target (todos los campos opcionales)."""

    value: Optional[Decimal] = Field(None, ge=0)
    color: Optional[str] = Field(None, max_length=20)
    line_style: Optional[LineStyle] = None
    label: Optional[str] = Field(None, max_length=100)

    @field_validator("color")
    @classmethod
    def validate_hex_color(cls, v: Optional[str]) -> Optional[str]:
        """Valida que el color sea un código hex válido."""
        if v is None:
            return v
        if not HEX_COLOR_PATTERN.match(v):
            raise ValueError("Color debe ser formato hex (#RRGGBB)")
        return v.upper()


class TargetResponse(TargetBase):
    """Schema de respuesta para un target."""

    id: UUID
    pharmacy_id: UUID
    user_configured: bool = False
    created_at: Optional[datetime] = None
    updated_at: Optional[datetime] = None

    class Config:
        from_attributes = True


class TargetChartFormat(BaseModel):
    """Formato optimizado para uso en Plotly charts."""

    value: float
    color: str
    line_style: str
    label: str


class BulkTargetCreate(BaseModel):
    """Schema para crear múltiples targets a la vez."""

    targets: List[TargetCreate] = Field(..., min_length=1, max_length=20)


class TargetListResponse(BaseModel):
    """Respuesta con lista de targets."""

    success: bool = True
    pharmacy_id: UUID
    targets: List[TargetResponse]
    count: int


class InitDefaultsResponse(BaseModel):
    """Respuesta al inicializar defaults."""

    success: bool = True
    message: str
    created_count: int
    targets: List[TargetResponse]


class DeleteResponse(BaseModel):
    """Respuesta al eliminar target."""

    success: bool = True
    message: str
    deleted_type: str
    deleted_category: Optional[str] = None
