# backend/app/schemas/partner_analysis.py
"""
Schemas Pydantic para Partner Analysis API

Define los modelos de request/response para el nuevo sistema de análisis de partners
siguiendo el diseño del Panel Partners Redesign con separación contexto/análisis dinámico
"""

from typing import List, Optional

from pydantic import BaseModel, Field, field_validator

# ========== REQUEST SCHEMAS ==========


class PartnerDynamicAnalysisRequest(BaseModel):
    """Request para análisis dinámico con partners seleccionados"""

    selected_partner_codes: List[str] = Field(
        ...,
        description="Lista de códigos de laboratorios partners (nomenclator oficial)",
        json_schema_extra={"example": ["111", "426", "1079"]},
    )
    # Issue #xxx: Fechas directas en lugar de period_months para consistencia con treemap
    start_date: Optional[str] = Field(
        None,
        description="Fecha inicio análisis (YYYY-MM-DD). Si no se proporciona, usa period_months",
        json_schema_extra={"example": "2024-01-01"},
    )
    end_date: Optional[str] = Field(
        None,
        description="Fecha fin análisis (YYYY-MM-DD). Si no se proporciona, usa period_months",
        json_schema_extra={"example": "2024-12-31"},
    )
    period_months: Optional[int] = Field(
        None,
        ge=1,
        le=24,
        description="DEPRECADO: Usar start_date/end_date. Período en meses desde hoy (fallback)",
        json_schema_extra={"example": 12},
    )
    discount_percentage: Optional[float] = Field(
        None,
        ge=0.0,
        le=100.0,
        description="Porcentaje de descuento esperado con partners (0-100, default: 10%)",
        json_schema_extra={"example": 10.0},
    )
    selected_employee_names: Optional[List[str]] = Field(
        None,
        description="Lista de nombres de empleados a filtrar (None o vacío = TODOS)",
        json_schema_extra={"example": ["EMPLEADO1", "EMPLEADO2", "__sin_empleado__"]},
    )

    @field_validator("selected_partner_codes")
    def validate_partner_codes_not_empty(cls, v):
        # ✅ Permitir lista vacía para análisis "100% no-partners"
        # Representa el caso válido de 0% partners seleccionados
        if v is None:
            raise ValueError("selected_partner_codes no puede ser None")

        # Validar que sea lista (puede estar vacía)
        if not isinstance(v, list):
            raise ValueError("selected_partner_codes debe ser una lista")

        # Si está vacía, es válido (0% partners seleccionados)
        if not v:
            return v  # [] es válido para análisis sin partners

        # Si no está vacía, validar cada código
        cleaned_codes = []
        for code in v:
            code_clean = str(code).strip()
            if not code_clean:
                continue

            # Validar que sea numérico (permite decimales como "111.0")
            try:
                float(code_clean)
            except ValueError:
                raise ValueError(f"Código de laboratorio inválido: '{code}'. Debe ser numérico.")

            cleaned_codes.append(code_clean)

        # Si había códigos pero todos eran vacíos/inválidos
        if v and not cleaned_codes:
            raise ValueError("Todos los códigos proporcionados están vacíos o son inválidos")

        return cleaned_codes


class TemporalBreakdownRequest(BaseModel):
    """Request para drill-down temporal"""

    partners: List[str] = Field(
        ...,
        description="Partners seleccionados para análisis temporal",
        json_schema_extra={"example": ["CINFA", "NORMON S.A."]},
    )
    level: str = Field(
        "quarter",
        description="Nivel de agrupación temporal",
        json_schema_extra={"example": "quarter"},
    )
    period_months: Optional[int] = Field(
        None,
        ge=1,
        le=24,
        description="Período de análisis en meses",
        json_schema_extra={"example": 12},
    )

    @field_validator("partners")
    def validate_partners_list(cls, v):
        if not v:
            raise ValueError("La lista de partners no puede estar vacía")

        # Validar cada partner
        cleaned_partners = []
        for partner in v:
            partner_clean = partner.strip()
            if not partner_clean:
                continue

            # Validar caracteres permitidos
            if not all(c.isalnum() or c.isspace() or c in ".-" for c in partner_clean):
                raise ValueError(
                    f"Nombre de laboratorio inválido: '{partner}'. Solo se permiten letras, números, espacios, puntos y guiones."
                )

            # Validar longitud
            if len(partner_clean) > 100:
                raise ValueError(f"Nombre de laboratorio muy largo: '{partner}'. Máximo 100 caracteres.")

            cleaned_partners.append(partner_clean)

        if not cleaned_partners:
            raise ValueError("Todos los partners proporcionados están vacíos o son inválidos")

        return cleaned_partners

    @field_validator("level")
    def validate_temporal_level(cls, v):
        valid_levels = ["quarter", "month", "fortnight"]
        if v not in valid_levels:
            raise ValueError(f"Nivel temporal debe ser uno de: {valid_levels}")
        return v


# ========== RESPONSE SCHEMAS ==========


class AnalysisPeriod(BaseModel):
    """Período de análisis común"""

    start_date: str = Field(..., description="Fecha inicio análisis (ISO)")
    end_date: str = Field(..., description="Fecha fin análisis (ISO)")
    months: int = Field(..., description="Meses de análisis", json_schema_extra={"example": 12})


class UniverseSummary(BaseModel):
    """Resumen del universo sustituible"""

    total_substitutable_groups: int = Field(
        ..., description="Total conjuntos homogéneos sustituibles", json_schema_extra={"example": 156}
    )
    total_substitutable_units: int = Field(
        ..., description="Total unidades vendidas en universo", json_schema_extra={"example": 8420}
    )
    total_substitutable_revenue: float = Field(
        ..., description="Ingresos totales universo sustituible (€)", json_schema_extra={"example": 25673.45}
    )
    total_substitutable_transactions: int = Field(
        ..., description="Total transacciones en universo", json_schema_extra={"example": 1234}
    )
    total_pharmacy_revenue: float = Field(
        ..., description="Ingresos totales REALES de la farmacia (€) - Issue #330", json_schema_extra={"example": 86914.24}
    )
    earliest_sale: Optional[str] = Field(None, description="Primera venta registrada (ISO)")
    latest_sale: Optional[str] = Field(None, description="Última venta registrada (ISO)")


class HomogeneousGroupSummary(BaseModel):
    """Resumen de conjunto homogéneo"""

    homogeneous_code: str = Field(..., description="Código conjunto homogéneo")
    homogeneous_name: Optional[str] = Field(None, description="Nombre conjunto")
    total_units: int = Field(..., description="Unidades totales vendidas")
    total_revenue: float = Field(..., description="Ingresos totales (€)")
    total_transactions: Optional[int] = Field(None, description="Total transacciones")


class LaboratoryInUniverse(BaseModel):
    """Laboratorio en universo sustituible"""

    laboratory_name: str = Field(..., description="Nombre laboratorio")
    homogeneous_groups_count: int = Field(..., description="Conjuntos homogéneos donde participa")
    total_units: int = Field(..., description="Unidades totales vendidas")
    total_revenue: float = Field(..., description="Ingresos totales (€)")


class SubstitutableUniverseResponse(BaseModel):
    """Response para contexto universo sustituible"""

    pharmacy_id: str = Field(..., description="ID farmacia")
    analysis_period: AnalysisPeriod
    universe_summary: UniverseSummary
    homogeneous_groups: List[HomogeneousGroupSummary]
    laboratories_in_universe: List[LaboratoryInUniverse]
    context_type: str = Field(default="substitutable_universe_fixed", description="Tipo de contexto")
    calculated_at: str = Field(..., description="Timestamp cálculo (ISO)")


class AnalyzableUniverse(BaseModel):
    """Universo analizable con partners seleccionados"""

    groups_count: int = Field(
        ..., description="Conjuntos analizables (con partners disponibles)", json_schema_extra={"example": 89}
    )
    total_units: int = Field(..., description="Unidades totales analizables", json_schema_extra={"example": 5240})
    total_revenue: float = Field(
        ..., description="Ingresos totales analizables (€)", json_schema_extra={"example": 15890.23}
    )
    total_transactions: int = Field(..., description="Transacciones totales", json_schema_extra={"example": 890})


class PartnerPerformance(BaseModel):
    """Performance de partners seleccionados"""

    partner_units: int = Field(..., description="Unidades vendidas de partners", json_schema_extra={"example": 1420})
    partner_revenue: float = Field(..., description="Ingresos de partners (€)", json_schema_extra={"example": 4567.89})
    partner_transactions: int = Field(..., description="Transacciones partners", json_schema_extra={"example": 234})
    penetration_percentage: float = Field(
        ..., description="Penetración partners (%)", json_schema_extra={"example": 27.1}
    )


class OpportunityMetrics(BaseModel):
    """Métricas de oportunidad (ventas NO-partners)"""

    opportunity_units: int = Field(
        ..., description="Unidades oportunidad (NO-partners)", json_schema_extra={"example": 3820}
    )
    opportunity_revenue: float = Field(
        ..., description="Ingresos oportunidad (€)", json_schema_extra={"example": 11322.34}
    )
    opportunity_transactions: int = Field(
        ..., description="Transacciones oportunidad", json_schema_extra={"example": 656}
    )
    potential_savings_base: float = Field(
        ..., description="Base ahorro potencial (PVL) (€)", json_schema_extra={"example": 9876.54}
    )
    estimated_savings_with_discount: float = Field(
        ..., description="Ahorro estimado aplicando descuento (€)", json_schema_extra={"example": 987.65}
    )
    applied_discount_percentage: float = Field(
        ..., description="Porcentaje de descuento aplicado (%)", json_schema_extra={"example": 10.0}
    )


class HomogeneousGroupDetail(BaseModel):
    """Detalle conjunto homogéneo en análisis dinámico"""

    homogeneous_code: str = Field(..., description="Código conjunto")
    homogeneous_name: Optional[str] = Field(None, description="Nombre conjunto")
    total_units: int = Field(..., description="Unidades totales")
    total_revenue: float = Field(..., description="Ingresos totales (€)")
    partner_units: int = Field(..., description="Unidades partners")
    opportunity_units: int = Field(..., description="Unidades oportunidad")
    savings_base: float = Field(..., description="Base ahorro (€)")
    partner_penetration: float = Field(..., description="Penetración partners (%)")


class PartnerDynamicAnalysisResponse(BaseModel):
    """Response para análisis dinámico con partners"""

    pharmacy_id: str = Field(..., description="ID farmacia")
    selected_partner_codes: List[str] = Field(..., description="Códigos partners seleccionados")
    analysis_period: AnalysisPeriod
    analyzable_universe: AnalyzableUniverse
    partner_performance: PartnerPerformance
    opportunity_metrics: OpportunityMetrics
    homogeneous_groups_detail: List[HomogeneousGroupDetail]
    analysis_type: str = Field(default="partner_dynamic", description="Tipo análisis")
    calculated_at: str = Field(..., description="Timestamp cálculo (ISO)")


class TemporalDataPoint(BaseModel):
    """Punto de datos temporal"""

    period_start: str = Field(..., description="Inicio período (ISO)")
    period_label: str = Field(..., description="Etiqueta período", json_schema_extra={"example": "Q1 2024"})
    total_units: int = Field(..., description="Unidades totales período")
    total_revenue: float = Field(..., description="Ingresos totales período (€)")
    partner_units: int = Field(..., description="Unidades partners")
    opportunity_units: int = Field(..., description="Unidades oportunidad")
    savings_base: float = Field(..., description="Base ahorro período (€)")
    partner_penetration: float = Field(..., description="Penetración partners (%)")


class TemporalSummary(BaseModel):
    """Resumen datos temporales"""

    periods_count: int = Field(..., description="Número de períodos")
    avg_partner_penetration: float = Field(..., description="Penetración promedio partners (%)")
    total_opportunity_units: int = Field(..., description="Total unidades oportunidad")
    total_savings_base: float = Field(..., description="Total base ahorro (€)")


class TemporalBreakdownResponse(BaseModel):
    """Response para drill-down temporal"""

    pharmacy_id: str = Field(..., description="ID farmacia")
    selected_partner_codes: List[str] = Field(..., description="Códigos partners seleccionados")
    temporal_level: str = Field(..., description="Nivel temporal", json_schema_extra={"example": "quarter"})
    analysis_period: AnalysisPeriod
    temporal_data: List[TemporalDataPoint]
    summary: TemporalSummary
    analysis_type: str = Field(default="temporal_breakdown", description="Tipo análisis")
    calculated_at: str = Field(..., description="Timestamp cálculo (ISO)")


class ProductDetail(BaseModel):
    """Detalle de producto en conjunto homogéneo"""

    national_code: str = Field(..., description="Código nacional producto")
    product_name: Optional[str] = Field(None, description="Nombre producto")
    laboratory: Optional[str] = Field(None, description="Laboratorio")
    pvp: float = Field(..., description="PVP (€)")
    precio_referencia: float = Field(..., description="Precio referencia/PVL (€)")
    drug_type: Optional[str] = Field(None, description="Tipo medicamento")
    units_sold: int = Field(..., description="Unidades vendidas")
    revenue_sold: float = Field(..., description="Ingresos vendidos (€)")
    transactions_count: int = Field(..., description="Número transacciones")
    is_partner: bool = Field(..., description="Es laboratorio partner")
    individual_savings_base: float = Field(..., description="Base ahorro individual (€)")


class LaboratoryInGroup(BaseModel):
    """Laboratorio en conjunto homogéneo"""

    laboratory_name: str = Field(..., description="Nombre laboratorio")
    is_partner: bool = Field(..., description="Es partner seleccionado")
    products_count: int = Field(..., description="Productos del laboratorio")
    total_units: int = Field(..., description="Unidades totales")
    total_revenue: float = Field(..., description="Ingresos totales (€)")


class GroupDetailSummary(BaseModel):
    """Resumen detalle conjunto homogéneo"""

    total_products: int = Field(..., description="Total productos en conjunto")
    total_units: int = Field(..., description="Unidades totales")
    total_revenue: float = Field(..., description="Ingresos totales (€)")
    partner_units: int = Field(..., description="Unidades partners")
    opportunity_units: int = Field(..., description="Unidades oportunidad")
    partner_penetration: float = Field(..., description="Penetración partners (%)")
    total_savings_base: float = Field(..., description="Base ahorro total (€)")


class HomogeneousGroupDetailResponse(BaseModel):
    """Response para detalle conjunto homogéneo"""

    pharmacy_id: str = Field(..., description="ID farmacia")
    homogeneous_code: str = Field(..., description="Código conjunto")
    selected_partner_codes: List[str] = Field(..., description="Códigos partners seleccionados")
    analysis_period: AnalysisPeriod
    group_summary: GroupDetailSummary
    products_detail: List[ProductDetail]
    laboratories_in_group: List[LaboratoryInGroup]
    analysis_type: str = Field(default="homogeneous_group_detail", description="Tipo análisis")
    calculated_at: str = Field(..., description="Timestamp cálculo (ISO)")


# ========== PARTNER PRODUCT REFERENCES ==========


class PartnerProductReference(BaseModel):
    """Referencia de producto de partners"""

    national_code: str = Field(..., description="Código nacional producto")
    product_name: str = Field(..., description="Descripción del producto")
    laboratory: str = Field(..., description="Laboratorio")
    homogeneous_code: Optional[str] = Field(None, description="Código conjunto homogéneo")
    homogeneous_name: Optional[str] = Field(None, description="Nombre conjunto homogéneo")
    units_sold: int = Field(..., description="Unidades vendidas", json_schema_extra={"example": 100})
    total_sales: float = Field(..., description="Ventas totales (€)", json_schema_extra={"example": 1250.50})
    transactions_count: int = Field(..., description="Número de transacciones", json_schema_extra={"example": 15})
    avg_price: float = Field(..., description="Precio medio por unidad (€)", json_schema_extra={"example": 12.51})


class PartnerProductReferencesResponse(BaseModel):
    """Response para referencias de productos de partners"""

    pharmacy_id: str = Field(..., description="ID farmacia")
    selected_partner_codes: List[str] = Field(..., description="Códigos partners seleccionados")
    homogeneous_code: Optional[str] = Field(None, description="Código conjunto homogéneo (si se filtró por uno)")
    analysis_period: AnalysisPeriod
    total_products: int = Field(..., description="Total productos encontrados", json_schema_extra={"example": 250})
    total_units: int = Field(..., description="Total unidades vendidas", json_schema_extra={"example": 5420})
    total_sales: float = Field(..., description="Total ventas (€)", json_schema_extra={"example": 67890.12})
    products: List[PartnerProductReference]
    analysis_type: str = Field(default="partner_product_references", description="Tipo análisis")
    calculated_at: str = Field(..., description="Timestamp cálculo (ISO)")
