"""
Schemas para NarrativeService - Issue #509.

Executive Summary con NLG (Natural Language Generation).
"""

from datetime import datetime
from enum import Enum
from typing import Any, Dict, List, Optional

from pydantic import BaseModel, Field


class AlertSeverity(str, Enum):
    """Severidad de alertas en el resumen ejecutivo."""

    INFO = "info"  # Verde - información positiva
    WARNING = "warning"  # Ámbar - precaución
    CRITICAL = "critical"  # Rojo - acción urgente


class ActionTargetType(str, Enum):
    """Tipo de destino para navegación desde botón de acción."""

    CATEGORY = "category"  # Navegar a categoría L1
    PRODUCT = "product"  # Navegar a producto específico
    L2 = "l2"  # Abrir modal L2 drill-down


class NarrativeSection(BaseModel):
    """
    Sección de narrativa generada por LLM.

    Estructura de 3 capas:
    - headline: EL QUÉ (hechos)
    - analysis: EL PORQUÉ (diagnóstico)
    - action: EL CÓMO (acción)
    """

    # EL QUÉ (Hechos)
    headline: str = Field(
        ...,
        description="Frase impactante de 10-15 palabras resumiendo el estado",
        min_length=10,
        max_length=200,
    )

    # EL PORQUÉ (Diagnóstico)
    analysis: str = Field(
        ...,
        description="2-3 frases conectando KPIs e insights. Usa **negrita** para cifras clave.",
        min_length=50,
        max_length=1000,
    )

    # Alerta opcional (si hay problema crítico)
    alert: Optional[str] = Field(
        default=None,
        description="Mensaje de alerta si hay insight CRITICAL/WARNING",
    )
    alert_severity: Optional[AlertSeverity] = Field(
        default=None,
        description="Severidad de la alerta para colorear UI",
    )

    # EL CÓMO (Acción)
    action: str = Field(
        ...,
        description="Recomendación específica basada en el insight de mayor impacto",
        min_length=20,
        max_length=500,
    )
    action_value: Optional[float] = Field(
        default=None,
        description="Impacto económico estimado de la acción en €",
        ge=0,
    )
    action_button_text: Optional[str] = Field(
        default="Ver detalles",
        description="Texto corto para botón de acción (ej: 'Ver oportunidad')",
        max_length=30,
    )

    # Navegación desde botón de acción
    action_target_type: Optional[ActionTargetType] = Field(
        default=None,
        description="Tipo de destino para navegación",
    )
    action_target_id: Optional[str] = Field(
        default=None,
        description="ID del destino (ml_category, product_id, l2_category)",
    )


class NarrativeResponse(BaseModel):
    """
    Respuesta completa del NarrativeService.

    Incluye la narrativa generada más metadata de validación.
    """

    summary: NarrativeSection = Field(
        ...,
        description="Contenido del resumen ejecutivo",
    )
    generated_at: str = Field(
        ...,
        description="Timestamp ISO de generación",
    )
    cache_until: str = Field(
        ...,
        description="Timestamp ISO hasta cuando es válido el cache",
    )
    grounding_valid: bool = Field(
        ...,
        description="True si todos los números coinciden con input (anti-alucinaciones)",
    )
    language: str = Field(
        default="es",
        description="Idioma de la narrativa (es, en, ca)",
    )

    # Metadata para debugging
    insights_used: int = Field(
        default=0,
        description="Número de insights del Insight Engine usados",
    )
    total_opportunity: float = Field(
        default=0.0,
        description="Oportunidad total detectada en €",
    )


class NarrativeFeedbackCreate(BaseModel):
    """Schema para crear feedback de narrativa."""

    narrative_hash: str = Field(
        ...,
        description="Hash SHA256[:16] del contenido para identificación",
        min_length=8,
        max_length=64,
    )
    is_helpful: bool = Field(
        ...,
        description="True = 👍, False = 👎",
    )
    feedback_text: Optional[str] = Field(
        default=None,
        description="Texto opcional explicando el feedback",
        max_length=500,
    )


class NarrativeFeedbackResponse(BaseModel):
    """Respuesta al guardar feedback."""

    id: str
    narrative_hash: str
    is_helpful: bool
    created_at: str


class NarrativeKPIs(BaseModel):
    """
    KPIs de entrada para generación de narrativa.

    Estos son los datos 'anclados' que el LLM debe mencionar exactamente.
    """

    total_sales: float = Field(..., description="Ventas totales en €")
    avg_margin: float = Field(..., description="Margen promedio en %")
    unique_products: int = Field(..., description="Productos únicos vendidos")
    trend_pct: float = Field(..., description="Tendencia vs período anterior en %")

    # Opcional: desglose por categoría principal
    sales_by_category: Optional[Dict[str, float]] = Field(
        default=None,
        description="Ventas por ml_category",
    )

    # Período de datos
    date_from: str = Field(..., description="Fecha inicio período")
    date_to: str = Field(..., description="Fecha fin período")


class InsightSummary(BaseModel):
    """
    Resumen de insight para input del LLM.

    Versión simplificada de InsightResult para el prompt.
    """

    rule_code: str
    category: str
    severity: str
    title: str
    description: str
    economic_value: float
    action_label: str
    deeplink: str


class NarrativeGenerateRequest(BaseModel):
    """
    Request para generar narrativa (uso interno API).
    """

    kpis: NarrativeKPIs
    insights: List[InsightSummary] = Field(default_factory=list)
    language: str = Field(default="es")


# Tipos para validación de grounding
ValidCategories = {
    # NecesidadPrincipal - categorías válidas del sistema
    "dolor_fiebre",
    "respiratorio",
    "digestivo",
    "piel",
    "bucal",
    "ocular",
    "oidos",
    "capilar",
    "bienestar",
    "circulatorio",
    "urinario",
    "sexual",
    "infantil",
    "nutricion",
    "movilidad",
    "incontinencia_cat",
    "diabetes_cat",
    "botiquin",
    "veterinaria_cat",
    "servicios_cat",
    "no_salud",
    # Categorías VentaLibre específicas (L1)
    "dermocosmetica",
    "suplementos",
    "higiene_bucal",
    "higiene_personal",
    "nutricion_deportiva",
    "bebe_infantil",
    "salud_sexual",
    "otros",
    "interno_no_venta",
}
