"""
Schemas Pydantic para exportación de productos en manual_review.

Issue #447: Exportar productos sin enriquecer a CSV para análisis offline.

Clasificación de códigos:
- CN_CATALOGED: CN válido, existe en CIMA/Nomenclator
- CN_OTC: CN válido checksum, NO en catálogos (parafarmacia)
- EAN_CATALOGED: EAN válido, existe en catálogos
- EAN_OTC: EAN válido checksum, NO en catálogos
- CN_UNVERIFIED: 6 dígitos numérico (puede ser CN legacy O código interno)
- INTERNAL: Código interno farmacia
"""

from datetime import date, datetime
from decimal import Decimal
from typing import Dict, List, Literal, Optional
from uuid import UUID

from pydantic import BaseModel, Field


# Tipos de código posibles
CodeType = Literal[
    "CN_CATALOGED",   # CN válido, existe en CIMA/Nomenclator
    "CN_OTC",         # CN válido checksum, NO en catálogos (parafarmacia)
    "EAN_CATALOGED",  # EAN válido, existe en catálogos
    "EAN_OTC",        # EAN válido checksum, NO en catálogos
    "CN_UNVERIFIED",  # 6 dígitos numérico (puede ser CN legacy O código interno)
    "INTERNAL",       # Código interno farmacia (confirmado no-universal)
]


# ============================================================================
# RESPONSE SCHEMAS
# ============================================================================


class ManualReviewProduct(BaseModel):
    """
    Producto individual en manual_review con métricas agregadas.

    IMPORTANTE: Cada producto se agrupa POR FARMACIA para evitar
    conciliar erróneamente códigos internos entre farmacias distintas.

    Incluye datos originales (codigo_nacional, ean13) sin transformar
    para máxima trazabilidad.
    """

    # Datos originales (sin transformar)
    codigo_nacional: Optional[str] = Field(
        None, description="Código nacional original del ERP (puede ser interno)"
    )
    ean13: Optional[str] = Field(
        None, description="EAN-13 original del ERP"
    )
    product_name: str = Field(..., description="Nombre del producto")

    # Identificación de farmacia (agrupamos por farmacia)
    pharmacy_id: Optional[UUID] = Field(
        None, description="ID de la farmacia (cada producto es único por farmacia)"
    )
    pharmacy_name: Optional[str] = Field(
        None, description="Nombre de la farmacia"
    )

    # Código para display (COALESCE de CN > EAN > 'SIN_CODIGO')
    product_code: str = Field(..., description="Código para display (CN, EAN o 'SIN_CODIGO')")

    # Clasificación (descriptiva, no prescriptiva)
    code_type: CodeType = Field(
        ...,
        description=(
            "Tipo de código: CN_CATALOGED (en CIMA), CN_OTC (parafarmacia), "
            "EAN_CATALOGED, EAN_OTC, CN_UNVERIFIED (6 dígitos sin validar), INTERNAL"
        )
    )
    checksum_valid: bool = Field(
        default=False,
        description="True si el checksum del código es válido (CN-7 o EAN-13)"
    )
    in_catalog: bool = Field(
        default=False,
        description="True si el código existe en CIMA o Nomenclator"
    )

    # Métricas (dentro de esta farmacia)
    pharmacy_count: int = Field(
        default=1,
        description="Número de farmacias que tienen este producto (1 si agrupado por farmacia)"
    )
    sale_count: int = Field(..., description="Número total de líneas de venta")
    total_units: int = Field(..., description="Unidades totales vendidas")
    total_amount: Decimal = Field(..., description="Importe total en euros")
    avg_price: Decimal = Field(..., description="Precio unitario promedio")
    first_sale: date = Field(..., description="Fecha de primera venta")
    last_sale: date = Field(..., description="Fecha de última venta")

    class Config:
        from_attributes = True


class ManualReviewStats(BaseModel):
    """Estadísticas resumen de productos en manual_review."""

    total_products: int = Field(..., description="Total de productos únicos sin enriquecer")
    total_sales: int = Field(..., description="Total de líneas de venta sin enriquecer")
    total_amount: Decimal = Field(..., description="Importe total sin analizar")
    by_code_type: Dict[str, int] = Field(
        ...,
        description="Desglose por tipo de código: {'CN': 4116, 'EAN': 375, 'INTERNAL': 0}",
    )


class ManualReviewExportResponse(BaseModel):
    """Respuesta de exportación de manual_review."""

    filename: str = Field(..., description="Nombre del archivo generado")
    row_count: int = Field(..., description="Número de productos exportados")
    generated_at: datetime = Field(..., description="Timestamp de generación")
    stats: ManualReviewStats = Field(..., description="Estadísticas de los datos exportados")


class ManualReviewListResponse(BaseModel):
    """Respuesta paginada de productos en manual_review."""

    products: List[ManualReviewProduct] = Field(
        default_factory=list, description="Lista de productos"
    )
    total_count: int = Field(..., description="Total de productos (sin paginar)")
    page: int = Field(default=1, description="Página actual")
    page_size: int = Field(default=100, description="Tamaño de página")
    stats: ManualReviewStats = Field(..., description="Estadísticas globales")
