# backend/app/schemas/llm_enrichment.py
"""
Schemas Pydantic para LLM Enrichment Pipeline (Issue #456)

Estos schemas se usan con la librería `instructor` para garantizar
salida JSON estructurada desde el LLM (Qwen2.5:7b-instruct via Ollama).

Uso con instructor:
    from instructor import from_ollama
    from ollama import AsyncClient

    client = from_ollama(AsyncClient(), mode=instructor.Mode.JSON)
    result = await client.chat.completions.create(
        model="qwen2.5:7b-instruct",
        response_model=ProductEnrichmentSchema,  # Pydantic model
        messages=[...]
    )
"""

from datetime import datetime
from enum import Enum
from typing import List, Optional

from pydantic import BaseModel, Field, field_validator


class NecesidadCategory(str, Enum):
    """
    Categorías de NECESIDAD para productos de venta libre.
    194 categorías reales extraídas del catálogo xFarma.
    El LLM DEBE usar una de estas categorías exactas.
    """
    # Accesorios y cosmética
    ACCESORIOS_BELLEZA = "accesorios_belleza"
    AFEITADO = "afeitado"
    DEPILACION = "depilacion"
    DESODORANTE = "desodorante"
    EXFOLIACION = "exfoliacion"
    MAQUILLAJE = "maquillaje"
    PERFUMERIA = "perfumeria"
    PESTANAS_CEJAS = "pestañas_cejas"
    SET_REGALO = "set_regalo"
    JOYERIA = "joyeria"

    # Digestivo
    ACIDEZ_ESTOMAGO = "acidez_estomago"
    ACIDEZ_REFLUJO = "acidez_reflujo"
    DIARREA = "diarrea"
    ESTRENIMIENTO = "estrenimiento"
    FLORA_INTESTINAL = "flora_intestinal"
    GASES_DIGESTION = "gases_digestion"
    PROBIOTICOS = "probioticos"
    PROBIOTICOS_GENERAL = "probioticos_general"
    DETOX_HIGADO = "detox_higado"
    SALUD_HEPATICA = "salud_hepatica"

    # Dolor y muscular
    DOLOR = "dolor"
    DOLOR_DENTAL = "dolor_dental"
    DOLOR_GARGANTA = "dolor_garganta"
    DOLOR_MUSCULAR = "dolor_muscular"
    ESPASMOS_DOLOR = "espasmos_dolor"
    ARTICULACIONES = "articulaciones"
    HEMATOMAS = "hematomas"

    # Respiratorio y alergias
    ALERGIA = "alergia"
    ALERGIA_NASAL = "alergia_nasal"
    ALERGIA_OCULAR = "alergia_ocular"
    CONGESTION_NASAL = "congestion_nasal"
    GRIPE_RESFRIADO = "gripe_resfriado"
    TOS_GARGANTA = "tos_garganta"
    MUCOSIDAD_RESPIRATORIA = "mucosidad_respiratoria"
    CARAMELOS_GARGANTA = "caramelos_garganta"
    HEMORRAGIA_NASAL = "hemorragia_nasal"

    # Bucal y dental (Issue #457: subcategorías desglosadas)
    CEPILLO_MANUAL = "cepillo_manual"
    CEPILLO_ELECTRICO = "cepillo_electrico"
    PASTA_DENTAL = "pasta_dental"
    ENJUAGUE_BUCAL = "enjuague_bucal"
    SEDA_DENTAL = "seda_dental"
    INTERDENTAL = "interdental"
    IRRIGADOR_DENTAL = "irrigador_dental"
    HIGIENE_BUCAL_INFANTIL = "higiene_bucal_infantil"
    HIGIENE_DENTAL = "higiene_dental"
    BLANQUEAMIENTO = "blanqueamiento"
    BLANQUEAMIENTO_DENTAL = "blanqueamiento_dental"
    BOCA_SECA = "boca_seca"
    CARIES = "caries"
    ENCIAS = "encias"
    HALITOSIS = "halitosis"
    ORTODONCIA = "ortodoncia"
    PROTESIS_DENTAL = "protesis_dental"
    SENSIBILIDAD_DENTAL = "sensibilidad_dental"

    # Piel - Cuidado facial
    ACNE = "acne"
    ARRUGAS_ANTIEDAD = "arrugas_antiedad"
    CONTORNO_OJOS = "contorno_ojos"
    CUIDADO_FACIAL_GENERAL = "cuidado_facial_general"
    HIDRATACION_FACIAL = "hidratacion_facial"
    LIMPIEZA_FACIAL = "limpieza_facial"
    MANCHAS = "manchas"
    ROJECES_ROSACEA = "rojeces_rosacea"
    CUIDADO_LABIOS = "cuidado_labios"

    # Piel - Cuerpo
    CICATRICES_ESTRIAS = "cicatrices_estrias"
    # CICATRIZACION movido a sección Heridas (Issue #457)
    CUIDADO_CORPORAL = "cuidado_corporal"
    DERMATITIS = "dermatitis"
    HIDRATACION_CORPORAL = "hidratacion_corporal"
    IRRITACION_PIEL = "irritacion_piel"
    MANOS = "manos"
    PIEL_ATOPICA = "piel_atopica"
    PIEL_SECA = "piel_seca"
    PIEL_SENSIBLE = "piel_sensible"
    PSORIASIS = "psoriasis"
    QUEMADURAS = "quemaduras"
    QUEMADURAS_AFTERSUN = "quemaduras_aftersun"
    SUDORACION = "sudoracion"

    # Piel - Pies
    HONGOS_PIES = "hongos_pies"
    PIES_SECOS = "pies_secos"
    PIES_TALONES = "pies_talones"
    VERRUGAS_CALLOS = "verrugas_callos"

    # Cabello
    CABELLO_GRASO = "cabello_graso"
    CABELLO_RUBIO = "cabello_rubio"
    CABELLO_SECO = "cabello_seco"
    CAIDA_CABELLO = "caida_cabello"
    CASPA = "caspa"
    CUERO_CABELLUDO = "cuero_cabelludo"
    CUIDADO_CABELLO = "cuidado_cabello"
    DERMATITIS_SEBORREICA = "dermatitis_seborreica"
    HIGIENE_CABELLO = "higiene_cabello"

    # Uñas
    HONGOS_UNAS = "hongos_unas"
    UNAS = "unas"

    # Solar
    PROTECCION_SOLAR = "proteccion_solar"

    # Ocular
    FATIGA_VISUAL = "fatiga_visual"
    HIGIENE_OCULAR = "higiene_ocular"
    IRRITACION_OCULAR = "irritacion_ocular"
    LENTES_CONTACTO = "lentes_contacto"
    OJO_SECO = "ojo_seco"
    OPTICA = "optica"
    SALUD_OCULAR = "salud_ocular"
    VISION = "vision"

    # Oídos
    ACUFENOS = "acufenos"
    HIGIENE_OIDOS = "higiene_oidos"
    PROTECCION_AUDITIVA = "proteccion_auditiva"

    # Nasal
    HIGIENE_NASAL = "higiene_nasal"

    # Íntima y sexual
    ANTICONCEPCION_EMERGENCIA = "anticoncepcion_emergencia"
    CISTITIS = "cistitis"
    FLORA_VAGINAL = "flora_vaginal"
    HIGIENE_INTIMA = "higiene_intima"
    HONGOS_VAGINALES = "hongos_vaginales"
    INFECCION_URINARIA = "infeccion_urinaria"
    SALUD_SEXUAL = "salud_sexual"
    SALUD_VAGINAL = "salud_vaginal"
    SEQUEDAD_VAGINAL = "sequedad_vaginal"

    # Bebé e infantil
    ALIMENTACION_BEBE = "alimentacion_bebe"
    BEBES_INFANTIL = "bebes_infantil"
    COLICOS_BEBE = "colicos_bebe"
    CONFORT_BEBE = "confort_bebe"
    DENTICION_BEBE = "denticion_bebe"
    DERMATITIS_PANAL = "dermatitis_panal"
    HIDRATACION_BEBE = "hidratacion_bebe"
    HIGIENE_BEBE = "higiene_bebe"
    IRRITACION_PANAL = "irritacion_panal"
    PANALES = "panales"
    VITAMINAS_INFANTIL = "vitaminas_infantil"

    # Vitaminas y suplementos
    CALCIO_HUESOS = "calcio_huesos"
    COLAGENO = "colageno"
    DEFENSAS = "defensas"
    DEFENSAS_INMUNIDAD = "defensas_inmunidad"
    DEPORTE_RENDIMIENTO = "deporte_rendimiento"
    ENERGIA_CANSANCIO = "energia_cansancio"
    HIERRO = "hierro"
    INMUNIDAD = "inmunidad"
    MAGNESIO = "magnesio"
    MEMORIA_COGNITIVO = "memoria_cognitivo"
    MEMORIA_CONCENTRACION = "memoria_concentracion"
    MINERALES = "minerales"
    OMEGA_3 = "omega_3"
    OMEGA3 = "omega3"
    VITAMINAS_GENERAL = "vitaminas_general"

    # Sueño y bienestar emocional
    ESTRES_ANSIEDAD = "estres_ansiedad"
    SUENO_INSOMNIO = "sueno_insomnio"

    # Peso y nutrición
    ALIMENTACION = "alimentacion"
    APETITO = "apetito"
    COLESTEROL = "colesterol"
    DRENANTE = "drenante"
    EDULCORANTE = "edulcorante"
    GLUCOSA = "glucosa"
    NUTRICION_CLINICA = "nutricion_clinica"
    NUTRICION_DEPORTIVA = "nutricion_deportiva"
    PERDIDA_PESO = "perdida_peso"

    # Mujer
    EMBARAZO_PRENATAL = "embarazo_prenatal"
    FERTILIDAD = "fertilidad"
    LACTANCIA = "lactancia"
    MENOPAUSIA = "menopausia"

    # Circulación y cardiovascular
    CIRCULACION = "circulacion"
    CIRCULACION_PIERNAS = "circulacion_piernas"
    HEMORROIDES = "hemorroides"
    TENSION_ARTERIAL = "tension_arterial"

    # Heridas y primeros auxilios (Issue #457: consolidación)
    DESINFECCION_HERIDAS = "desinfeccion_heridas"  # Antisépticos (betadine, alcohol, clorhexidina)
    CICATRIZACION = "cicatrizacion"                # Cicatrizantes (Bio-Oil, Bepanthol)
    APOSITOS_CURAS = "apositos_curas"              # Tiritas, gasas, vendas, apósitos
    HERPES = "herpes"
    HERPES_LABIAL = "herpes_labial"
    PICADURAS = "picaduras"
    REPELENTE_INSECTOS = "repelente_insectos"

    # Material sanitario y dispositivos
    CUIDADO_DEPENDIENTES = "cuidado_dependientes"
    DISPOSITIVOS_MEDICOS = "dispositivos_medicos"
    GLUCOSA_MEDICION = "glucosa_medicion"
    INCONTINENCIA = "incontinencia"
    MATERIAL_SANITARIO = "material_sanitario"
    MONITORIZACION = "monitorizacion"
    MOVILIDAD_REDUCIDA = "movilidad_reducida"
    ORTOPEDIA = "ortopedia"
    OSTOMIA = "ostomia"
    TENSION_MEDICION = "tension_medicion"
    TERMOMETRO = "termometro"
    TERMOTERAPIA = "termoterapia"
    TEST_DIAGNOSTICO = "test_diagnostico"
    TEST_EMBARAZO = "test_embarazo"
    TEST_OVULACION = "test_ovulacion"  # Issue #457
    TEST_COVID_GRIPE = "test_covid_gripe"  # Issue #457
    VENDAJE_COMPRESION = "vendaje_compresion"

    # Otros
    AFTAS = "aftas"
    ANTIPARASITARIOS = "antiparasitarios"
    COMPLEMENTOS = "complementos"
    DIABETES = "diabetes"
    FITOTERAPIA = "fitoterapia"
    HIGIENE_CORPORAL = "higiene_corporal"
    HIGIENE_MANOS = "higiene_manos"
    HOMEOPATIA = "homeopatia"
    INFUSIONES = "infusiones"
    MAREO_VIAJE = "mareo_viaje"
    NEUROPATIA = "neuropatia"
    PEDICULOSIS = "pediculosis"
    PIOJOS = "piojos"
    RENAL = "renal"
    SERVICIO = "servicio"
    TABAQUISMO = "tabaquismo"
    TIROIDES = "tiroides"
    VETERINARIA = "veterinaria"

    # No-salud (Issue #457)
    DULCERIA = "dulceria"  # Caramelos sin propiedades medicinales

    # Fallback
    OTROS = "otros"


class ProductEnrichmentSchema(BaseModel):
    """
    Schema para extracción estructurada por LLM.

    El campo `necesidad` usa NecesidadCategory enum para forzar
    categorías válidas. Si el LLM devuelve una categoría inválida,
    Pydantic lanza ValidationError y tenacity reintenta.
    """
    necesidad: str = Field(
        description="Categoría de necesidad del producto - DEBE ser una de las categorías válidas"
    )
    subcategoria: Optional[str] = Field(
        default=None,
        description="Subcategoría opcional"
    )
    indicaciones: Optional[str] = Field(
        default=None,
        description="Indicaciones de uso del producto"
    )
    composicion_principal: Optional[str] = Field(
        default=None,
        description="Ingredientes o principios activos principales"
    )
    modo_empleo: Optional[str] = Field(
        default=None,
        description="Cómo usar el producto"
    )
    confianza: float = Field(
        default=0.7,
        ge=0.0,
        le=1.0,
        description="Confianza del LLM en la clasificación (0.0-1.0)"
    )

    @field_validator('necesidad')
    @classmethod
    def validate_necesidad(cls, v: str) -> str:
        """Valida y normaliza la necesidad contra el enum."""
        normalized = v.lower().strip().replace(" ", "_").replace("-", "_")

        # Verificar si es una categoría válida
        valid_categories = {cat.value for cat in NecesidadCategory}
        if normalized not in valid_categories:
            # Intentar fuzzy match simple
            for cat in valid_categories:
                if normalized in cat or cat in normalized:
                    return cat
            # Si no hay match, lanzar error para que tenacity reintente
            raise ValueError(
                f"Categoría '{normalized}' no válida. "
                f"Debe ser una de: {', '.join(sorted(valid_categories)[:20])}..."
            )
        return normalized

    @field_validator('subcategoria')
    @classmethod
    def validate_subcategoria(cls, v: Optional[str]) -> Optional[str]:
        """Normaliza la subcategoría si existe."""
        if v:
            return v.lower().strip().replace(" ", "_").replace("-", "_")
        return v


class ProductEnrichmentInput(BaseModel):
    """Input para el servicio de enriquecimiento LLM."""
    product_name: str = Field(
        min_length=3,
        description="Nombre del producto a enriquecer"
    )
    existing_brand: Optional[str] = Field(
        default=None,
        description="Marca ya detectada por BrandDetectionService"
    )
    existing_necesidad: Optional[str] = Field(
        default=None,
        description="NECESIDAD ya clasificada (para validación cruzada)"
    )
    existing_atc: Optional[str] = Field(
        default=None,
        description="Código ATC si disponible (para productos OTC)"
    )


class LLMEnrichmentResult(BaseModel):
    """Resultado completo del enriquecimiento LLM."""
    success: bool = Field(description="Si el enriquecimiento fue exitoso")
    product_id: Optional[str] = Field(
        default=None,
        description="UUID del SalesEnrichment actualizado"
    )
    product_name: str = Field(description="Nombre del producto procesado")
    enrichment: Optional[ProductEnrichmentSchema] = Field(
        default=None,
        description="Datos extraídos por el LLM"
    )
    error_message: Optional[str] = Field(
        default=None,
        description="Mensaje de error si falló"
    )
    processing_time_ms: int = Field(
        description="Tiempo de procesamiento en milisegundos"
    )
    model_version: str = Field(
        default="qwen2.5:7b-instruct",
        description="Versión del modelo LLM usado"
    )
    retry_count: int = Field(
        default=0,
        description="Número de reintentos realizados"
    )
    raw_response: Optional[str] = Field(
        default=None,
        description="Respuesta cruda del LLM (para debugging)"
    )


class BatchEnrichmentProgress(BaseModel):
    """Progreso del enriquecimiento batch."""
    total_products: int = Field(description="Total de productos a procesar")
    processed: int = Field(description="Productos procesados")
    enriched: int = Field(description="Productos enriquecidos exitosamente")
    failed: int = Field(description="Productos que fallaron")
    skipped: int = Field(
        default=0,
        description="Productos omitidos (ya enriquecidos)"
    )
    current_product: Optional[str] = Field(
        default=None,
        description="Producto siendo procesado actualmente"
    )
    estimated_remaining_seconds: Optional[int] = Field(
        default=None,
        description="Tiempo estimado restante"
    )
    avg_time_per_product_ms: Optional[float] = Field(
        default=None,
        description="Tiempo promedio por producto"
    )


class BatchEnrichmentResult(BaseModel):
    """Resultado final del enriquecimiento batch."""
    total_processed: int = Field(description="Total de productos procesados")
    enriched_count: int = Field(description="Productos enriquecidos exitosamente")
    failed_count: int = Field(description="Productos que fallaron")
    skipped_count: int = Field(
        default=0,
        description="Productos omitidos"
    )
    avg_confidence: float = Field(
        description="Confianza promedio de los enriquecimientos"
    )
    processing_time_seconds: float = Field(
        description="Tiempo total de procesamiento"
    )
    checkpoint_path: Optional[str] = Field(
        default=None,
        description="Ruta al archivo de checkpoint"
    )
    errors: List[str] = Field(
        default_factory=list,
        description="Lista de errores encontrados"
    )
    started_at: datetime = Field(description="Inicio del batch")
    completed_at: datetime = Field(description="Fin del batch")
    model_version: str = Field(
        default="qwen2.5:7b-instruct",
        description="Versión del modelo usado"
    )


class CheckpointData(BaseModel):
    """Datos del checkpoint para resume de batch."""
    checkpoint_id: str = Field(description="ID único del checkpoint")
    processed_ids: List[str] = Field(
        description="IDs de productos ya procesados"
    )
    enriched_count: int = Field(description="Productos enriquecidos hasta ahora")
    failed_count: int = Field(description="Productos fallidos hasta ahora")
    last_product_id: Optional[str] = Field(
        default=None,
        description="Último producto procesado"
    )
    timestamp: datetime = Field(description="Momento del checkpoint")
    model_version: str = Field(description="Versión del modelo")


# === SCHEMAS PARA API ===

class EnrichProductRequest(BaseModel):
    """Request para enriquecer un producto individual."""
    product_name: str = Field(
        min_length=3,
        description="Nombre del producto"
    )
    sales_enrichment_id: Optional[str] = Field(
        default=None,
        description="UUID del SalesEnrichment a actualizar"
    )
    force: bool = Field(
        default=False,
        description="Forzar re-enriquecimiento aunque ya exista"
    )


class EnrichProductResponse(BaseModel):
    """Response del enriquecimiento de un producto."""
    success: bool
    product_name: str
    necesidad: Optional[str] = None
    subcategoria: Optional[str] = None
    indicaciones: Optional[str] = None
    composicion: Optional[str] = None
    modo_empleo: Optional[str] = None
    confidence: Optional[float] = None
    processing_time_ms: int
    model_version: str
    error: Optional[str] = None


class StartBatchRequest(BaseModel):
    """Request para iniciar batch de enriquecimiento."""
    limit: Optional[int] = Field(
        default=None,
        ge=1,
        le=50000,
        description="Límite de productos a procesar"
    )
    checkpoint_every: int = Field(
        default=100,
        ge=10,
        le=1000,
        description="Guardar checkpoint cada N productos"
    )
    dry_run: bool = Field(
        default=False,
        description="Simular sin guardar en BD"
    )
    resume_from: Optional[str] = Field(
        default=None,
        description="ID de checkpoint para continuar"
    )
    filter_necesidad: Optional[str] = Field(
        default=None,
        description="Filtrar por NECESIDAD específica"
    )
    filter_status: Optional[str] = Field(
        default="manual_review",
        description="Filtrar por enrichment_status"
    )


class BatchStatusResponse(BaseModel):
    """Response con estado del batch."""
    batch_id: str
    status: str  # 'running', 'completed', 'failed', 'cancelled'
    progress: BatchEnrichmentProgress
    started_at: datetime
    estimated_completion: Optional[datetime] = None
