# backend/app/schemas/keyword_override.py
"""
Schemas Pydantic para API de keyword overrides.

Issue #449: Dynamic keyword management for NECESIDAD classifier.

Note: Uses Annotated + BeforeValidator pattern for OpenAPI compatibility.
"""

from datetime import datetime
from typing import Annotated, Any, Optional
from uuid import UUID

from pydantic import BaseModel, BeforeValidator, Field


# === Validators as functions (OpenAPI compatible) ===


def normalize_keyword(v: Any) -> str:
    """Normalize keyword to lowercase and strip whitespace."""
    if v is None:
        return v
    if isinstance(v, str):
        return v.lower().strip()
    return str(v).lower().strip()


def validate_keyword_type(v: Any) -> str:
    """Validate and normalize keyword_type."""
    if v is None:
        return v
    allowed = {"keyword", "brand", "blacklist"}
    val = str(v).lower()
    if val not in allowed:
        raise ValueError(f"keyword_type must be one of: {allowed}")
    return val


# === Annotated types for reuse ===

KeywordStr = Annotated[str, BeforeValidator(normalize_keyword)]
KeywordTypeStr = Annotated[str, BeforeValidator(validate_keyword_type)]
OptionalKeywordStr = Annotated[Optional[str], BeforeValidator(normalize_keyword)]
OptionalKeywordTypeStr = Annotated[Optional[str], BeforeValidator(validate_keyword_type)]


# === Request Schemas ===


class KeywordOverrideBase(BaseModel):
    """Base schema for keyword override."""

    keyword: KeywordStr = Field(..., min_length=2, max_length=200, description="Keyword to match")
    category: str = Field(
        ..., min_length=2, max_length=100, description="Target NECESIDAD category"
    )
    keyword_type: KeywordTypeStr = Field(
        default="keyword", description="Type: keyword, brand, blacklist"
    )
    priority: int = Field(default=100, ge=0, le=1000, description="Priority (higher = first)")
    is_active: bool = Field(default=True, description="Whether to use in classification")
    notes: Optional[str] = Field(None, description="Admin notes")


class KeywordOverrideCreate(KeywordOverrideBase):
    """Schema for creating keyword override."""

    pass


class KeywordOverrideUpdate(BaseModel):
    """Schema for updating keyword override (all fields optional)."""

    keyword: OptionalKeywordStr = Field(None, min_length=2, max_length=200)
    category: Optional[str] = Field(None, min_length=2, max_length=100)
    keyword_type: OptionalKeywordTypeStr = None
    priority: Optional[int] = Field(None, ge=0, le=1000)
    is_active: Optional[bool] = None
    notes: Optional[str] = None


# === Response Schemas ===


class KeywordOverrideResponse(BaseModel):
    """Response schema for keyword override."""

    id: int
    keyword: str
    category: str
    keyword_type: str
    priority: int
    is_active: bool
    created_by: Optional[UUID] = None
    created_at: datetime
    updated_at: datetime
    source: str
    notes: Optional[str] = None
    creator_email: Optional[str] = None  # From relationship

    model_config = {"from_attributes": True}


class KeywordOverrideStats(BaseModel):
    """Statistics about keyword overrides."""

    by_type: dict[str, int]  # {keyword: N, brand: M, blacklist: K}
    by_category: dict[str, int]  # {caida_cabello: N, ...}


class KeywordOverrideListResponse(BaseModel):
    """Response for keyword list endpoint."""

    items: list[KeywordOverrideResponse]
    total: int
    offset: int
    limit: int
    stats: KeywordOverrideStats


# === Preview Schemas ===


class KeywordPreviewRequest(BaseModel):
    """Request to preview keyword impact."""

    keyword: KeywordStr = Field(..., min_length=2, max_length=200)
    category: str = Field(..., min_length=2, max_length=100)
    keyword_type: str = Field(default="keyword")
    limit: int = Field(default=50, ge=1, le=200)


class KeywordPreviewProduct(BaseModel):
    """Product affected by keyword change."""

    product_name: str
    current_category: Optional[str]
    new_category: str
    confidence_change: float


class KeywordPreviewResponse(BaseModel):
    """Response for keyword preview."""

    keyword: str
    new_category: str
    affected_count: int
    sample_products: list[KeywordPreviewProduct]


# === Import/Export Schemas ===


class KeywordBulkImportItem(BaseModel):
    """Single item in bulk import."""

    keyword: KeywordStr = Field(..., min_length=2, max_length=200)
    category: str = Field(..., min_length=2, max_length=100)
    keyword_type: str = Field(default="keyword")
    priority: int = Field(default=100, ge=0, le=1000)


class KeywordBulkImportRequest(BaseModel):
    """Request for bulk import."""

    items: list[KeywordBulkImportItem]
    overwrite_existing: bool = Field(
        default=False, description="If true, update existing keywords"
    )


class KeywordBulkImportResponse(BaseModel):
    """Response for bulk import."""

    created: int
    updated: int
    skipped: int
    errors: list[dict]


# === Category Schema ===


class AvailableCategoryResponse(BaseModel):
    """Available NECESIDAD category for dropdowns."""

    value: str
    label: str
    description: Optional[str] = None


# === Apply Keywords Schemas (Issue #449 Phase 3) ===


class ApplyKeywordsRequest(BaseModel):
    """Request to apply keywords to existing products."""

    dry_run: bool = Field(
        default=True,
        description="If true, only preview changes without applying"
    )
    keyword_ids: Optional[list[int]] = Field(
        None,
        description="Specific keyword IDs to apply. If None, applies all active."
    )


class ApplyKeywordsProduct(BaseModel):
    """Product that would be/was reclassified."""

    product_id: UUID
    product_name: str
    old_category: Optional[str]
    new_category: str
    matched_keyword: str
    keyword_type: str


class ApplyKeywordsResponse(BaseModel):
    """Response for apply keywords endpoint."""

    dry_run: bool
    total_products_scanned: int
    products_reclassified: int
    sample_changes: list[ApplyKeywordsProduct]
    message: str
