# backend/app/schemas/invitation.py
"""
Pydantic schemas para el sistema de invitaciones
"""

from datetime import datetime
from typing import Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, EmailStr, Field, field_validator


# Request schemas
class InvitationCreate(BaseModel):
    """Schema para crear una invitación"""

    email: EmailStr
    role: str = Field(default="user", pattern="^(admin|user)$")
    full_name: Optional[str] = Field(None, max_length=255)
    dni_nie: Optional[str] = Field(None, max_length=20)
    message: Optional[str] = Field(None, max_length=500)
    expires_in_days: int = Field(default=7, ge=1, le=30)

    @field_validator("dni_nie")
    @classmethod
    def validate_dni_nie(cls, v: Optional[str]) -> Optional[str]:
        """Valida y normaliza DNI/NIE español"""
        if v is None:
            return v

        from app.utils.validators import sanitize_dni_nie

        cleaned = sanitize_dni_nie(v)
        if cleaned is None:
            raise ValueError("Formato de DNI/NIE inválido")
        return cleaned


class InvitationAccept(BaseModel):
    """Schema para aceptar una invitación"""

    token: str = Field(..., min_length=36, max_length=36)
    password: str = Field(..., min_length=8)
    username: Optional[str] = Field(None, min_length=3, max_length=50)


class InvitationValidate(BaseModel):
    """Schema para validar un token de invitación"""

    token: str = Field(..., min_length=36, max_length=36)


# Response schemas
class InvitationResponse(BaseModel):
    """Schema para respuesta de invitación"""

    id: UUID
    pharmacy_id: UUID
    email: str
    role: str
    full_name: Optional[str]
    status: str
    expires_at: datetime
    created_at: datetime
    message: Optional[str]
    invitation_url: Optional[str] = None

    model_config = ConfigDict(from_attributes=True)


class InvitationListResponse(BaseModel):
    """Schema para lista de invitaciones"""

    invitations: list[InvitationResponse]
    total: int
    pending_count: int
    accepted_count: int


class InvitationValidationResponse(BaseModel):
    """Schema para respuesta de validación de token"""

    valid: bool
    email: Optional[str] = None
    pharmacy_name: Optional[str] = None
    role: Optional[str] = None
    full_name: Optional[str] = None
    message: Optional[str] = None
    error: Optional[str] = None


class InvitationAcceptResponse(BaseModel):
    """Schema para respuesta de aceptación de invitación"""

    user: dict
    message: str = "Usuario creado exitosamente desde invitación"
