# backend/app/schemas/insights.py
"""
Pydantic schemas para Insight Engine API.

Issue #506: Schemas para request/response de insights.
"""

from datetime import datetime
from typing import Any, Optional
from uuid import UUID

from pydantic import BaseModel, Field


class InsightItem(BaseModel):
    """Schema para un insight individual."""

    rule_code: str = Field(..., description="Código único de la regla (ej: STOCK_001)")
    category: str = Field(..., description="Categoría: stock, margin, hhi, trend, surtido")
    severity: str = Field(..., description="Severidad: high, medium, low")
    title: str = Field(..., description="Título corto para UI")
    description: str = Field(..., description="Descripción detallada")
    impact_score: int = Field(..., ge=0, le=100, description="Score de impacto 0-100")
    economic_impact: str = Field(..., description="Texto legible del impacto (ej: +450€/mes)")
    economic_value: float = Field(..., description="Valor numérico en € para cálculos")
    action_label: str = Field(..., description="Texto del botón de acción")
    deeplink: str = Field(..., description="URL relativa para navegación")
    affected_items: list[dict[str, Any]] = Field(
        default_factory=list,
        description="Items afectados para drill-down"
    )
    insight_hash: str = Field(..., description="Hash para identificación y feedback")

    class Config:
        json_schema_extra = {
            "example": {
                "rule_code": "STOCK_001",
                "category": "stock",
                "severity": "high",
                "title": "Stock Crítico: 12 productos de alta rotación",
                "description": "Tienes 12 productos con alta demanda y menos de 3 días de cobertura.",
                "impact_score": 85,
                "economic_impact": "Est. +1.200€/mes en ventas recuperables",
                "economic_value": 1200.0,
                "action_label": "Ver productos afectados",
                "deeplink": "/ventalibre/inventario?alert=low_stock",
                "affected_items": [
                    {"product_code": "123456", "product_name": "IBUPROFENO 600MG", "coverage_days": 2}
                ],
                "insight_hash": "a1b2c3d4e5f6g7h8",
            }
        }


class InsightsResponse(BaseModel):
    """Schema para respuesta de lista de insights."""

    insights: list[InsightItem] = Field(
        default_factory=list,
        description="Lista de insights ordenados por economic_value DESC"
    )
    total_opportunity: float = Field(
        default=0.0,
        description="Suma de economic_value de todos los insights"
    )
    by_category: dict[str, int] = Field(
        default_factory=dict,
        description="Conteo de insights por categoría"
    )
    by_severity: dict[str, int] = Field(
        default_factory=dict,
        description="Conteo de insights por severidad"
    )
    suppressed_count: int = Field(
        default=0,
        description="Número de insights filtrados por feedback"
    )
    audit_duration_ms: int = Field(
        default=0,
        description="Tiempo de ejecución del audit en ms"
    )

    class Config:
        json_schema_extra = {
            "example": {
                "insights": [],
                "total_opportunity": 3500.0,
                "by_category": {"stock": 3, "margin": 2, "trend": 1},
                "by_severity": {"high": 2, "medium": 3, "low": 1},
                "suppressed_count": 2,
                "audit_duration_ms": 450,
            }
        }


class InsightFeedbackRequest(BaseModel):
    """Schema para registrar feedback sobre un insight."""

    insight_hash: str = Field(..., description="Hash del insight")
    action: str = Field(
        ...,
        description="Acción: snooze, dismiss, resolve",
        pattern="^(snooze|dismiss|resolve)$"
    )
    rule_code: Optional[str] = Field(
        None,
        description="Código de la regla (opcional, para logging)"
    )
    notes: Optional[str] = Field(
        None,
        max_length=500,
        description="Notas opcionales del usuario"
    )
    snooze_days: int = Field(
        default=7,
        ge=1,
        le=90,
        description="Días de snooze (solo si action=snooze)"
    )

    class Config:
        json_schema_extra = {
            "example": {
                "insight_hash": "a1b2c3d4e5f6g7h8",
                "action": "snooze",
                "rule_code": "STOCK_001",
                "notes": "Pedido ya realizado, llegará mañana",
                "snooze_days": 7,
            }
        }


class InsightFeedbackResponse(BaseModel):
    """Schema para respuesta de feedback."""

    success: bool = Field(..., description="Si el feedback se registró correctamente")
    message: str = Field(..., description="Mensaje descriptivo")
    feedback_id: Optional[str] = Field(None, description="ID del feedback creado")


class FeedbackHistoryItem(BaseModel):
    """Schema para un item del historial de feedback."""

    id: str = Field(..., description="ID del feedback")
    rule_code: str = Field(..., description="Código de la regla")
    insight_hash: str = Field(..., description="Hash del insight")
    action: str = Field(..., description="Acción tomada")
    snoozed_until: Optional[datetime] = Field(None, description="Fecha fin del snooze")
    notes: Optional[str] = Field(None, description="Notas del usuario")
    created_at: datetime = Field(..., description="Fecha de creación")
    is_active: bool = Field(..., description="Si está activamente suprimiendo")


class FeedbackHistoryResponse(BaseModel):
    """Schema para respuesta de historial de feedback."""

    feedbacks: list[FeedbackHistoryItem] = Field(
        default_factory=list,
        description="Lista de feedbacks ordenados por fecha DESC"
    )
    total_count: int = Field(default=0, description="Total de feedbacks")


class RuleInfo(BaseModel):
    """Schema para información de una regla."""

    rule_code: str = Field(..., description="Código único de la regla")
    category: str = Field(..., description="Categoría de la regla")
    severity: str = Field(..., description="Severidad por defecto")
    default_config: dict[str, Any] = Field(
        default_factory=dict,
        description="Configuración por defecto con umbrales"
    )


class RulesConfigResponse(BaseModel):
    """Schema para respuesta de configuración de reglas."""

    rules: list[RuleInfo] = Field(
        default_factory=list,
        description="Lista de reglas disponibles"
    )
    total_rules: int = Field(default=0, description="Total de reglas")


class TriggerAuditRequest(BaseModel):
    """Schema para forzar ejecución de audit."""

    force: bool = Field(
        default=True,
        description="Si True, ignora cache y re-ejecuta"
    )
    custom_config: Optional[dict[str, dict[str, Any]]] = Field(
        None,
        description="Configuración personalizada por regla {rule_code: config}"
    )


class TriggerAuditResponse(BaseModel):
    """Schema para respuesta de trigger audit."""

    status: str = Field(..., description="Estado: initiated, completed, error")
    message: str = Field(..., description="Mensaje descriptivo")
    insights_count: Optional[int] = Field(None, description="Número de insights generados")
    total_opportunity: Optional[float] = Field(None, description="Oportunidad total")
    duration_ms: Optional[int] = Field(None, description="Duración del audit")
