# backend/app/schemas/feedback.py
"""
Schemas Pydantic para API de feedback del clasificador.

Issue #457: M4 Feedback Loop
"""

from datetime import datetime
from typing import Optional
from uuid import UUID

from pydantic import BaseModel, Field

from app.models.product_correction import (
    CorrectionType,
    OutlierReason,
    OutlierStatus,
    PredictionSource,
)


# === Request Schemas ===


class ApproveRequest(BaseModel):
    """Request para aprobar clasificación actual."""

    sales_enrichment_id: UUID
    reviewer_notes: Optional[str] = None


class CorrectRequest(BaseModel):
    """Request para corregir clasificación."""

    sales_enrichment_id: UUID
    corrected_category: str = Field(..., min_length=1, max_length=100)
    reviewer_notes: Optional[str] = None


class OutlierRequest(BaseModel):
    """Request para marcar como outlier."""

    sales_enrichment_id: UUID
    reviewer_notes: Optional[str] = None
    # Issue #457: Outlier investigation fields
    outlier_reason: Optional[OutlierReason] = OutlierReason.OTRO


class SkipRequest(BaseModel):
    """Request para saltar un producto temporalmente."""

    sales_enrichment_id: UUID
    reviewer_notes: Optional[str] = None


# Issue #457: Outlier Investigation Schemas

class OutlierResolveRequest(BaseModel):
    """Request para resolver un outlier."""

    new_status: OutlierStatus
    resolution_notes: Optional[str] = None
    new_category: Optional[str] = None  # Si se corrigió, qué categoría


class OutlierStatusUpdateRequest(BaseModel):
    """Request para actualizar estado de outlier."""

    new_status: OutlierStatus


# === Response Schemas ===


class ReviewQueueItem(BaseModel):
    """Item en la cola de revisión."""

    id: str
    product_name: Optional[str]
    current_category: Optional[str]
    confidence_score: Optional[float]
    prediction_source: str
    brand_detected: Optional[str]
    sales_data_id: Optional[str]
    created_at: Optional[str]

    class Config:
        from_attributes = True


class ReviewQueueBreakdown(BaseModel):
    """Breakdown de prioridades en la cola."""

    P1_fallback: int = 0
    P2_low_confidence: int = 0
    P3_other: int = 0


class ReviewQueueResponse(BaseModel):
    """Response para cola de revisión."""

    items: list[ReviewQueueItem]
    total: int
    breakdown: ReviewQueueBreakdown
    offset: int
    limit: int


class CorrectionResponse(BaseModel):
    """Response para operaciones de corrección."""

    id: int
    sales_enrichment_id: UUID
    product_name: str
    predicted_category: str
    prediction_source: str
    confidence_score: float
    correction_type: str
    corrected_category: Optional[str]
    reviewer_notes: Optional[str]
    created_at: datetime

    class Config:
        from_attributes = True


class CorrectionStats(BaseModel):
    """Estadísticas de correcciones."""

    total_corrections: int
    approved: int
    corrected: int
    outliers: int
    classifier_accuracy: Optional[float] = None


class AvailableCategory(BaseModel):
    """Categoría disponible para corrección."""

    slug: str
    name: str
    description: Optional[str] = None


# Issue #457: Outlier Investigation Response Schemas

class OutlierItem(BaseModel):
    """Item en la lista de outliers."""

    id: int
    product_name: str
    predicted_category: str
    prediction_source: Optional[str]
    confidence_score: float
    outlier_reason: Optional[str]
    outlier_status: Optional[str]
    reviewer_notes: Optional[str]
    outlier_resolution_notes: Optional[str]
    created_at: Optional[str]
    outlier_resolved_at: Optional[str]

    class Config:
        from_attributes = True


class OutlierStatusBreakdown(BaseModel):
    """Breakdown por status de outliers."""

    pending_review: int = 0
    investigating: int = 0
    resolved_category: int = 0
    resolved_corrected: int = 0
    true_outlier: int = 0


class OutlierReasonBreakdown(BaseModel):
    """Breakdown por razón de outliers."""

    pack_promocional: int = 0
    descatalogado: int = 0
    ambiguo: int = 0
    nueva_categoria: int = 0
    otro: int = 0


class OutliersResponse(BaseModel):
    """Response para lista de outliers."""

    items: list[OutlierItem]
    total: int
    status_breakdown: dict[str, int]
    reason_breakdown: dict[str, int]
    offset: int
    limit: int


class OutlierReasonEnum(BaseModel):
    """Info sobre razones de outlier disponibles."""

    value: str
    label: str


class OutlierStatusEnum(BaseModel):
    """Info sobre estados de outlier disponibles."""

    value: str
    label: str
