# backend/app/schemas/employee.py
"""
Schemas Pydantic para Employee API (Issue #402)

Define los modelos de request/response para el sistema de filtrado por empleado
en análisis de genéricos. Feature exclusiva de plan PRO+.

DECISIONES APROBADAS:
- Filtrar solo por employee_name (sin employee_code)
- Soporte para ventas sin empleado asignado (__sin_empleado__)
- Cache invalidado por current-upload-id (storage_type=session)
- UX: Dropdown deshabilitado para plan FREE
- Comportamiento: Mostrar TODOS por default si no hay selección
"""

from datetime import date, datetime
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, Field, field_validator


# ========== RESPONSE SCHEMAS ==========


class EmployeeResponse(BaseModel):
    """
    Response para un empleado de la farmacia.

    NOTA: Solo usa employee_name (decisión #2 aprobada).
    El valor especial "__sin_empleado__" representa ventas sin empleado asignado.
    """

    name: str = Field(
        ...,
        description="Nombre del empleado (o '__sin_empleado__' para ventas sin asignar)",
        json_schema_extra={"example": "María García"},
    )
    sales_count: int = Field(
        ...,
        description="Número total de ventas registradas",
        json_schema_extra={"example": 1250},
    )
    last_sale_date: date = Field(
        ...,
        description="Fecha de la última venta registrada",
        json_schema_extra={"example": "2025-11-05"},
    )

    class Config:
        from_attributes = True  # Permite crear desde ORM models


class EmployeeListResponse(BaseModel):
    """
    Response para lista de empleados de una farmacia.

    Usado por GET /api/v1/employees/{pharmacy_id}.
    """

    employees: List[EmployeeResponse] = Field(
        ...,
        description="Lista de empleados ordenada por sales_count descendente",
    )
    total_count: int = Field(
        ...,
        description="Número total de empleados",
        json_schema_extra={"example": 5},
    )

    class Config:
        from_attributes = True
        json_schema_extra = {
            "example": {
                "employees": [
                    {
                        "name": "María García",
                        "sales_count": 1250,
                        "last_sale_date": "2025-11-05",
                    },
                    {
                        "name": "Juan Pérez",
                        "sales_count": 890,
                        "last_sale_date": "2025-11-04",
                    },
                    {
                        "name": "__sin_empleado__",
                        "sales_count": 320,
                        "last_sale_date": "2025-11-03",
                    },
                ],
                "total_count": 3,
            }
        }


# ========== REQUEST SCHEMAS ==========


class GenericOpportunitiesWithEmployeesRequest(BaseModel):
    """
    Request extendido para generic_opportunities con filtro de empleados.

    Combina filtros de partners + empleados para análisis granular.

    DECISIÓN UX #2 (Aprobada):
    - Si selected_employee_names = [] → Mostrar TODOS los empleados (default)
    - Si selected_employee_names = ["María García"] → Filtrar solo ese empleado
    - Puede incluir "__sin_empleado__" para ventas sin empleado asignado
    """

    # Filtros de partners (existente)
    selected_partners: List[str] = Field(
        default_factory=list,
        description="Códigos de laboratorios partners",
    )

    # Filtros de empleados (NUEVO - Issue #402)
    selected_employee_names: List[str] = Field(
        default_factory=list,
        description="Nombres de empleados a filtrar (PRO feature). "
        "Lista vacía = TODOS los empleados. "
        "Puede incluir '__sin_empleado__' para ventas sin empleado.",
        json_schema_extra={"example": ["María García", "Juan Pérez"]},
    )

    # Parámetros de análisis (existente)
    analysis_months: Optional[int] = Field(
        None,
        ge=1,
        le=24,
        description="Período de análisis en meses",
    )
    limit: int = Field(
        1000,
        ge=1,
        le=10000,
        description="Límite de resultados",
    )
    offset: int = Field(
        0,
        ge=0,
        description="Offset para paginación",
    )

    @field_validator("selected_employee_names")
    def validate_employees_list(cls, v):
        """Normalizar y validar lista de nombres de empleados"""
        if v is None:
            return []
        if not isinstance(v, list):
            raise ValueError("selected_employee_names debe ser una lista")
        # Eliminar vacíos y duplicados (mantener __sin_empleado__ si existe)
        return list(set(name.strip() for name in v if name and name.strip()))


class PartnerAnalysisWithEmployeesRequest(BaseModel):
    """
    Request extendido para partner_analysis con filtro de empleados.

    Permite combinar análisis por partners con filtrado por empleados.
    """

    # Filtros de partners (existente)
    selected_partner_codes: List[str] = Field(
        ...,
        description="Códigos de laboratorios partners",
    )

    # Filtros de empleados (NUEVO - Issue #402)
    selected_employee_names: List[str] = Field(
        default_factory=list,
        description="Nombres de empleados a filtrar (PRO feature). "
        "Lista vacía = TODOS los empleados.",
        json_schema_extra={"example": ["María García", "Juan Pérez"]},
    )

    # Parámetros de análisis (existente)
    period_months: Optional[int] = Field(
        None,
        ge=1,
        le=24,
        description="Período de análisis en meses",
    )
    discount_percentage: Optional[float] = Field(
        None,
        ge=0.0,
        le=100.0,
        description="Porcentaje de descuento esperado",
    )

    @field_validator("selected_employee_names")
    def validate_employees(cls, v):
        """Normalizar lista de nombres de empleados"""
        if v is None:
            return []
        if not isinstance(v, list):
            raise ValueError("selected_employee_names debe ser una lista")
        return list(set(name.strip() for name in v if name and name.strip()))
