# backend/app/schemas/curated_groups.py
"""
Schemas Pydantic para Grupos Intercambiables Curados.

Issue #521: Admin UI para gestión de grupos curados.
ADR-004: Simplificación del Sistema de Clasificación
"""

from datetime import datetime
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, Field, field_validator, model_validator


# === GROUP SCHEMAS ===

class CuratedGroupBase(BaseModel):
    """Base schema for curated groups."""
    group_name: str = Field(..., min_length=1, max_length=100, description="Nombre del grupo")
    description: Optional[str] = Field(None, max_length=500, description="Descripción")
    necesidad_l1: Optional[str] = Field(None, max_length=50, description="Categoría L1")
    subcategory_l2: Optional[str] = Field(None, max_length=50, description="Subcategoría L2")


class CuratedGroupCreate(CuratedGroupBase):
    """Schema for creating a new curated group."""
    group_slug: Optional[str] = Field(None, max_length=100, description="Slug (auto-generated if not provided)")

    @field_validator("group_slug", mode="before")
    @classmethod
    def generate_slug(cls, v, _):
        """Auto-generate slug from group_name if not provided."""
        if v:
            return v
        # Slug will be generated in the service from group_name
        return None


class CuratedGroupUpdate(BaseModel):
    """Schema for updating a curated group."""
    group_name: Optional[str] = Field(None, min_length=1, max_length=100)
    description: Optional[str] = Field(None, max_length=500)
    necesidad_l1: Optional[str] = Field(None, max_length=50)
    subcategory_l2: Optional[str] = Field(None, max_length=50)
    is_active: Optional[bool] = None


class CuratedGroupResponse(CuratedGroupBase):
    """Schema for curated group response."""
    id: UUID
    group_slug: str
    source: str
    is_active: bool
    product_count: int
    brand_count: int
    total_sales_amount: Optional[float] = 0
    created_at: datetime
    updated_at: datetime
    created_by: Optional[str] = None

    class Config:
        from_attributes = True


class CuratedGroupListResponse(BaseModel):
    """Schema for list of curated groups."""
    groups: List[CuratedGroupResponse]
    total: int
    page: int = 1
    page_size: int = 20


# === MEMBER SCHEMAS ===

class GroupMemberBase(BaseModel):
    """Base schema for group members."""
    ean13: Optional[str] = Field(None, max_length=13, description="Código EAN13")
    codigo_nacional: Optional[str] = Field(None, max_length=20, description="Código nacional")
    product_name: Optional[str] = Field(None, max_length=255, description="Nombre del producto")
    detected_brand: Optional[str] = Field(None, max_length=100, description="Marca detectada")


class GroupMemberCreate(GroupMemberBase):
    """Schema for adding a member to a group."""

    @model_validator(mode="after")
    def validate_at_least_one_code(self):
        """Ensure at least one product code is provided."""
        if not self.ean13 and not self.codigo_nacional:
            raise ValueError("Se requiere al menos ean13 o codigo_nacional")
        return self


class GroupMemberBulkCreate(BaseModel):
    """Schema for bulk adding members."""
    members: List[GroupMemberCreate] = Field(..., min_length=1, max_length=100)


class GroupMemberResponse(GroupMemberBase):
    """Schema for group member response."""
    id: UUID
    group_id: UUID
    added_at: datetime
    added_by: Optional[str] = None

    class Config:
        from_attributes = True


class GroupMemberListResponse(BaseModel):
    """Schema for list of group members."""
    members: List[GroupMemberResponse]
    total: int


# === SEARCH/FILTER SCHEMAS ===

class ProductSearchResult(BaseModel):
    """Schema for product search results (for adding to groups)."""
    ean13: Optional[str] = None
    codigo_nacional: Optional[str] = None
    product_name: str
    detected_brand: Optional[str] = None
    sales_count: int = 0
    total_sales: float = 0


class CuratedGroupFilter(BaseModel):
    """Schema for filtering curated groups."""
    necesidad_l1: Optional[str] = None
    is_active: Optional[bool] = None
    source: Optional[str] = None
    search: Optional[str] = None
    order_by: str = Field("total_sales_amount", description="Order by field")
    order_desc: bool = True
    page: int = 1
    page_size: int = 20


# === STATS SCHEMAS ===

class CuratedGroupStats(BaseModel):
    """Schema for curated groups statistics."""
    total_groups: int
    active_groups: int
    legacy_groups: int
    manual_groups: int
    total_products: int
    total_brands: int
    total_sales: float


class CuratedGroupDetailResponse(CuratedGroupResponse):
    """Schema for detailed group response with members."""
    members: List[GroupMemberResponse] = []
    top_brands: List[dict] = []
