"""
Pydantic schemas for clustering visualization (Issue #458).

Schemas for UMAP 2D visualization, ghost points, and cluster detail endpoints.
"""

from datetime import datetime
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, Field, computed_field


class UMAPPoint(BaseModel):
    """Single product point in UMAP 2D space."""

    id: UUID
    product_name: str
    umap_x: float
    umap_y: float

    # Cluster info
    cluster_id: Optional[int] = None
    cluster_name: Optional[str] = None

    # Classification
    necesidad: Optional[str] = None
    detected_brand: Optional[str] = None
    ml_category: Optional[str] = None
    ml_confidence: Optional[float] = None

    # Validation status
    human_verified: bool = False

    # "Lobo con piel de cordero" detector (Issue #458 comment)
    centroid_distance_zscore: Optional[float] = Field(
        default=None,
        description="Z-score of distance to cluster centroid. High value = potential outlier"
    )

    @computed_field
    @property
    def risk_level(self) -> str:
        """
        Visual risk indicator based on centroid distance.

        High similarity + High Z-Score = "Wolf in sheep's clothing"
        - "high": Z-Score > 2.0 (outlier potential)
        - "medium": Z-Score 1.5-2.0 (frontier)
        - "low": Z-Score < 1.5 (core)
        """
        if self.centroid_distance_zscore is None:
            return "unknown"
        if self.centroid_distance_zscore > 2.0:
            return "high"
        elif self.centroid_distance_zscore > 1.5:
            return "medium"
        else:
            return "low"

    class Config:
        from_attributes = True


class GhostPoint(BaseModel):
    """
    Ghost point representing a moved product's original position.

    Ghost points are shown as gray shadows in the UMAP scatter to provide
    immediate visual feedback when products are moved between clusters.
    """

    id: UUID
    product_name: str
    original_umap_x: float
    original_umap_y: float
    moved_to_cluster_id: Optional[int] = None
    moved_to_cluster_name: Optional[str] = None
    moved_at: datetime


class UMAPMetadata(BaseModel):
    """Metadata about the UMAP model and dataset."""

    version: str = Field(description="UMAP model version (e.g., 'v1.0')")
    total_products: int
    products_with_coords: int
    last_trained: Optional[datetime] = None

    # Cluster stats
    total_clusters: int = 0
    locked_clusters: int = 0

    # Coverage
    @computed_field
    @property
    def coverage_percent(self) -> float:
        """Percentage of products with UMAP coordinates."""
        if self.total_products == 0:
            return 0.0
        return round(100 * self.products_with_coords / self.total_products, 1)


class UMAPDataResponse(BaseModel):
    """Response for /api/v1/clustering/umap-data endpoint."""

    points: List[UMAPPoint]
    metadata: UMAPMetadata

    # Optional: ghost points from current session (passed from frontend)
    # These are not stored in DB, just echoed back for consistency
    ghost_points: List[GhostPoint] = []


class UMAPDataRequest(BaseModel):
    """Request parameters for filtering UMAP data."""

    necesidad: Optional[str] = Field(
        default=None,
        description="Filter by NECESIDAD category"
    )
    cluster_id: Optional[int] = Field(
        default=None,
        description="Filter by specific cluster"
    )
    verified_only: bool = Field(
        default=False,
        description="Only return human-verified products"
    )
    min_confidence: Optional[float] = Field(
        default=None,
        ge=0.0,
        le=1.0,
        description="Minimum ML confidence threshold"
    )
    limit: int = Field(
        default=5000,
        le=10000,
        description="Maximum number of points to return"
    )


# === Cluster Detail Schemas ===

class ClusterSummary(BaseModel):
    """Summary info for a cluster."""

    id: int
    name: str
    status: str = Field(description="LOCKED, PROVISIONAL, or ARCHIVED")
    product_count: int
    brand_count: int
    sales_amount: float = 0.0
    purity_score: float = Field(
        description="1 - CV(distances to centroid). Low = heterogeneous cluster"
    )
    silhouette_score: Optional[float] = None


class ClusterDetailResponse(BaseModel):
    """Detailed cluster information with products."""

    cluster: ClusterSummary
    products: List[UMAPPoint]
    top_brands: List[str] = []
    suggested_splits: List[str] = Field(
        default=[],
        description="Suggested sub-clusters if purity is low"
    )


# === Validation Queue Schemas ===

class ValidationQueueItem(BaseModel):
    """Single item in the validation queue ("Tinder for products")."""

    id: UUID
    product_name: str
    detected_brand: Optional[str] = None

    # Suggested classification
    suggested_cluster_id: int
    suggested_cluster_name: str
    similarity_score: float

    # Alternative suggestion (second best match)
    alternative_cluster_id: Optional[int] = None
    alternative_cluster_name: Optional[str] = None
    alternative_similarity: Optional[float] = None

    # Centroid distance for outlier detection
    centroid_distance_zscore: float

    # Similar validated products in suggested cluster (for context)
    similar_validated_products: List[str] = Field(
        default=[],
        max_items=3,
        description="Up to 3 similar products already validated in this cluster"
    )

    # Priority (P1, P2, P3)
    priority: str = Field(description="P1 (fallback), P2 (low confidence), P3 (frontier)")


class ValidationQueueResponse(BaseModel):
    """Response for validation queue endpoint."""

    items: List[ValidationQueueItem]
    total_pending: int
    filters_applied: dict = {}


class ValidationAction(BaseModel):
    """Action taken on a validation queue item."""

    product_id: UUID
    action: str = Field(description="APPROVE, MOVE, SKIP")
    target_cluster_id: Optional[int] = Field(
        default=None,
        description="Required if action is MOVE"
    )
    notes: Optional[str] = None


# === Taxonomy Labeling Schemas (Issue #462) ===

class TaxonomyLabelRequest(BaseModel):
    """Request para etiquetado de clusters."""

    force: bool = Field(
        default=False,
        description="Forzar re-etiquetado (ignora threshold 10%)"
    )
    cluster_ids: Optional[List[UUID]] = Field(
        default=None,
        description="Lista específica de clusters a etiquetar (opcional)"
    )


class TaxonomyLabelStatusResponse(BaseModel):
    """Response para estado de etiquetado de clusters."""

    total_clusters: int
    labeled_clusters: int
    unlabeled_clusters: int
    stale_clusters: int = Field(
        description="Clusters etiquetados hace más de 7 días"
    )
    last_labeled_at: Optional[datetime] = None
    coverage_percent: float
    tier1_distribution: dict = Field(
        default={},
        description="Distribución de clusters por macro-categoría (tier1)"
    )
