# backend/app/schemas/cluster_management.py
"""
Pydantic schemas for Cluster Management (Issue #464).

Schemas for split/merge operations on VentaLibre product clusters.
Clusters are represented by `ml_category` field in ProductCatalogVentaLibre.
"""

from datetime import datetime
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, Field, field_validator


# === Request Schemas ===


class ClusterSplitRequest(BaseModel):
    """
    Request to split a cluster by moving selected products to a new category.

    This operation moves specific products from their current category
    (source_category) to a new or existing category (new_category).
    """

    source_category: str = Field(
        ...,
        min_length=1,
        max_length=100,
        description="Current category of the products to move"
    )
    product_ids: List[UUID] = Field(
        ...,
        min_length=1,
        description="List of product IDs to move to the new category"
    )
    new_category: str = Field(
        ...,
        min_length=1,
        max_length=100,
        description="Target category (must be a valid NecesidadEspecifica value)"
    )
    notes: Optional[str] = Field(
        default=None,
        max_length=500,
        description="Notes explaining the reason for the split"
    )

    @field_validator("product_ids")
    @classmethod
    def validate_product_ids_not_empty(cls, v: List[UUID]) -> List[UUID]:
        """Validate that at least one product is provided."""
        if not v:
            raise ValueError("At least one product ID is required")
        return v


class ClusterMergeRequest(BaseModel):
    """
    Request to merge multiple categories into a single destination category.

    This operation moves ALL products from source categories to
    the destination category.
    """

    source_categories: List[str] = Field(
        ...,
        min_length=1,
        description="List of categories to merge (will be emptied)"
    )
    destination_category: str = Field(
        ...,
        min_length=1,
        max_length=100,
        description="Target category to merge into (must be valid NecesidadEspecifica)"
    )
    notes: Optional[str] = Field(
        default=None,
        max_length=500,
        description="Notes explaining the reason for the merge"
    )

    @field_validator("source_categories")
    @classmethod
    def validate_source_categories(cls, v: List[str]) -> List[str]:
        """Validate source categories."""
        if not v:
            raise ValueError("At least one source category is required")
        # Remove duplicates while preserving order
        seen = set()
        unique = []
        for cat in v:
            if cat not in seen:
                seen.add(cat)
                unique.append(cat)
        return unique


# === Preview Schemas (for dry-run before execution) ===


class ProductPreview(BaseModel):
    """Preview info for a single product."""

    id: UUID
    product_name: str
    current_category: Optional[str] = None
    detected_brand: Optional[str] = None
    total_sales_count: int = 0
    human_verified: bool = False


class SplitPreviewResponse(BaseModel):
    """
    Preview response for split operation.

    Shows what will happen without executing the operation.
    """

    source_category: str
    new_category: str
    is_valid_category: bool = Field(
        description="Whether new_category is a valid NecesidadEspecifica value"
    )

    # Products to be moved
    products_to_move: List[ProductPreview]
    products_count: int

    # Impact summary
    source_category_remaining: int = Field(
        description="Products remaining in source category after split"
    )
    total_sales_impact: int = Field(
        description="Total sales count of products being moved"
    )

    # Warnings
    warnings: List[str] = Field(
        default=[],
        description="Any warnings about the operation"
    )


class CategoryImpact(BaseModel):
    """Impact info for a single source category in merge."""

    category: str
    product_count: int
    total_sales: int
    verified_count: int


class MergePreviewResponse(BaseModel):
    """
    Preview response for merge operation.

    Shows what will happen without executing the operation.
    """

    source_categories: List[str]
    destination_category: str
    is_valid_destination: bool = Field(
        description="Whether destination_category is a valid NecesidadEspecifica value"
    )

    # Impact per source category
    source_impacts: List[CategoryImpact]

    # Totals
    total_products_to_merge: int
    total_sales_impact: int
    total_verified_products: int

    # Destination info
    destination_current_count: int = Field(
        description="Current products in destination category"
    )
    destination_after_merge: int = Field(
        description="Products in destination after merge"
    )

    # Warnings
    warnings: List[str] = Field(
        default=[],
        description="Any warnings about the operation"
    )


# === Response Schemas ===


class ClusterSplitResponse(BaseModel):
    """Response after executing a cluster split."""

    success: bool
    message: str

    # Execution details
    source_category: str
    new_category: str
    products_moved: int
    enrichments_synced: int = Field(
        description="Number of SalesEnrichment records updated"
    )

    # Audit info
    audit_log_id: Optional[str] = None
    executed_at: datetime
    executed_by: Optional[str] = None

    # Post-operation stats
    source_category_remaining: int
    new_category_total: int


class ClusterMergeResponse(BaseModel):
    """Response after executing a cluster merge."""

    success: bool
    message: str

    # Execution details
    source_categories: List[str]
    destination_category: str
    total_products_merged: int
    enrichments_synced: int = Field(
        description="Number of SalesEnrichment records updated"
    )

    # Per-source breakdown
    products_per_source: dict = Field(
        default={},
        description="Number of products merged from each source category"
    )

    # Audit info
    audit_log_id: Optional[str] = None
    executed_at: datetime
    executed_by: Optional[str] = None

    # Post-operation stats
    destination_total: int


# === Statistics Schemas ===


class ClusterStats(BaseModel):
    """Statistics for a single cluster (category)."""

    category: str
    display_name: str

    # Product counts
    total_products: int
    verified_products: int
    pending_verification: int

    # Sales metrics
    total_sales_count: int = Field(
        description="Sum of total_sales_count for all products"
    )
    pharmacies_count: int = Field(
        description="Number of distinct pharmacies with products in this cluster"
    )

    # Brand distribution
    top_brands: List[str] = Field(
        default=[],
        description="Top 5 brands in this cluster"
    )
    brand_count: int = Field(
        default=0,
        description="Number of distinct brands"
    )

    # Quality metrics
    avg_confidence: Optional[float] = Field(
        default=None,
        description="Average ML confidence for products"
    )
    verification_rate: float = Field(
        default=0.0,
        description="Percentage of products verified"
    )


class ClusterListResponse(BaseModel):
    """Response with list of clusters and their stats."""

    clusters: List[ClusterStats]
    total_clusters: int
    total_products: int
    total_verified: int


class CategoryValidation(BaseModel):
    """Result of category name validation."""

    category: str
    is_valid: bool
    display_name: Optional[str] = None
    parent_category: Optional[str] = None
    reason: Optional[str] = Field(
        default=None,
        description="Explanation if invalid"
    )
