# backend/app/schemas/category_alias.py
"""
Schemas para CategoryAlias.

Issue #459: Category normalization en base de datos.
"""

from datetime import datetime
from typing import Optional

from pydantic import BaseModel, Field, field_validator


class CategoryAliasBase(BaseModel):
    """Campos compartidos para CategoryAlias."""

    source_category: str = Field(
        ...,
        min_length=2,
        max_length=100,
        description="Categoría del clasificador (input a normalizar)"
    )
    target_category: str = Field(
        ...,
        min_length=2,
        max_length=100,
        description="Categoría normalizada (output)"
    )
    reason: Optional[str] = Field(
        None,
        max_length=500,
        description="Razón del alias"
    )


class CategoryAliasCreate(CategoryAliasBase):
    """Schema para crear un CategoryAlias."""

    @field_validator("source_category", "target_category", mode="before")
    @classmethod
    def normalize_category(cls, v: str) -> str:
        """Normalizar a lowercase y strip."""
        if isinstance(v, str):
            return v.lower().strip()
        return v


class CategoryAliasUpdate(BaseModel):
    """Schema para actualizar un CategoryAlias (todos los campos opcionales)."""

    source_category: Optional[str] = Field(None, min_length=2, max_length=100)
    target_category: Optional[str] = Field(None, min_length=2, max_length=100)
    reason: Optional[str] = Field(None, max_length=500)
    is_active: Optional[bool] = None

    @field_validator("source_category", "target_category", mode="before")
    @classmethod
    def normalize_category(cls, v: Optional[str]) -> Optional[str]:
        """Normalizar a lowercase y strip."""
        if isinstance(v, str):
            return v.lower().strip()
        return v


class CategoryAliasResponse(BaseModel):
    """Schema de respuesta para CategoryAlias."""

    id: int
    source_category: str
    target_category: str
    is_active: bool
    reason: Optional[str] = None
    usage_count: int = 0
    last_used_at: Optional[datetime] = None
    created_at: datetime

    model_config = {"from_attributes": True}


class CategoryAliasListResponse(BaseModel):
    """Schema de respuesta para lista de CategoryAlias."""

    items: list[CategoryAliasResponse]
    total: int
    offset: int
    limit: int
    stats: dict


class CategoryAliasStatsResponse(BaseModel):
    """Estadísticas de CategoryAlias."""

    total: int
    active: int
    inactive: int
    total_usage: int
    most_used: Optional[str] = None
    least_used: Optional[str] = None
