# backend/app/schemas/brand_alias.py
"""
Schemas para BrandAlias.

Issue #XXX: Sistema de corrección de marcas para BrandDetectionService.
"""

from datetime import datetime
from typing import Optional

from pydantic import BaseModel, Field, field_validator, model_validator

from app.models.brand_alias import BrandAliasAction


class BrandAliasBase(BaseModel):
    """Campos compartidos para BrandAlias."""

    source_brand: str = Field(
        ...,
        min_length=2,
        max_length=100,
        description="Marca detectada (input a normalizar/excluir)",
    )
    target_brand: Optional[str] = Field(
        None,
        max_length=100,
        description="Marca normalizada (output). NULL si action=EXCLUDE",
    )
    action: BrandAliasAction = Field(
        default=BrandAliasAction.ALIAS,
        description="Acción: 'alias' (renombrar) o 'exclude' (eliminar detección)",
    )
    reason: Optional[str] = Field(
        None,
        max_length=500,
        description="Razón del alias/exclusión",
    )


class BrandAliasCreate(BrandAliasBase):
    """Schema para crear un BrandAlias."""

    @field_validator("source_brand", mode="before")
    @classmethod
    def normalize_source_brand(cls, v: str) -> str:
        """Normalizar source_brand a lowercase y strip."""
        if isinstance(v, str):
            return v.lower().strip()
        return v

    @field_validator("target_brand", mode="before")
    @classmethod
    def normalize_target_brand(cls, v: Optional[str]) -> Optional[str]:
        """Normalizar target_brand a lowercase y strip."""
        if isinstance(v, str):
            return v.lower().strip() if v.strip() else None
        return v

    @model_validator(mode="after")
    def validate_action_target(self) -> "BrandAliasCreate":
        """Validar coherencia entre action y target_brand."""
        if self.action == BrandAliasAction.ALIAS:
            if not self.target_brand:
                raise ValueError(
                    "target_brand es requerido cuando action='alias'"
                )
        elif self.action == BrandAliasAction.EXCLUDE:
            # Para exclude, forzar target_brand a None
            self.target_brand = None
        return self


class BrandAliasUpdate(BaseModel):
    """Schema para actualizar un BrandAlias (todos los campos opcionales)."""

    source_brand: Optional[str] = Field(None, min_length=2, max_length=100)
    target_brand: Optional[str] = Field(None, max_length=100)
    action: Optional[BrandAliasAction] = None
    reason: Optional[str] = Field(None, max_length=500)
    is_active: Optional[bool] = None

    @field_validator("source_brand", mode="before")
    @classmethod
    def normalize_source_brand(cls, v: Optional[str]) -> Optional[str]:
        """Normalizar source_brand a lowercase y strip."""
        if isinstance(v, str):
            return v.lower().strip()
        return v

    @field_validator("target_brand", mode="before")
    @classmethod
    def normalize_target_brand(cls, v: Optional[str]) -> Optional[str]:
        """Normalizar target_brand a lowercase y strip."""
        if isinstance(v, str):
            return v.lower().strip() if v.strip() else None
        return v


class BrandAliasResponse(BaseModel):
    """Schema de respuesta para BrandAlias."""

    id: int
    source_brand: str
    target_brand: Optional[str] = None
    action: BrandAliasAction
    is_active: bool
    reason: Optional[str] = None
    usage_count: int = 0
    last_used_at: Optional[datetime] = None
    created_at: datetime
    # Count de productos afectados (para re-proceso)
    affected_count: int = 0

    model_config = {"from_attributes": True}


class BrandAliasListResponse(BaseModel):
    """Schema de respuesta para lista de BrandAlias."""

    items: list[BrandAliasResponse]
    total: int
    offset: int
    limit: int
    stats: dict


class BrandAliasStatsResponse(BaseModel):
    """Estadísticas de BrandAlias."""

    total: int
    active: int
    inactive: int
    aliases: int  # Cuántos son action=ALIAS
    excludes: int  # Cuántos son action=EXCLUDE
    total_usage: int
    most_used: Optional[str] = None
    least_used: Optional[str] = None


class BrandAliasReprocessResponse(BaseModel):
    """Respuesta del endpoint de re-proceso."""

    alias_id: int
    source_brand: str
    target_brand: Optional[str] = None
    action: BrandAliasAction
    updated_count: int
    message: str
