# backend/app/schemas/auth.py
"""
Pydantic schemas for authentication endpoints
"""

from datetime import datetime
from typing import Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, EmailStr, Field, field_validator


# Request schemas
class UserLogin(BaseModel):
    """Schema for user login request"""

    email: EmailStr
    password: str = Field(..., min_length=6)


class UserRegister(BaseModel):
    """Schema for user registration request"""

    email: EmailStr
    username: str = Field(..., min_length=3, max_length=50)
    password: str = Field(..., min_length=8)
    full_name: Optional[str] = Field(None, max_length=255)
    phone: Optional[str] = Field(None, max_length=20)
    dni_nie: Optional[str] = Field(None, max_length=20, description="DNI/NIE español")

    # Datos de la farmacia (se crea automáticamente durante el registro)
    pharmacy_name: str = Field(..., min_length=3, max_length=255, description="Nombre de la farmacia")
    pharmacy_code: Optional[str] = Field(None, max_length=50, description="Código de la farmacia")
    pharmacy_email: Optional[EmailStr] = Field(None, description="Email de contacto de la farmacia")
    pharmacy_address: Optional[str] = Field(None, max_length=500, description="Dirección de la farmacia")
    pharmacy_city: Optional[str] = Field(None, max_length=100, description="Ciudad")
    pharmacy_postal_code: Optional[str] = Field(None, max_length=10, description="Código postal")
    pharmacy_phone: Optional[str] = Field(None, max_length=20, description="Teléfono de la farmacia")

    # DEPRECATED: pharmacy_id ya no se usa, se crea automáticamente
    pharmacy_id: Optional[UUID] = None

    @field_validator("dni_nie")
    @classmethod
    def validate_dni_nie(cls, v: Optional[str]) -> Optional[str]:
        """Valida y normaliza DNI/NIE español"""
        if v is None:
            return v

        from app.utils.validators import sanitize_dni_nie

        cleaned = sanitize_dni_nie(v)
        if cleaned is None:
            raise ValueError("Formato de DNI/NIE inválido")
        return cleaned


class OAuthCallback(BaseModel):
    """Schema for OAuth callback data"""

    code: str
    state: Optional[str] = None


class PasswordResetRequest(BaseModel):
    """Schema for password reset request"""

    email: EmailStr


class PasswordReset(BaseModel):
    """Schema for password reset confirmation"""

    reset_token: str
    new_password: str = Field(..., min_length=8)


class EmailVerification(BaseModel):
    """Schema for email verification"""

    user_id: UUID
    verification_token: str


class RefreshTokenRequest(BaseModel):
    """Schema for refresh token request"""

    refresh_token: str


# Response schemas
class Token(BaseModel):
    """Schema for token response"""

    access_token: str
    refresh_token: Optional[str] = None
    token_type: str = "bearer"


class TokenPayload(BaseModel):
    """Schema for decoded token payload"""

    sub: str  # User ID
    email: str
    role: str
    exp: datetime
    type: str  # 'access' or 'refresh'


class UserResponse(BaseModel):
    """Schema for user response"""

    id: UUID
    email: str
    username: str
    full_name: Optional[str]
    phone: Optional[str]
    dni_nie: Optional[str]
    pharmacy_id: Optional[UUID]
    role: str
    subscription_plan: str = "free"  # Issue #420: FREE tier experience
    is_active: bool
    is_verified: bool
    created_at: datetime
    last_login: Optional[datetime]

    model_config = ConfigDict(from_attributes=True)


class LoginResponse(BaseModel):
    """Schema for login response"""

    user: UserResponse
    access_token: str
    refresh_token: str
    token_type: str = "bearer"


class RegisterResponse(BaseModel):
    """Schema for registration response"""

    user: UserResponse
    message: str = "Registration successful. Please verify your email."


class OAuthProviders(BaseModel):
    """Schema for available OAuth providers"""

    google_enabled: bool
    microsoft_enabled: bool
    google_url: Optional[str]
    microsoft_url: Optional[str]


class UserUpdate(BaseModel):
    """Schema for user profile update"""

    full_name: Optional[str] = Field(None, max_length=255)
    phone: Optional[str] = Field(None, max_length=20)
    dni_nie: Optional[str] = Field(None, max_length=20, description="DNI/NIE español")
    notify_uploads: Optional[bool] = None
    notify_errors: Optional[bool] = None
    notify_analysis: Optional[bool] = None

    # Campos para cambio de contraseña (Issue: Ajustes usuario)
    current_password: Optional[str] = Field(None, description="Contraseña actual (requerido para cambiar)")
    new_password: Optional[str] = Field(None, min_length=8, description="Nueva contraseña (mínimo 8 caracteres)")

    @field_validator("dni_nie")
    @classmethod
    def validate_dni_nie(cls, v: Optional[str]) -> Optional[str]:
        """Valida y normaliza DNI/NIE español"""
        if v is None:
            return v

        from app.utils.validators import sanitize_dni_nie

        cleaned = sanitize_dni_nie(v)
        if cleaned is None:
            raise ValueError("Formato de DNI/NIE inválido")
        return cleaned

    @field_validator("new_password")
    @classmethod
    def validate_new_password(cls, v: Optional[str]) -> Optional[str]:
        """
        Valida fortaleza de nueva contraseña según OWASP guidelines.

        Requisitos:
        - Mínimo 8 caracteres
        - Al menos 3 de 4 categorías: mayúsculas, minúsculas, números, especiales
        - No contraseñas comunes
        """
        if v is None:
            return v

        if len(v) < 8:
            raise ValueError("La contraseña debe tener al menos 8 caracteres")

        # OWASP: Requerir al menos 3 de 4 categorías
        has_upper = any(c.isupper() for c in v)
        has_lower = any(c.islower() for c in v)
        has_digit = any(c.isdigit() for c in v)
        has_special = any(c in "!@#$%^&*()_+-=[]{}|;:,.<>?" for c in v)

        categories = sum([has_upper, has_lower, has_digit, has_special])

        if categories < 3:
            raise ValueError(
                "La contraseña debe contener al menos 3 de: "
                "mayúsculas, minúsculas, números, caracteres especiales (!@#$%^&*)"
            )

        # Prevenir contraseñas comunes
        common_passwords = ["password", "12345678", "qwerty123", "admin123", "farmacia"]
        if v.lower() in common_passwords:
            raise ValueError("Contraseña demasiado común, elige otra más segura")

        return v
