# backend/app/schemas/action_plans.py
"""
Pydantic schemas para Exportación de Planes de Acción (Issue #513).

Define estructuras de datos para:
- Documento Word con plan de acción estructurado
- Insights priorizados con acciones concretas
- Seguimiento de responsables y plazos
"""

from datetime import date
from enum import Enum
from typing import Optional

from pydantic import BaseModel, Field


class ActionPriority(str, Enum):
    """Prioridad de la acción."""
    CRITICAL = "critical"  # Actuar inmediatamente
    HIGH = "high"          # Esta semana
    MEDIUM = "medium"      # Este mes
    LOW = "low"            # Cuando sea posible


class ActionStatus(str, Enum):
    """Estado de la acción para seguimiento."""
    PENDING = "pending"
    IN_PROGRESS = "in_progress"
    COMPLETED = "completed"
    DISMISSED = "dismissed"


class ActionPlanItem(BaseModel):
    """
    Acción individual del plan de acción.

    Basado en InsightItem pero con campos adicionales para seguimiento.
    """

    # Identificación
    rule_code: str = Field(..., description="Código de la regla (ej: STOCK_001)")
    category: str = Field(..., description="Categoría: stock, margin, hhi, trend, surtido")
    severity: str = Field(..., description="Severidad original: high, medium, low")

    # Descripción
    title: str = Field(..., description="Título corto del insight")
    description: str = Field(..., description="Descripción detallada del problema")
    economic_impact: str = Field(..., description="Impacto legible (ej: +450€/mes)")
    economic_value: float = Field(..., description="Valor numérico en €")

    # Plan de acción
    action_label: str = Field(..., description="Acción sugerida")
    action_steps: list[str] = Field(
        default_factory=list,
        description="Pasos concretos a seguir"
    )
    priority: ActionPriority = Field(
        ActionPriority.MEDIUM,
        description="Prioridad asignada"
    )
    suggested_deadline: Optional[str] = Field(
        None,
        description="Plazo sugerido (ej: 'Esta semana', '15 días')"
    )

    # Seguimiento (para documento Word editable)
    responsible: str = Field(
        "",
        description="Persona responsable (campo vacío para rellenar)"
    )
    status: ActionStatus = Field(
        ActionStatus.PENDING,
        description="Estado actual"
    )
    notes: str = Field(
        "",
        description="Notas adicionales (campo vacío para rellenar)"
    )

    # Productos afectados (resumido)
    affected_count: int = Field(
        0,
        ge=0,
        description="Número de productos afectados"
    )
    top_affected: list[str] = Field(
        default_factory=list,
        description="Top 3 productos afectados (nombres)"
    )


class ActionPlanSummary(BaseModel):
    """Resumen ejecutivo del plan de acción."""

    total_actions: int = Field(..., ge=0, description="Total de acciones")
    critical_count: int = Field(0, ge=0, description="Acciones críticas")
    high_count: int = Field(0, ge=0, description="Acciones alta prioridad")
    medium_count: int = Field(0, ge=0, description="Acciones media prioridad")
    low_count: int = Field(0, ge=0, description="Acciones baja prioridad")

    total_economic_opportunity: float = Field(
        0.0,
        description="Oportunidad económica total (€)"
    )

    by_category: dict[str, int] = Field(
        default_factory=dict,
        description="Acciones por categoría"
    )

    main_recommendation: str = Field(
        "",
        description="Recomendación principal basada en prioridades"
    )


class ActionPlanData(BaseModel):
    """
    Datos completos para generar documento Word de Plan de Acción.

    Estructura del documento:
    1. Portada: Farmacia, período, fecha generación
    2. Resumen Ejecutivo: KPIs y oportunidad total
    3. Acciones por Prioridad: Tablas con acciones agrupadas
    4. Checklist de Seguimiento: Tabla para marcar progreso
    """

    # Header/Portada
    pharmacy_name: str = Field(..., description="Nombre de la farmacia")
    pharmacy_nif: Optional[str] = Field(None, description="NIF de la farmacia")
    period_start: date = Field(..., description="Inicio del período analizado")
    period_end: date = Field(..., description="Fin del período analizado")
    generated_at: str = Field(..., description="Timestamp de generación")

    # Resumen ejecutivo
    summary: ActionPlanSummary = Field(
        default_factory=ActionPlanSummary,
        description="Resumen del plan"
    )

    # Acciones (ordenadas por prioridad y economic_value)
    actions: list[ActionPlanItem] = Field(
        default_factory=list,
        description="Lista de acciones priorizadas"
    )

    # Metadata
    insights_source: str = Field(
        "InsightEngine v1.0",
        description="Fuente de los insights"
    )
    export_version: str = Field(
        "1.0",
        description="Versión del formato de exportación"
    )

    class Config:
        json_schema_extra = {
            "example": {
                "pharmacy_name": "Farmacia Central",
                "pharmacy_nif": "B12345678",
                "period_start": "2025-12-01",
                "period_end": "2025-12-31",
                "generated_at": "2026-01-06T10:30:00",
                "summary": {
                    "total_actions": 8,
                    "critical_count": 1,
                    "high_count": 3,
                    "medium_count": 3,
                    "low_count": 1,
                    "total_economic_opportunity": 4500.0,
                    "by_category": {"stock": 3, "margin": 2, "trend": 2, "hhi": 1},
                    "main_recommendation": "Priorizar reposición de stock crítico",
                },
                "actions": [
                    {
                        "rule_code": "STOCK_001",
                        "category": "stock",
                        "severity": "high",
                        "title": "Stock Crítico: 12 productos de alta rotación",
                        "description": "Productos con alta demanda y menos de 3 días de cobertura.",
                        "economic_impact": "Est. +1.200€/mes en ventas recuperables",
                        "economic_value": 1200.0,
                        "action_label": "Reponer stock urgente",
                        "action_steps": [
                            "Revisar lista de productos afectados",
                            "Contactar proveedor para pedido urgente",
                            "Verificar fechas de entrega",
                        ],
                        "priority": "critical",
                        "suggested_deadline": "Inmediato",
                        "responsible": "",
                        "status": "pending",
                        "notes": "",
                        "affected_count": 12,
                        "top_affected": ["IBUPROFENO 600MG", "PARACETAMOL 1G", "OMEPRAZOL 20MG"],
                    }
                ],
                "insights_source": "InsightEngine v1.0",
                "export_version": "1.0",
            }
        }


class ActionPlanExportRequest(BaseModel):
    """Request para exportar plan de acción."""

    format: str = Field(
        "docx",
        description="Formato de exportación: docx, pdf",
        pattern="^(docx|pdf)$"
    )
    include_affected_products: bool = Field(
        True,
        description="Incluir lista detallada de productos afectados"
    )
    max_actions: int = Field(
        20,
        ge=1,
        le=50,
        description="Máximo número de acciones a incluir"
    )
    min_economic_value: float = Field(
        0.0,
        ge=0,
        description="Valor económico mínimo para incluir acción"
    )

    class Config:
        json_schema_extra = {
            "example": {
                "format": "docx",
                "include_affected_products": True,
                "max_actions": 20,
                "min_economic_value": 50.0,
            }
        }


class ActionPlanExportResponse(BaseModel):
    """Response de exportación de plan de acción."""

    success: bool = Field(..., description="Si la exportación fue exitosa")
    filename: str = Field(..., description="Nombre del archivo generado")
    file_size_kb: float = Field(..., description="Tamaño del archivo en KB")
    actions_included: int = Field(..., description="Número de acciones incluidas")
    total_economic_opportunity: float = Field(
        ..., description="Oportunidad económica total"
    )
    download_url: Optional[str] = Field(
        None,
        description="URL temporal de descarga (si aplica)"
    )
    message: str = Field("", description="Mensaje adicional")

    class Config:
        json_schema_extra = {
            "example": {
                "success": True,
                "filename": "plan_accion_farmacia_central_2026-01.docx",
                "file_size_kb": 45.2,
                "actions_included": 8,
                "total_economic_opportunity": 4500.0,
                "download_url": None,
                "message": "Plan de acción generado exitosamente",
            }
        }
