# backend/app/parsers/parser_factory.py
import logging
from typing import Dict, Optional, Tuple, Type

import pandas as pd
from sqlalchemy.orm import Session

from app.exceptions import InvalidERPFormatError, ProcessingError
from app.parsers.base_parser import BaseParser
from app.parsers.farmanager_parser import FarmanagerParser
from app.parsers.farmatic_parser import FarmaticParser

logger = logging.getLogger(__name__)


class ParserFactory:
    """
    Factory para crear el parser adecuado según el formato del archivo
    """

    # Registrar todos los parsers disponibles
    PARSERS = [
        FarmaticParser,
        FarmanagerParser,
        # Aquí se agregarán futuros parsers: NixfarmaParser, UnycopParser, etc.
    ]

    @classmethod
    def get_parser(
        cls,
        file_path: str,
        pharmacy_id: str,
        upload_id: str,
        erp_type: Optional[str] = None,
        db_session: Optional[Session] = None,
    ) -> Optional[BaseParser]:
        """
        Detecta y retorna el parser adecuado para el archivo

        Args:
            file_path: Ruta al archivo a parsear
            pharmacy_id: ID de la farmacia
            upload_id: ID del upload
            erp_type: Tipo de ERP si se conoce (opcional)
            db_session: Sesión de DB para actualizar pharmacy.erp_type automáticamente

        Returns:
            Instancia del parser adecuado o None si no se puede detectar
        """
        # DIAGNÓSTICO: Inicio de get_parser
        logger.warning("[GET_PARSER] === INICIO ===")
        logger.warning(f"[GET_PARSER] file_path: {file_path}")
        logger.warning(f"[GET_PARSER] pharmacy_id: {pharmacy_id[:8]}...")
        logger.warning(f"[GET_PARSER] upload_id: {upload_id[:8]}...")
        logger.warning(f"[GET_PARSER] erp_type: {erp_type}")

        # Si se especifica el tipo de ERP, intentar usar ese parser directamente
        if erp_type:
            parser_map = {
                "farmatic": FarmaticParser,
                "farmanager": FarmanagerParser,
            }

            parser_class = parser_map.get(erp_type.lower())
            if parser_class:
                logger.warning(f"[GET_PARSER] [SUCCESS] Usando parser especificado: {erp_type}")
                parser = parser_class(pharmacy_id, upload_id)
                # CRITICAL FIX: Marcar parser como selección manual del usuario
                # para que BaseParser.parse_file() NO valide con detect_format()
                parser.manual_selection = True
                logger.warning("[GET_PARSER] Parser configurado con manual_selection=True (bypass validation)")
                return parser
            else:
                logger.warning(f"[GET_PARSER] [WARNING] erp_type '{erp_type}' no encontrado en parser_map")

        # Si no, intentar detectar automáticamente
        logger.warning("[GET_PARSER] Detectando formato automáticamente...")

        # Leer una muestra del archivo para detectar el formato
        try:
            # Leer solo las primeras filas para la detección
            sample_df = cls._read_sample(file_path)

            logger.warning(f"[GET_PARSER] sample_df shape: {sample_df.shape}")

            if sample_df.empty:
                logger.error("[GET_PARSER] [ERROR] sample_df está vacío después de _read_sample()")
                raise ProcessingError(
                    process="parser_factory",
                    step="read_file",
                    reason=f"No se pudo leer el archivo {file_path}: archivo vacío o no encontrado",
                )

            # Probar cada parser hasta encontrar uno que coincida
            logger.warning(f"[GET_PARSER] Probando {len(cls.PARSERS)} parsers: {[p.__name__ for p in cls.PARSERS]}")

            for parser_class in cls.PARSERS:
                parser_name = parser_class.__name__
                logger.warning(f"[GET_PARSER] === PROBANDO PARSER: {parser_name} ===")

                parser = parser_class(pharmacy_id, upload_id)
                try:
                    detection_result = parser.detect_format(sample_df)
                    logger.warning(f"[GET_PARSER] {parser_name}.detect_format() -> {detection_result}")

                    if detection_result:
                        logger.warning(f"[GET_PARSER] [SUCCESS] FORMATO DETECTADO: {parser_name}")

                        # AUTO-ACTUALIZAR pharmacy.erp_type para futuros uploads
                        if db_session:
                            cls._update_pharmacy_erp_type(
                                db_session,
                                pharmacy_id,
                                parser_class
                            )

                        return parser
                    else:
                        # Parser retornó False, probar el siguiente
                        logger.warning(f"[GET_PARSER] [FAIL] {parser_name} retornó False, probando siguiente...")
                        continue

                except InvalidERPFormatError as e:
                    # Este parser no es compatible, probar el siguiente
                    logger.warning(f"[GET_PARSER] [FAIL] {parser_name} lanzó InvalidERPFormatError: {e.message}")
                    continue
                except Exception as e:
                    logger.error(f"[GET_PARSER] [ERROR] {parser_name} lanzó excepción inesperada: {type(e).__name__}: {str(e)}")
                    continue

            # Si ningún parser detecta el formato, lanzar excepción
            logger.error("[GET_PARSER] [ERROR] NINGÚN PARSER DETECTÓ EL FORMATO")
            raise InvalidERPFormatError(
                file_name=file_path.split("/")[-1],
                parser_type="AutoDetect",
                reason="No se pudo detectar el formato del archivo. Formatos soportados: Farmatic, Farmanager",
            )

        except InvalidERPFormatError:
            # Re-lanzar excepciones de formato inválido
            raise
        except Exception as e:
            logger.error(f"[GET_PARSER] [ERROR] ERROR GENERAL: {type(e).__name__}: {str(e)}")
            raise ProcessingError(
                process="detección_formato", step="get_parser", reason=f"Error inesperado detectando formato: {str(e)}"
            )

    @classmethod
    def _update_pharmacy_erp_type(
        cls,
        db_session: Session,
        pharmacy_id: str,
        parser_class: Type[BaseParser]
    ) -> None:
        """
        Actualiza pharmacy.erp_type después de detección automática exitosa.

        Args:
            db_session: SQLAlchemy Session (synchronous) from file processing thread
            pharmacy_id: ID de la farmacia a actualizar
            parser_class: Clase del parser detectado (FarmaticParser, FarmanagerParser, etc.)

        Returns:
            None. Actualiza la BD o hace rollback en caso de error.

        Note:
            Este método NO falla el upload si la actualización falla.
            Usa try-except con rollback para seguridad transaccional.
        """
        try:
            from app.models.pharmacy import Pharmacy

            # Mapeo inverso: Parser class → erp_type string
            parser_to_erp = {
                "FarmaticParser": "farmatic",
                "FarmanagerParser": "farmanager",
            }

            detected_erp = parser_to_erp.get(parser_class.__name__)

            if detected_erp:
                pharmacy = db_session.query(Pharmacy).filter(
                    Pharmacy.id == pharmacy_id
                ).first()

                if pharmacy:
                    # Solo actualizar si está NULL o diferente
                    if pharmacy.erp_type != detected_erp:
                        old_value = pharmacy.erp_type or "NULL"
                        pharmacy.erp_type = detected_erp
                        db_session.commit()
                        logger.info(
                            f"[AUTO_UPDATE] pharmacy.erp_type actualizado: "
                            f"{old_value} → {detected_erp} para farmacia {pharmacy_id[:8]}"
                        )
                    else:
                        logger.debug(f"[AUTO_UPDATE] pharmacy.erp_type ya configurado: {detected_erp}")
                else:
                    logger.warning(f"[AUTO_UPDATE] Farmacia {pharmacy_id} no encontrada")

        except Exception as e:
            # NO fallar el upload si la actualización falla
            logger.warning(f"[AUTO_UPDATE] Error actualizando pharmacy.erp_type: {str(e)}")
            db_session.rollback()

    @classmethod
    def _read_sample(cls, file_path: str, nrows: int = 100) -> pd.DataFrame:
        """
        Lee una muestra del archivo para la detección de formato
        """
        # DIAGNÓSTICO: Inicio de lectura
        logger.warning("[READ_SAMPLE] === INICIO LECTURA ===")
        logger.warning(f"[READ_SAMPLE] file_path: {file_path}")
        logger.warning(f"[READ_SAMPLE] nrows: {nrows}")

        try:
            # Intentar leer como CSV con diferentes separadores y codificaciones
            if file_path.endswith(".csv"):
                encodings = ["utf-8", "latin1", "cp1252", "iso-8859-1"]
                separators = [";", ",", "\t", "|"]

                logger.warning(f"[READ_SAMPLE] Intentando {len(encodings)} encodings x {len(separators)} separadores = {len(encodings) * len(separators)} combinaciones")

                for encoding in encodings:
                    for sep in separators:
                        try:
                            df = pd.read_csv(file_path, nrows=nrows, sep=sep, encoding=encoding)

                            # DIAGNÓSTICO: Resultado de cada intento
                            logger.warning(f"[READ_SAMPLE] [SUCCESS] Lectura exitosa: encoding={encoding}, sep='{sep}', shape={df.shape}, columns={len(df.columns)}")

                            if len(df.columns) > 1:
                                logger.warning(f"[READ_SAMPLE] [SUCCESS] DataFrame válido con encoding={encoding}, sep='{sep}'")
                                logger.warning(f"[READ_SAMPLE] Columnas leídas: {list(df.columns)[:5]}...")
                                logger.warning(f"[READ_SAMPLE] Primeras filas:\n{df.head(2).to_dict('records')}")
                                return df
                            else:
                                logger.warning(f"[READ_SAMPLE] [WARNING] Solo 1 columna (separador incorrecto): encoding={encoding}, sep='{sep}'")

                        except Exception as e:
                            logger.warning(f"[READ_SAMPLE] [FAIL] Falló: encoding={encoding}, sep='{sep}' -> {type(e).__name__}: {str(e)[:100]}")
                            continue

                # Si llegamos aquí, ninguna combinación funcionó
                logger.error("[READ_SAMPLE] [ERROR] FALLÓ: Ninguna combinación de encoding/sep funcionó para CSV")

            # Intentar leer como Excel
            elif file_path.endswith((".xlsx", ".xls")):
                logger.warning("[READ_SAMPLE] Intentando lectura Excel")
                df = pd.read_excel(file_path, nrows=nrows)
                logger.warning(f"[READ_SAMPLE] [SUCCESS] Excel leído exitosamente: shape={df.shape}")
                return df

        except Exception as e:
            logger.error(f"[READ_SAMPLE] [ERROR] ERROR GENERAL: {type(e).__name__}: {str(e)}")

        logger.error("[READ_SAMPLE] [ERROR] Retornando DataFrame vacío")
        return pd.DataFrame()

    @classmethod
    def parse_file(
        cls,
        file_path: str,
        pharmacy_id: str,
        upload_id: str,
        erp_type: Optional[str] = None,
        db_session: Optional[Session] = None,
    ) -> Tuple[pd.DataFrame, Dict]:
        """
        Método conveniente para parsear un archivo directamente

        Args:
            file_path: Ruta al archivo a parsear
            pharmacy_id: ID de la farmacia
            upload_id: ID del upload
            erp_type: Tipo de ERP si se conoce (opcional)
            db_session: Sesión de DB para auto-actualizar pharmacy.erp_type

        Returns:
            Tuple de (DataFrame parseado, estadísticas)
        """
        try:
            parser = cls.get_parser(file_path, pharmacy_id, upload_id, erp_type, db_session)
        except InvalidERPFormatError as e:
            return pd.DataFrame(), {
                "errors": [e.message],
                "total_rows": 0,
                "processed_rows": 0,
                "error_type": "format_detection",
            }
        except ProcessingError as e:
            return pd.DataFrame(), {
                "errors": [e.message],
                "total_rows": 0,
                "processed_rows": 0,
                "error_type": "processing",
            }

        # Intentar diferentes encodings si el primero falla
        encodings = ["utf-8", "latin1", "cp1252", "iso-8859-1"]

        for encoding in encodings:
            try:
                logger.info(f"Intentando parsing con encoding: {encoding}")
                result_df, stats = parser.parse_file(file_path, encoding)

                # Verificar si el parsing fue realmente exitoso
                if len(result_df) > 0 and len(stats.get("errors", [])) == 0:
                    logger.info(f"Parsing exitoso con encoding: {encoding}")
                    # Issue #412: Agregar encoding usado a stats para warnings
                    stats["encoding_used"] = encoding
                    stats["encoding_fallback"] = encoding != "utf-8"
                    return result_df, stats
                else:
                    # El parsing retornó un DataFrame vacío o con errores
                    error_msg = f"Parsing falló: {stats.get('errors', ['DataFrame vacío'])}"
                    logger.warning(f"Parsing falló con encoding {encoding}: {error_msg}")
                    continue

            except Exception as e:
                logger.warning(f"Parsing falló con encoding {encoding}: {str(e)}")
                continue

        # Si todos los encodings fallaron
        return pd.DataFrame(), {
            "errors": ["No se pudo parsear el archivo con ninguna codificación"],
            "total_rows": 0,
            "processed_rows": 0,
        }
