# backend/app/parsers/inventory/farmanager_inventory_parser.py
"""
Parser de inventario para ERP Farmanager.

Issue #476: Carga de ficheros de inventario desde ERPs.

Formato detectado del fichero real (Informe_Articulos):
- Separador: ; (punto y coma)
- Encoding: Latin-1 / CP1252
- Decimal: , (coma española)
- Fechas: YYYY-MM-DD HH:MM:SS.0

Columnas:
| Original | Normalizado | Descripción |
|----------|-------------|-------------|
| Código | ean13 | EAN-13 del producto |
| Descripción | product_name | Nombre del producto |
| Stock | stock_quantity | Unidades en stock |
| PCM | unit_cost | Precio de coste |
| IVA/IGIC | tax_rate | Tipo de IVA |
| Ultima Compra | last_purchase_date | Fecha última compra |
| Ultima venta | last_sale_date | Fecha última venta |
| P.V.P. | unit_price | Precio de venta |
"""

import logging
from typing import Dict, List

import pandas as pd

from .base_inventory_parser import BaseInventoryParser

logger = logging.getLogger(__name__)


class FarmanagerInventoryParser(BaseInventoryParser):
    """
    Parser de inventario para ERP Farmanager.

    Procesa ficheros de tipo "Informe_Articulos" exportados desde Farmanager.
    """

    # Indicadores únicos de formato Farmanager inventario
    FORMAT_INDICATORS = [
        "CODIGO",          # Código (EAN)
        "DESCRIPCION",     # Descripción
        "STOCK",           # Stock
        "PCM",             # Precio de Coste Medio
        "IVAIGIC",         # IVA/IGIC (normalizado sin /)
        "ULTIMACOMPRA",    # Ultima Compra
        "ULTIMAVENTA",     # Ultima venta
        "PVP",             # P.V.P.
    ]

    # Mapeo de columnas ERP → formato normalizado
    COLUMN_MAPPING = {
        # Columnas principales
        "CODIGO": "ean13",
        "DESCRIPCION": "product_name",
        "STOCK": "stock_quantity",
        "PCM": "unit_cost",
        "IVAIGIC": "tax_rate",
        "ULTIMACOMPRA": "last_purchase_date",
        "ULTIMAVENTA": "last_sale_date",
        "PVP": "unit_price",

        # Variantes alternativas
        "CODIGONACIONAL": "product_code",
        "CN": "product_code",
        "ARTICULO": "ean13",
        "EXISTENCIAS": "stock_quantity",
        "UNIDADES": "stock_quantity",
        "PRECIOCOSTE": "unit_cost",
        "COSTE": "unit_cost",
        "PRECIOVENTA": "unit_price",
        "IVA": "tax_rate",
    }

    # Umbral mínimo de indicadores para detección
    MIN_INDICATORS = 3

    def detect_format(self, df: pd.DataFrame) -> bool:
        """
        Detecta si el DataFrame corresponde al formato Farmanager inventario.

        Busca al menos MIN_INDICATORS columnas características de Farmanager.

        Args:
            df: DataFrame con las primeras filas del fichero

        Returns:
            True si es formato Farmanager inventario
        """
        if df is None or df.empty:
            return False

        # Normalizar nombres de columnas
        normalized_columns = [self.normalize_column_name(col) for col in df.columns]

        # Contar indicadores encontrados
        matches = 0
        for indicator in self.FORMAT_INDICATORS:
            for col in normalized_columns:
                if self.column_matches_indicator(col, indicator):
                    matches += 1
                    break

        is_match = matches >= self.MIN_INDICATORS

        if is_match:
            logger.info(
                f"[FarmanagerInventoryParser] Detected format: {matches}/{len(self.FORMAT_INDICATORS)} indicators"
            )
        else:
            logger.debug(
                f"[FarmanagerInventoryParser] Format not detected: {matches}/{len(self.FORMAT_INDICATORS)} indicators"
            )

        return is_match

    def map_columns(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Mapea columnas del formato Farmanager al formato normalizado.

        Args:
            df: DataFrame con columnas originales

        Returns:
            DataFrame con columnas normalizadas
        """
        result_df = pd.DataFrame()

        # Crear diccionario de columna normalizada → columna original
        original_columns = df.columns.tolist()
        normalized_map = {}

        for original in original_columns:
            normalized = self.normalize_column_name(original)
            normalized_map[normalized] = original

        # Mapear columnas conocidas
        for normalized_col, target_col in self.COLUMN_MAPPING.items():
            # Buscar match exacto o fuzzy
            matched_original = None

            # Primero intentar match exacto
            if normalized_col in normalized_map:
                matched_original = normalized_map[normalized_col]
            else:
                # Intentar match fuzzy
                for norm_col, orig_col in normalized_map.items():
                    if self.column_matches_indicator(norm_col, normalized_col):
                        matched_original = orig_col
                        break

            if matched_original and matched_original in df.columns:
                # Solo añadir si no existe ya (priorizar primer mapping)
                if target_col not in result_df.columns:
                    result_df[target_col] = df[matched_original]

        # Log columnas mapeadas
        logger.info(f"[FarmanagerInventoryParser] Mapped columns: {list(result_df.columns)}")

        return result_df

    def clean_data(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Limpia y transforma datos según formato Farmanager.

        Operaciones:
        1. Limpiar nombres de producto
        2. Convertir decimales (coma → punto)
        3. Convertir stock a entero
        4. Parsear fechas
        5. Extraer código nacional de EAN

        Args:
            df: DataFrame con columnas mapeadas

        Returns:
            DataFrame con datos limpios
        """
        df = df.copy()

        # 1. Limpiar nombres de producto
        if "product_name" in df.columns:
            df["product_name"] = df["product_name"].astype(str).str.strip()
            # Filtrar nombres vacíos o placeholder
            df = df[
                (df["product_name"].notna()) &
                (df["product_name"] != "") &
                (~df["product_name"].str.startswith("---"))
            ]

        # 2. Limpiar EAN13
        if "ean13" in df.columns:
            df["ean13"] = df["ean13"].astype(str).str.strip()
            # Extraer código nacional
            df["product_code"] = df["ean13"].apply(self.extract_cn_from_ean)

        # 3. Convertir stock a entero
        if "stock_quantity" in df.columns:
            df["stock_quantity"] = pd.to_numeric(
                df["stock_quantity"].astype(str).str.replace(",", "."),
                errors="coerce"
            ).fillna(0).astype(int)

        # 4. Convertir precios (formato español con coma)
        for price_col in ["unit_price", "unit_cost", "tax_rate"]:
            if price_col in df.columns:
                df[price_col] = df[price_col].apply(self.parse_spanish_decimal)

        # 5. Parsear fechas
        for date_col in ["last_sale_date", "last_purchase_date"]:
            if date_col in df.columns:
                df[date_col] = df[date_col].apply(self.parse_datetime)

        # 6. Filtrar productos con stock 0 si son placeholders
        # Mantener productos con stock 0 pero nombre válido (pueden ser productos sin stock)
        # df = df[df["stock_quantity"] > 0]  # OPCIONAL: descomentar para excluir stock 0

        # Log estadísticas
        logger.info(
            f"[FarmanagerInventoryParser] Cleaned data: "
            f"{len(df)} products, "
            f"{df['stock_quantity'].sum()} total units, "
            f"{(df['stock_quantity'] > 0).sum()} products with stock"
        )

        return df
