# backend/app/parsers/inventory/base_inventory_parser.py
"""
Clase base para parsers de inventario de ERPs farmacéuticos.

Issue #476: Carga de ficheros de inventario desde ERPs.
"""

import logging
import re
import unicodedata
from abc import ABC, abstractmethod
from datetime import date
from pathlib import Path
from typing import Dict, List, Optional, Tuple

import pandas as pd

logger = logging.getLogger(__name__)


class BaseInventoryParser(ABC):
    """
    Clase base para parsers de inventario de ERPs farmacéuticos.

    Similar a BaseParser pero especializada para ficheros de inventario
    en lugar de ventas.
    """

    # Columnas requeridas en el formato normalizado
    REQUIRED_COLUMNS = ["product_name", "stock_quantity"]

    # Columnas de identificación de producto (al menos una recomendada)
    PRODUCT_ID_COLUMNS = ["ean13", "product_code"]

    # Columnas opcionales
    OPTIONAL_COLUMNS = [
        "unit_price",
        "unit_cost",
        "last_sale_date",
        "last_purchase_date",
        "tax_rate",
    ]

    # Encodings a probar en orden de preferencia
    ENCODINGS = ["utf-8", "latin1", "cp1252", "iso-8859-1"]

    # Separadores comunes en ficheros ERP
    SEPARATORS = [";", ",", "\t", "|"]

    def __init__(self, pharmacy_id: str, upload_id: str):
        self.pharmacy_id = pharmacy_id
        self.upload_id = upload_id
        self.errors: List[str] = []
        self.warnings: List[str] = []
        self.manual_selection = False

    @staticmethod
    def normalize_column_name(column_name: str) -> str:
        """
        Normaliza nombres de columnas para comparación insensible a encoding/locale.

        Maneja encoding corrupto (Artículo → Art□culo) común en ficheros ERP españoles.

        Args:
            column_name: Nombre de columna original

        Returns:
            Nombre normalizado: solo letras/números, sin acentos, mayúsculas
        """
        try:
            nfkd_form = unicodedata.normalize('NFKD', column_name)
        except Exception:
            nfkd_form = column_name

        # Remover acentos/combinantes y mantener SOLO alfanuméricos
        cleaned_chars = [c for c in nfkd_form if not unicodedata.combining(c) and c.isalnum()]
        temp_result = "".join(cleaned_chars)

        # Convertir a mayúsculas
        return re.sub(r'[^A-Z0-9]', '', temp_result.upper())

    @staticmethod
    def column_matches_indicator(
        col_normalized: str,
        indicator_normalized: str,
        min_length: int = 3,
        fuzzy_threshold: float = 0.85
    ) -> bool:
        """
        Determina si una columna normalizada coincide con un indicador.

        Usa tres estrategias:
        1. Exact match
        2. Substring bidireccional (1-2 caracteres faltantes)
        3. Fuzzy matching (85% similarity)
        """
        if not col_normalized or not indicator_normalized:
            return False

        # Estrategia 1: Exact match
        if col_normalized == indicator_normalized:
            return True

        # Estrategia 2: Substring bidireccional
        if len(col_normalized) >= min_length and len(indicator_normalized) >= min_length:
            if col_normalized in indicator_normalized or indicator_normalized in col_normalized:
                return True

        # Estrategia 3: Fuzzy matching
        try:
            from difflib import SequenceMatcher
            ratio = SequenceMatcher(None, col_normalized, indicator_normalized).ratio()
            if ratio >= fuzzy_threshold:
                return True
        except Exception:
            pass

        return False

    @abstractmethod
    def detect_format(self, df: pd.DataFrame) -> bool:
        """
        Detecta si un DataFrame corresponde a este formato de ERP.

        Args:
            df: DataFrame con los primeros N registros del fichero

        Returns:
            True si el formato coincide con este parser
        """
        pass

    @abstractmethod
    def map_columns(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Mapea las columnas del ERP al formato normalizado.

        Args:
            df: DataFrame con columnas originales del ERP

        Returns:
            DataFrame con columnas normalizadas
        """
        pass

    @abstractmethod
    def clean_data(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Limpia y transforma los datos según las reglas del ERP.

        Args:
            df: DataFrame con columnas mapeadas

        Returns:
            DataFrame con datos limpios
        """
        pass

    def read_file(
        self,
        file_path: str,
        encoding: Optional[str] = None,
        separator: Optional[str] = None,
        sample_size: int = 100
    ) -> Tuple[pd.DataFrame, Dict]:
        """
        Lee un fichero de inventario con detección automática de encoding y separador.

        Args:
            file_path: Ruta al fichero
            encoding: Encoding específico (None para auto-detectar)
            separator: Separador específico (None para auto-detectar)
            sample_size: Número de filas para sample

        Returns:
            Tuple de (DataFrame, stats dict)
        """
        path = Path(file_path)
        stats = {
            "filename": path.name,
            "encoding_used": None,
            "encoding_fallback": False,
            "separator_used": None,
            "total_rows": 0,
            "errors": [],
            "warnings": [],
        }

        # Auto-detectar encoding
        encodings_to_try = [encoding] if encoding else self.ENCODINGS

        df = None
        for enc in encodings_to_try:
            try:
                # Auto-detectar separador
                separators_to_try = [separator] if separator else self.SEPARATORS

                for sep in separators_to_try:
                    try:
                        df = pd.read_csv(
                            file_path,
                            encoding=enc,
                            sep=sep,
                            nrows=sample_size if sample_size else None,
                            dtype=str,  # Todo como string inicialmente
                            on_bad_lines='warn',
                        )

                        # Verificar que hay columnas válidas
                        if len(df.columns) > 1:
                            stats["encoding_used"] = enc
                            stats["encoding_fallback"] = (enc != "utf-8")
                            stats["separator_used"] = sep
                            stats["total_rows"] = len(df)
                            break

                    except Exception:
                        continue

                if df is not None and len(df.columns) > 1:
                    break

            except Exception as e:
                logger.warning(f"Failed to read with encoding {enc}: {e}")
                continue

        if df is None or len(df.columns) <= 1:
            raise ValueError(f"No se pudo leer el fichero: {file_path}")

        if stats["encoding_fallback"]:
            stats["warnings"].append(
                f"Se usó encoding alternativo: {stats['encoding_used']} (el fichero no es UTF-8)"
            )

        return df, stats

    def parse_file(
        self,
        file_path: str,
        encoding: Optional[str] = None,
        snapshot_date: Optional[date] = None
    ) -> Tuple[pd.DataFrame, Dict]:
        """
        Parsea un fichero de inventario completo.

        Args:
            file_path: Ruta al fichero
            encoding: Encoding específico (None para auto-detectar)
            snapshot_date: Fecha del snapshot (None para usar fecha actual)

        Returns:
            Tuple de (DataFrame normalizado, stats dict)
        """
        # 1. Leer fichero completo
        df, stats = self.read_file(file_path, encoding, sample_size=None)

        # 2. Validar formato (si no es selección manual)
        if not self.manual_selection:
            sample_df, _ = self.read_file(file_path, encoding, sample_size=100)
            if not self.detect_format(sample_df):
                raise ValueError(f"El fichero no coincide con el formato esperado")

        # 3. Mapear columnas
        df = self.map_columns(df)

        # 4. Limpiar datos
        df = self.clean_data(df)

        # 5. Añadir metadata
        df["pharmacy_id"] = self.pharmacy_id
        df["file_upload_id"] = self.upload_id
        df["snapshot_date"] = snapshot_date or date.today()

        # 6. Calcular valores de stock
        self._calculate_stock_values(df)

        # 7. Validar columnas requeridas
        missing = [col for col in self.REQUIRED_COLUMNS if col not in df.columns]
        if missing:
            raise ValueError(f"Columnas requeridas faltantes: {missing}")

        # 8. Filtrar filas válidas
        initial_count = len(df)
        df = df[df["product_name"].notna() & (df["product_name"].str.strip() != "")]
        valid_count = len(df)

        stats["total_rows"] = initial_count
        stats["valid_rows"] = valid_count
        stats["invalid_rows"] = initial_count - valid_count
        stats["errors"] = self.errors
        stats["warnings"] = self.warnings + stats.get("warnings", [])

        return df, stats

    def _calculate_stock_values(self, df: pd.DataFrame) -> None:
        """Calcula stock_value y cost_value si hay precios y cantidades."""
        if "stock_quantity" in df.columns and "unit_price" in df.columns:
            df["stock_value"] = pd.to_numeric(df["stock_quantity"], errors="coerce") * \
                                pd.to_numeric(df["unit_price"], errors="coerce")

        if "stock_quantity" in df.columns and "unit_cost" in df.columns:
            df["cost_value"] = pd.to_numeric(df["stock_quantity"], errors="coerce") * \
                               pd.to_numeric(df["unit_cost"], errors="coerce")

    @staticmethod
    def extract_cn_from_ean(ean13: str) -> Optional[str]:
        """
        Extrae el código nacional (CN) de un EAN-13 español.

        Formato EAN español farmacia: 847000XXXXXX (check digit)
        El CN son los 6 dígitos después de 847000.

        Args:
            ean13: Código EAN-13 completo

        Returns:
            Código nacional de 6 dígitos o None si no es EAN español
        """
        if not ean13 or not isinstance(ean13, str):
            return None

        ean13 = ean13.strip()
        if len(ean13) < 12:
            return None

        # EAN español de farmacia: empieza por 847000
        if ean13.startswith("847000"):
            return ean13[6:12]

        return None

    @staticmethod
    def parse_spanish_decimal(value: str) -> Optional[float]:
        """
        Parsea un número decimal en formato español (coma como separador).

        Args:
            value: String con número (ej: "13,44")

        Returns:
            Float o None si no es válido
        """
        if not value or not isinstance(value, str):
            return None

        try:
            # Reemplazar coma por punto
            cleaned = value.strip().replace(",", ".")
            return float(cleaned)
        except (ValueError, AttributeError):
            return None

    @staticmethod
    def parse_datetime(value: str) -> Optional[date]:
        """
        Parsea fecha/hora en varios formatos comunes de ERPs.

        Formatos soportados:
        - YYYY-MM-DD HH:MM:SS.0
        - DD/MM/YYYY
        - DD-MM-YYYY

        Args:
            value: String con fecha

        Returns:
            date o None si no es válido
        """
        if not value or not isinstance(value, str):
            return None

        value = value.strip()
        if not value:
            return None

        try:
            # Formato YYYY-MM-DD HH:MM:SS.0 (Farmanager)
            if " " in value and "-" in value.split(" ")[0]:
                dt = pd.to_datetime(value, format="%Y-%m-%d %H:%M:%S.%f", errors="coerce")
                if pd.notna(dt):
                    return dt.date()

            # Formato DD/MM/YYYY
            if "/" in value:
                dt = pd.to_datetime(value, dayfirst=True, errors="coerce")
                if pd.notna(dt):
                    return dt.date()

            # Formato genérico
            dt = pd.to_datetime(value, errors="coerce")
            if pd.notna(dt):
                return dt.date()

        except Exception:
            pass

        return None
