# backend/app/parsers/file_type_detector.py
"""
Detector automático de tipo de archivo ERP (ventas vs inventario).

Issue #476: Detección automática para eliminar selector manual en frontend.

Lógica de detección basada en columnas distintivas:
- VENTAS: idOperación, Unidades, Importe, Empleado, Cliente, Descuento
- INVENTARIO: Stock, PCM, IVA/IGIC, Ultima Compra, Ultima venta
"""

import logging
import re
import unicodedata
from enum import Enum
from typing import List, Optional, Tuple

import pandas as pd

logger = logging.getLogger(__name__)


class FileContentType(str, Enum):
    """Tipo de contenido del archivo ERP."""
    SALES = "sales"
    INVENTORY = "inventory"
    UNKNOWN = "unknown"


# Indicadores de columnas para cada tipo de archivo
# Normalizados: sin acentos, sin espacios, mayúsculas
SALES_INDICATORS = [
    "IDOPERACION",      # idOperación - ID único de venta
    "FECHAHORA",        # Fecha/Hora - timestamp de venta
    "UNIDADES",         # Unidades vendidas
    "IMPORTE",          # Importe de la venta
    "EMPLEADO",         # Empleado que realizó la venta
    "CLIENTE",          # Cliente
    "DESCUENTO",        # Descuento aplicado
    "PRECIOCOSTE",      # Precio coste (en contexto de venta)
    "OPERACION",        # Número de operación
]

INVENTORY_INDICATORS = [
    "STOCK",            # Stock - cantidad en inventario
    "PCM",              # Precio Coste Medio
    "IVAIGIC",          # IVA/IGIC
    "ULTIMACOMPRA",     # Ultima Compra
    "ULTIMAVENTA",      # Ultima venta
    "EXISTENCIAS",      # Sinónimo de Stock
]

# Umbrales de detección
MIN_SALES_INDICATORS = 2      # Mínimo 2 indicadores de ventas
MIN_INVENTORY_INDICATORS = 2  # Mínimo 2 indicadores de inventario


def normalize_column_name(column: str) -> str:
    """
    Normaliza nombre de columna para comparación.

    - Elimina acentos (Artículo → Articulo)
    - Elimina espacios
    - Convierte a mayúsculas
    - Elimina caracteres especiales (/, -, etc.)

    Args:
        column: Nombre de columna original

    Returns:
        Nombre normalizado
    """
    if not column:
        return ""

    # Convertir a string
    col = str(column)

    # Normalizar caracteres Unicode (eliminar acentos)
    col = unicodedata.normalize("NFKD", col)
    col = "".join(c for c in col if not unicodedata.combining(c))

    # Eliminar caracteres especiales y espacios
    col = re.sub(r"[^a-zA-Z0-9]", "", col)

    # Mayúsculas
    return col.upper()


def column_matches_indicator(column: str, indicator: str) -> bool:
    """
    Verifica si una columna coincide con un indicador.

    Estrategias de matching:
    1. Match exacto
    2. Columna contiene indicador
    3. Indicador contiene columna como prefijo/sufijo (para abreviaturas)

    Args:
        column: Nombre de columna normalizado
        indicator: Indicador a buscar

    Returns:
        True si hay coincidencia
    """
    if not column or not indicator:
        return False

    # Match exacto
    if column == indicator:
        return True

    # Columna contiene indicador
    if indicator in column:
        return True

    # Indicador contiene columna como prefijo o sufijo (para abreviaturas como "IVA" en "IVAIGIC")
    # Evita falsos positivos donde la columna aparece en medio de otra palabra
    if len(column) >= 3 and len(indicator) > len(column) and column in indicator:
        if indicator.startswith(column) or indicator.endswith(column):
            return True

    return False


def count_indicator_matches(columns: List[str], indicators: List[str]) -> int:
    """
    Cuenta cuántos indicadores coinciden con las columnas.

    Args:
        columns: Lista de nombres de columna normalizados
        indicators: Lista de indicadores a buscar

    Returns:
        Número de coincidencias
    """
    matches = 0
    for indicator in indicators:
        for column in columns:
            if column_matches_indicator(column, indicator):
                matches += 1
                break  # Contar cada indicador solo una vez
    return matches


def detect_file_content_type(df: pd.DataFrame) -> Tuple[FileContentType, dict]:
    """
    Detecta si el DataFrame contiene datos de ventas o inventario.

    Analiza las columnas del DataFrame y cuenta coincidencias con
    indicadores de cada tipo para determinar el tipo de contenido.

    Args:
        df: DataFrame con las primeras filas del archivo

    Returns:
        Tuple de (FileContentType, dict con detalles de detección)
    """
    if df is None or df.empty:
        return FileContentType.UNKNOWN, {
            "error": "DataFrame vacío",
            "sales_matches": 0,
            "inventory_matches": 0,
        }

    # Normalizar todas las columnas
    normalized_columns = [normalize_column_name(col) for col in df.columns]

    # Contar coincidencias para cada tipo
    sales_matches = count_indicator_matches(normalized_columns, SALES_INDICATORS)
    inventory_matches = count_indicator_matches(normalized_columns, INVENTORY_INDICATORS)

    # Construir detalles de detección
    details = {
        "original_columns": list(df.columns),
        "normalized_columns": normalized_columns,
        "sales_matches": sales_matches,
        "inventory_matches": inventory_matches,
        "sales_indicators": SALES_INDICATORS,
        "inventory_indicators": INVENTORY_INDICATORS,
    }

    logger.info(
        f"[FILE_TYPE_DETECT] Columnas: {len(df.columns)}, "
        f"Ventas: {sales_matches}/{MIN_SALES_INDICATORS}, "
        f"Inventario: {inventory_matches}/{MIN_INVENTORY_INDICATORS}"
    )

    # Determinar tipo basado en coincidencias
    # Priorizar el tipo con más coincidencias
    if sales_matches >= MIN_SALES_INDICATORS and sales_matches > inventory_matches:
        details["detected_type"] = "sales"
        details["confidence"] = "high" if sales_matches >= 4 else "medium"
        logger.info(f"[FILE_TYPE_DETECT] Detectado: VENTAS (matches: {sales_matches})")
        return FileContentType.SALES, details

    if inventory_matches >= MIN_INVENTORY_INDICATORS and inventory_matches > sales_matches:
        details["detected_type"] = "inventory"
        details["confidence"] = "high" if inventory_matches >= 4 else "medium"
        logger.info(f"[FILE_TYPE_DETECT] Detectado: INVENTARIO (matches: {inventory_matches})")
        return FileContentType.INVENTORY, details

    # Empate o ninguno tiene suficientes coincidencias
    if sales_matches >= MIN_SALES_INDICATORS and inventory_matches >= MIN_INVENTORY_INDICATORS:
        # Empate - usar el que tenga más coincidencias
        if sales_matches >= inventory_matches:
            details["detected_type"] = "sales"
            details["confidence"] = "low"
            details["note"] = "Empate resuelto a favor de ventas"
            return FileContentType.SALES, details
        else:
            details["detected_type"] = "inventory"
            details["confidence"] = "low"
            details["note"] = "Empate resuelto a favor de inventario"
            return FileContentType.INVENTORY, details

    # No se pudo determinar
    details["detected_type"] = "unknown"
    details["confidence"] = "none"
    logger.warning(
        f"[FILE_TYPE_DETECT] No se pudo determinar tipo: "
        f"ventas={sales_matches}, inventario={inventory_matches}"
    )
    return FileContentType.UNKNOWN, details


def detect_file_content_type_from_path(
    file_path: str,
    nrows: int = 100
) -> Tuple[FileContentType, dict]:
    """
    Detecta el tipo de contenido de un archivo ERP.

    Lee las primeras filas del archivo y analiza las columnas
    para determinar si es ventas o inventario.

    Args:
        file_path: Ruta al archivo
        nrows: Número de filas a leer para detección

    Returns:
        Tuple de (FileContentType, dict con detalles)
    """
    logger.info(f"[FILE_TYPE_DETECT] Analizando archivo: {file_path}")

    try:
        # Leer muestra del archivo
        df = _read_file_sample(file_path, nrows)

        if df.empty:
            return FileContentType.UNKNOWN, {
                "error": "No se pudo leer el archivo",
                "file_path": file_path,
            }

        return detect_file_content_type(df)

    except Exception as e:
        logger.error(f"[FILE_TYPE_DETECT] Error: {e}")
        return FileContentType.UNKNOWN, {
            "error": str(e),
            "file_path": file_path,
        }


def _read_file_sample(file_path: str, nrows: int = 100) -> pd.DataFrame:
    """
    Lee una muestra del archivo para análisis.

    Intenta diferentes encodings y separadores.

    Args:
        file_path: Ruta al archivo
        nrows: Número de filas a leer

    Returns:
        DataFrame con la muestra o DataFrame vacío si falla
    """
    encodings = ["utf-8", "latin1", "cp1252", "iso-8859-1"]
    separators = [";", ",", "\t", "|"]

    # CSV
    if file_path.lower().endswith(".csv"):
        for encoding in encodings:
            for sep in separators:
                try:
                    df = pd.read_csv(
                        file_path,
                        nrows=nrows,
                        sep=sep,
                        encoding=encoding,
                        on_bad_lines="skip",
                    )
                    if len(df.columns) > 1:
                        return df
                except Exception:
                    continue

    # Excel
    elif file_path.lower().endswith((".xlsx", ".xls")):
        try:
            return pd.read_excel(file_path, nrows=nrows)
        except Exception as e:
            logger.warning(f"[FILE_TYPE_DETECT] Error leyendo Excel: {e}")

    return pd.DataFrame()
