# backend/app/parsers/farmatic_parser.py
import logging

import pandas as pd

from app.exceptions import InvalidDataTypeError, InvalidERPFormatError, MissingRequiredColumnsError, ProcessingError
from app.parsers.base_parser import BaseParser

logger = logging.getLogger(__name__)


class FarmaticParser(BaseParser):
    """
    Parser específico para archivos del ERP Farmatic
    """

    # Mapeo de columnas de Farmatic a nuestro formato basado en datos reales
    COLUMN_MAPPING = {
        # Mapeo basado en muestra real: Fecha, Hora, Vendedor, Código, Descripción, Cantidad, Aportación, R.P., P.V.P, Importe Bruto, Importe Neto, Cliente, C.Cliente
        "Fecha": "sale_date",
        "Hora": "sale_time",
        "Código": "product_code",
        "Descripción": "product_name",
        "Cantidad": "quantity",
        "Aportación": "contribution_type",  # R1, etc
        "R.P.": "reference_price",  # Precio de referencia
        "P.V.P": "sale_price",
        "Importe Bruto": "gross_amount",
        "Importe Neto": "net_amount",  # Importe neto después de copagos (NO usar para análisis)
        "Cliente": "client_name",
        "C.Cliente": "client_code",
        "Vendedor": "employee_code",
        # Mapeos adicionales para retrocompatibilidad
        "Fecha Venta": "sale_date",
        "Codigo": "product_code",
        "Cod. Nacional": "product_code",
        "Código Nacional": "product_code",
        "CN": "product_code",
        "Descripcion": "product_name",
        "Artículo": "product_name",
        "Articulo": "product_name",
        "Nombre": "product_name",
        "Laboratorio": "laboratory",
        "Proveedor": "supplier",
        "Unidades": "quantity",
        "Uds": "quantity",
        "PVP": "sale_price",
        "Precio Venta": "sale_price",
        "P.V.P.": "sale_price",
        "PUC": "purchase_price",
        "Precio Compra": "purchase_price",
        "P.U.C": "purchase_price",
        "Importe": "total_amount",
        "Total": "total_amount",
        "Descuento": "discount_amount",
        "Dto": "discount_amount",
        "Dto.": "discount_amount",
        "Margen": "margin_amount",
        "Beneficio": "margin_amount",
        "EAN": "product_ean",
        "Código Barras": "product_ean",
        "Familia": "category",
        "Categoría": "category",
        "Subfamilia": "subcategory",
        "Empleado": "employee_code",
        "Tipo": "product_type",
        "EFG": "is_generic_flag",
    }

    def detect_format(self, df: pd.DataFrame) -> bool:
        """
        Detecta si el DataFrame es del formato Farmatic

        Usa normalización de columnas para ser robusto ante problemas de encoding/locale.
        Resuelve: Archivos cp1252/latin1 leídos en locale UTF-8 (Render) producen
        caracteres corruptos que fallan detección.

        Raises:
            InvalidERPFormatError: Si el archivo está vacío o tiene formato incorrecto
        """
        if df.empty:
            raise InvalidERPFormatError(
                file_name="archivo_farmatic",
                parser_type="Farmatic",
                reason="El archivo está vacío o no contiene datos válidos",
            )

        # Columnas específicas de Farmatic (que NO tiene Farmanager)
        # Basado en archivo real: Fecha, Hora, Código, Aportación, R.P., Importe Neto
        farmatic_indicators = [
            "Aportación",      # Tipo de aportación (R1, etc) - SOLO Farmatic
            "R.P.",            # Precio de referencia - SOLO Farmatic
            "Importe Neto",    # Importe final - Farmatic usa "Neto", Farmanager usa "Importe"
            "Importe Bruto",   # SOLO Farmatic tiene bruto/neto separados
            "C.Cliente",       # Código de cliente - formato Farmatic
        ]

        # Columnas que indican que NO es Farmatic (son de Farmanager)
        farmanager_exclusions = [
            "idOperacion",
            "Fecha/Hora",
            "Artículo",  # Solo en Farmanager
        ]

        # Normalizar columnas del DataFrame (remover acentos, mayúsculas)
        columns_normalized = [self.normalize_column_name(col) for col in df.columns]

        # Log para debugging en producción (nivel WARNING para que aparezca en Render)
        logger.warning(f"[FARMATIC_DETECT] Columnas normalizadas: {columns_normalized[:5]}...")

        # Si tiene columnas exclusivas de Farmanager, NO es Farmatic
        # Usar matching multi-estrategia para robustez
        for exclusion in farmanager_exclusions:
            exclusion_normalized = self.normalize_column_name(exclusion)
            for col in columns_normalized:
                if self.column_matches_indicator(col, exclusion_normalized):
                    logger.warning(f"[FARMATIC_DETECT] Exclusión detectada: {exclusion} → NO es Farmatic")
                    return False

        # Verificar si tiene al menos 2 columnas indicadoras de Farmatic
        # Usar matching multi-estrategia (exacto + substring + fuzzy) para robustez
        matches = 0
        matched_indicators = []
        for indicator in farmatic_indicators:
            indicator_normalized = self.normalize_column_name(indicator)
            # Usar nuevo método de matching robusto (3 estrategias)
            for col in columns_normalized:
                if self.column_matches_indicator(col, indicator_normalized):
                    matches += 1
                    matched_indicators.append(indicator)
                    break  # Ya matcheó, pasar al siguiente indicator

        logger.warning(f"[FARMATIC_DETECT] Matches: {matches}/5, Indicators: {matched_indicators}")

        if matches >= 2:
            logger.warning("Formato Farmatic detectado")
            return True

        logger.warning(f"[FARMATIC_DETECT] Formato NO reconocido (necesita >=2 matches, encontrados: {matches})")
        return False

    def map_columns(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Mapea las columnas de Farmatic al formato normalizado

        Raises:
            MissingRequiredColumnsError: Si faltan columnas críticas
        """
        # Crear un nuevo DataFrame con las columnas mapeadas
        mapped_df = pd.DataFrame()

        # Mapear columnas conocidas
        for farmatic_col, standard_col in self.COLUMN_MAPPING.items():
            # Buscar la columna sin distinguir mayúsculas
            for col in df.columns:
                if col.upper() == farmatic_col.upper():
                    mapped_df[standard_col] = df[col]
                    break

        # Si no encontramos algunas columnas esenciales, intentar con patrones
        if "sale_date" not in mapped_df.columns:
            date_patterns = ["fecha", "date", "fech"]
            for col in df.columns:
                if any(pattern in col.lower() for pattern in date_patterns):
                    mapped_df["sale_date"] = df[col]
                    break

        if "product_code" not in mapped_df.columns:
            code_patterns = ["codigo", "código", "cod", "cn"]
            for col in df.columns:
                if any(pattern in col.lower() for pattern in code_patterns):
                    mapped_df["product_code"] = df[col]
                    break

        # Validar columnas críticas
        # NOTA: total_amount se calcula en clean_data(), no se requiere del CSV
        missing_columns = []
        required = ["sale_date", "product_code", "product_name", "quantity", "sale_price"]
        for col in required:
            if col not in mapped_df.columns or mapped_df[col].isna().all():
                missing_columns.append(col)

        if missing_columns:
            raise MissingRequiredColumnsError(
                file_name="archivo_farmatic", parser_type="Farmatic", missing_columns=missing_columns
            )

        # Mantener columnas no mapeadas por si son útiles
        for col in df.columns:
            if col not in mapped_df.columns:
                mapped_df[f"original_{col}"] = df[col]

        return mapped_df

    def clean_data(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Limpia y prepara los datos específicos de Farmatic

        Raises:
            ProcessingError: Si hay errores durante la limpieza de datos
            InvalidDataTypeError: Si los tipos de datos no son válidos
        """
        try:
            # Limpiar códigos de producto
            if "product_code" in df.columns:
                df["product_code"] = df["product_code"].astype(str).str.strip()

                # NORMALIZACIÓN CÓDIGOS NACIONALES FARMATIC
                # Farmatic usa formato XXXXXX.X (6 dígitos + punto + dígito verificador)
                # CIMA/nomenclator usa solo 6 dígitos → extraer parte antes del punto
                def normalize_farmatic_code(code):
                    if pd.isna(code) or code in ["0", "00000000", "", "nan"]:
                        return None

                    code_str = str(code).strip()

                    # Si tiene punto decimal, extraer solo la parte entera
                    if "." in code_str:
                        code_str = code_str.split(".")[0]

                    # Remover caracteres no numéricos
                    code_str = "".join(c for c in code_str if c.isdigit())

                    if not code_str or code_str == "0":
                        return None

                    # Padding a 6 dígitos con ceros a la izquierda
                    # Ejemplos: "48926" → "048926", "672138" → "672138"
                    return code_str.zfill(6)

                df["product_code"] = df["product_code"].apply(normalize_farmatic_code)

            # Normalizar nombres de productos
            if "product_name" in df.columns:
                df["product_name"] = df["product_name"].apply(self.normalize_product_name)

            # CAPA 1: Normalizar nombres de laboratorios (prevenir corruption UTF-8)
            if "laboratory" in df.columns:
                from app.utils.text_normalization import normalize_text
                df["laboratory"] = df["laboratory"].apply(
                    lambda lab: normalize_text(str(lab), max_length=255, context="laboratory_parser")
                    if pd.notna(lab) else None
                )

            # Detectar genéricos
            if "product_name" in df.columns:
                df["is_generic"] = df.apply(
                    lambda row: self.detect_generic(row.get("product_name", ""), row.get("laboratory", "")),
                    axis=1,
                )

            # Verificar si hay un flag EFG
            if "is_generic_flag" in df.columns:
                df["is_generic"] = df["is_generic_flag"].fillna("").str.upper() == "S"

            # Convertir tipos de producto
            if "product_type" in df.columns:
                type_mapping = {
                    "M": "prescription",  # Medicamento
                    "P": "otc",  # Parafarmacia
                    "EF": "prescription",  # Especialidad Farmacéutica
                    "PS": "otc",  # Producto Sanitario
                    "D": "diet",  # Dietoterápico
                    "F": "formula",  # Fórmula magistral
                }
                df["product_type"] = df["product_type"].map(type_mapping).fillna("other")
            else:
                # Si no hay tipo, intentar deducirlo
                df["product_type"] = "unknown"

            # Calcular márgenes si no existen
            if "margin_amount" not in df.columns:
                if "sale_price" in df.columns and "purchase_price" in df.columns:
                    df["margin_amount"] = df["sale_price"] - df["purchase_price"]
                    df["margin_percentage"] = (df["margin_amount"] / df["sale_price"] * 100).round(2)

            # Procesar fecha y hora si están separadas
            if "sale_date" in df.columns and "sale_time" in df.columns:
                # Combinar fecha y hora en un datetime
                # CRÍTICO: dayfirst=True para formato español DD/MM/YYYY
                df["sale_datetime"] = pd.to_datetime(
                    df["sale_date"].astype(str) + " " + df["sale_time"].astype(str),
                    errors="coerce",
                    dayfirst=True,
                )

            # Limpiar valores numéricos - Farmatic usa coma como separador decimal
            numeric_columns = [
                "quantity",
                "sale_price",
                "purchase_price",
                "net_amount",  # Importe neto (después de copagos)
                "discount_amount",
                "margin_amount",
                "gross_amount",
                "reference_price",
            ]
            for col in numeric_columns:
                if col in df.columns:
                    # Reemplazar comas por puntos para decimales
                    df[col] = df[col].astype(str).str.replace(",", ".")
                    df[col] = pd.to_numeric(df[col], errors="coerce")
                    # Valores negativos en cantidad pueden ser devoluciones
                    if col == "quantity":
                        df["is_return"] = df[col] < 0
                        df[col] = df[col].abs()

            # ✅ CORRECCIÓN CRÍTICA (Issue #XXX): Calcular total_amount correctamente
            # Para análisis de ahorro de genéricos, necesitamos PVP × Cantidad
            # NO usar "Importe Neto" que incluye copagos/descuentos
            if "sale_price" in df.columns and "quantity" in df.columns:
                df["total_amount"] = df["sale_price"] * df["quantity"]
                logger.info("[FARMATIC] Calculado total_amount = sale_price × quantity para análisis de ahorro")
            elif "gross_amount" in df.columns:
                # Fallback: usar importe bruto si está disponible
                df["total_amount"] = df["gross_amount"]
                logger.warning("[FARMATIC] Usando gross_amount como fallback para total_amount")
            elif "net_amount" in df.columns:
                # Último recurso: importe neto (subdeclaración esperada ~50%)
                df["total_amount"] = df["net_amount"]
                logger.warning("[FARMATIC] ADVERTENCIA: Usando net_amount - resultados subdeclarados")

            # Eliminar filas completamente vacías
            df = df.dropna(how="all")

            # Procesar tipo de aportación (R1, etc.) para detectar financiación
            if "contribution_type" in df.columns:
                df["is_financed"] = df["contribution_type"].notna() & (df["contribution_type"] != "")

            # Agregar información específica de Farmatic
            df["erp_type"] = "farmatic"

            return df

        except Exception as e:
            # Si hay un error durante el procesamiento, lanzar excepción específica
            if "quantity" in str(e) or "sale_price" in str(e) or "total_amount" in str(e):
                raise InvalidDataTypeError(
                    file_name="archivo_farmatic",
                    parser_type="Farmatic",
                    column=str(e).split("'")[1] if "'" in str(e) else "columna_numérica",
                    expected_type="numeric",
                )
            else:
                raise ProcessingError(
                    process="limpieza_datos_farmatic",
                    step="clean_data",
                    reason=f"Error procesando datos de Farmatic: {str(e)}",
                )
