# backend/app/parsers/farmanager_parser.py
import logging

import pandas as pd

from app.exceptions import InvalidDataTypeError, InvalidERPFormatError, MissingRequiredColumnsError, ProcessingError
from app.parsers.base_parser import BaseParser

logger = logging.getLogger(__name__)


class FarmanagerParser(BaseParser):
    """
    Parser específico para archivos del ERP Farmanager
    """

    # Mapeo de columnas de Farmanager a nuestro formato basado en datos reales
    COLUMN_MAPPING = {
        # Mapeo basado en muestra real: idOperacion, Fecha/Hora, Artículo, Descripción, Unidades, Precio coste, P.V.P., Descuento, Importe, Empleado, Cliente
        "idOperacion": "operation_id",
        "Fecha/Hora": "sale_datetime",
        "Artículo": "ean13",  # Código EAN13 del producto en Farmanager
        "Descripción": "product_name",
        "Unidades": "quantity",
        "Precio coste": "purchase_price",
        "P.V.P.": "sale_price",
        "Descuento": "discount_amount",
        "Importe": "total_amount",
        "Empleado": "employee_name",
        "Cliente": "client_name",
        # Mapeos adicionales para retrocompatibilidad
        "FechaVenta": "sale_date",
        "Fecha": "sale_date",
        "FechaOperacion": "sale_date",
        "CodigoNacional": "codigo_nacional",
        "CodNacional": "codigo_nacional",
        "CodigoProducto": "ean13",  # Usualmente EAN13 en la mayoría de ERPs
        "CodArticulo": "ean13",
        "DescripcionProducto": "product_name",
        "Descripcion": "product_name",
        "NombreProducto": "product_name",
        "NombreArticulo": "product_name",
        "Laboratorio": "laboratory",
        "Fabricante": "laboratory",
        "Proveedor": "supplier",
        "CantidadVendida": "quantity",
        "Cantidad": "quantity",
        "UnidadesVendidas": "quantity",
        "PrecioVenta": "sale_price",
        "PVP": "sale_price",
        "PrecioUnitario": "unit_price",
        "PrecioCompra": "purchase_price",
        "PC": "purchase_price",
        "ImporteTotal": "total_amount",
        "ImporteVenta": "total_amount",
        "TotalLinea": "total_amount",
        "ImporteDescuento": "discount_amount",
        "Margen": "margin_amount",
        "Beneficio": "margin_amount",
        "CodigoBarras": "ean13",
        "EAN13": "ean13",
        "EAN": "ean13",
        "Familia": "category",
        "GrupoTerapeutico": "therapeutic_group",
        "Subfamilia": "subcategory",
        "TipoProducto": "product_type",
        "TipoArticulo": "product_type",
        "Vendedor": "employee_code",
        "Usuario": "employee_code",
        "NumeroTicket": "invoice_number",
        "NumeroFactura": "invoice_number",
        "TipoCliente": "client_type",
        "Generico": "is_generic_flag",
        "EsGenerico": "is_generic_flag",
    }

    def detect_format(self, df: pd.DataFrame) -> bool:
        """
        Detecta si el DataFrame es del formato Farmanager

        Usa normalización de columnas para ser robusto ante problemas de encoding/locale.
        Resuelve: Archivos cp1252/latin1 leídos en locale UTF-8 (Render) producen
        caracteres corruptos (Artículo → Art�culo) que fallan detección.

        Raises:
            InvalidERPFormatError: Si el archivo está vacío o tiene formato incorrecto
        """
        # DIAGNÓSTICO: Información crítica del DataFrame
        logger.warning("[FARMANAGER_DETECT] === INICIO DETECCIÓN ===")
        logger.warning(f"[FARMANAGER_DETECT] DataFrame shape: {df.shape}")
        logger.warning(f"[FARMANAGER_DETECT] Columnas originales ({len(df.columns)}): {list(df.columns)}")

        if df.empty:
            logger.error("[FARMANAGER_DETECT] ERROR: DataFrame está vacío")
            raise InvalidERPFormatError(
                file_name="archivo_farmanager",
                parser_type="Farmanager",
                reason="El archivo está vacío o no contiene datos válidos",
            )

        # Columnas típicas de Farmanager basadas en datos reales
        farmanager_indicators = [
            "idOperacion",
            "Fecha/Hora",
            "Artículo",
            "Descripción",
            "Unidades",
            "Precio coste",
            "P.V.P.",
            "Descuento",
            "Importe",
            "Empleado",
            "Cliente",
        ]

        # Normalizar columnas del DataFrame (remover acentos, mayúsculas)
        # Esto hace que "Artículo" y "Art�culo" sean equivalentes
        columns_normalized = [self.normalize_column_name(col) for col in df.columns]

        # DIAGNÓSTICO: Normalización completa
        logger.warning(f"[FARMANAGER_DETECT] Columnas normalizadas ({len(columns_normalized)}): {columns_normalized}")

        # DIAGNÓSTICO: Indicators normalizados
        indicators_normalized = [self.normalize_column_name(ind) for ind in farmanager_indicators]
        logger.warning(f"[FARMANAGER_DETECT] Indicators normalizados: {indicators_normalized}")

        # Verificar si tiene al menos 2 columnas indicadoras de Farmanager
        # Usar matching multi-estrategia (exacto + substring + fuzzy) para robustez
        matches = 0
        matched_indicators = []
        match_details = []  # Para logging detallado

        for indicator in farmanager_indicators:
            indicator_normalized = self.normalize_column_name(indicator)
            # Usar nuevo método de matching robusto (3 estrategias)
            for col in columns_normalized:
                # DIAGNÓSTICO: Log de cada intento de match
                match_result = self.column_matches_indicator(col, indicator_normalized)

                if match_result:
                    matches += 1
                    matched_indicators.append(indicator)
                    match_details.append(f"{indicator} -> {col}")
                    logger.warning(f"[FARMANAGER_DETECT] [MATCH] '{indicator}' ({indicator_normalized}) -> '{col}'")
                    break  # Ya matcheó, pasar al siguiente indicator

        # DIAGNÓSTICO: Resumen detallado
        logger.warning("[FARMANAGER_DETECT] === RESULTADO ===")
        logger.warning(f"[FARMANAGER_DETECT] Total matches: {matches}/{len(farmanager_indicators)}")
        logger.warning(f"[FARMANAGER_DETECT] Indicators matcheados: {matched_indicators}")
        logger.warning(f"[FARMANAGER_DETECT] Detalles matches: {match_details}")

        if matches >= 2:
            logger.warning(f"[FARMANAGER_DETECT] [SUCCESS] Formato Farmanager DETECTADO ({matches} matches)")
            return True

        logger.warning(f"[FARMANAGER_DETECT] [FAIL] Formato Farmanager NO detectado (solo {matches} matches, necesita >=2)")
        return False

    def map_columns(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Mapea las columnas de Farmanager al formato normalizado

        Raises:
            MissingRequiredColumnsError: Si faltan columnas críticas
        """
        # Crear un nuevo DataFrame con las columnas mapeadas
        mapped_df = pd.DataFrame()

        # Mapear columnas conocidas (sin espacios, Farmanager usa CamelCase)
        for farmanager_col, standard_col in self.COLUMN_MAPPING.items():
            # Buscar la columna exacta o sin distinguir mayúsculas
            for col in df.columns:
                # Eliminar espacios para comparación
                col_clean = col.replace(" ", "")
                if col_clean.upper() == farmanager_col.upper():
                    mapped_df[standard_col] = df[col]
                    break

        # Si no encontramos algunas columnas esenciales, intentar con patrones
        if "sale_date" not in mapped_df.columns:
            date_patterns = ["fecha", "date"]
            for col in df.columns:
                if any(pattern in col.lower() for pattern in date_patterns):
                    mapped_df["sale_date"] = df[col]
                    break

        if "product_code" not in mapped_df.columns:
            code_patterns = ["codigo", "code", "nacional"]
            for col in df.columns:
                col_lower = col.lower()
                if any(pattern in col_lower for pattern in code_patterns):
                    mapped_df["product_code"] = df[col]
                    break

        # Mantener columnas no mapeadas por si son útiles
        for col in df.columns:
            if col not in mapped_df.columns:
                mapped_df[f"original_{col}"] = df[col]

        # Validar columnas críticas
        missing_columns = []
        required = ["product_name", "quantity", "total_amount"]
        for col in required:
            if col not in mapped_df.columns or mapped_df[col].isna().all():
                missing_columns.append(col)

        # Para Farmanager, necesitamos al menos sale_date o sale_datetime
        if "sale_date" not in mapped_df.columns and "sale_datetime" not in mapped_df.columns:
            missing_columns.append("sale_date o sale_datetime")

        if missing_columns:
            raise MissingRequiredColumnsError(
                file_name="archivo_farmanager", parser_type="Farmanager", missing_columns=missing_columns
            )

        return mapped_df

    def clean_data(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Limpia y prepara los datos específicos de Farmanager

        Raises:
            ProcessingError: Si hay errores durante la limpieza de datos
            InvalidDataTypeError: Si los tipos de datos no son válidos
        """
        try:
            # Procesar EAN13 y extraer código nacional
            if "ean13" in df.columns:
                df["ean13"] = df["ean13"].astype(str).str.strip()
                # Eliminar prefijos comunes en Farmanager
                df["ean13"] = df["ean13"].str.replace("CN:", "", regex=False)
                df["ean13"] = df["ean13"].str.replace("EAN:", "", regex=False)

                # CONVERSIÓN CRÍTICA: EAN847000 → Código Nacional
                # Si el EAN13 empieza por 847000, los siguientes 6 caracteres son el CN
                def extract_codigo_nacional(ean_code):
                    if pd.isna(ean_code) or ean_code == "":
                        return None
                    ean_str = str(ean_code).strip()
                    if len(ean_str) >= 12 and ean_str.startswith("847000"):
                        # Extraer los 6 dígitos siguientes como CN
                        cn = ean_str[6:12]
                        logger.debug(f"Extrayendo CN {cn} de EAN13 {ean_str}")
                        return cn
                    return None

                # Crear campo código nacional basado en EAN13
                df["codigo_nacional"] = df["ean13"].apply(extract_codigo_nacional)

                # Eliminar EAN13 inválidos
                df.loc[df["ean13"].isin(["0", "00000000", "", "SIN CODIGO"]), "ean13"] = None

            # Normalizar nombres de productos
            if "product_name" in df.columns:
                df["product_name"] = df["product_name"].apply(self.normalize_product_name)
                # Farmanager a veces incluye el código en el nombre
                df["product_name"] = df["product_name"].str.replace(r"\[\d+\]", "", regex=True)

            # CAPA 1: Normalizar nombres de laboratorios (prevenir corruption UTF-8)
            if "laboratory" in df.columns:
                from app.utils.text_normalization import normalize_text
                df["laboratory"] = df["laboratory"].apply(
                    lambda lab: normalize_text(str(lab), max_length=255, context="laboratory_parser")
                    if pd.notna(lab) else None
                )

            # Detectar genéricos
            if "is_generic_flag" in df.columns:
                # Farmanager puede usar 'S'/'N' o 'SI'/'NO' o 1/0
                df["is_generic"] = df["is_generic_flag"].astype(str).str.upper().isin(["S", "SI", "1", "TRUE"])
            elif "product_name" in df.columns:
                df["is_generic"] = df.apply(
                    lambda row: self.detect_generic(row.get("product_name", ""), row.get("laboratory", "")),
                    axis=1,
                )

            # Convertir tipos de producto (Farmanager usa códigos numéricos)
            if "product_type" in df.columns:
                # CRÍTICO: Solo 'prescription' o 'venta_libre' son válidos en sales_enrichment
                # Este mapeo es para datos temporales del parser, pero enrichment_service
                # sobrescribirá usando _derive_product_type() basado en el catálogo
                type_mapping = {
                    "1": "prescription",  # Medicamento con receta
                    "2": "venta_libre",   # Venta libre (antes: otc)
                    "3": "venta_libre",   # Parafarmacia → venta_libre
                    "4": "venta_libre",   # Dietético → venta_libre
                    "5": "prescription",  # Fórmula magistral → prescription
                    "6": "venta_libre",   # Producto sanitario → venta_libre
                    "MEDICAMENTO": "prescription",
                    "PARAFARMACIA": "venta_libre",
                    "OTC": "venta_libre",
                    "DIETETICO": "venta_libre",
                }
                df["product_type"] = df["product_type"].astype(str).str.upper()
                df["product_type"] = df["product_type"].map(lambda x: type_mapping.get(x, "other"))
            else:
                df["product_type"] = "unknown"

            # Procesar grupo terapéutico si existe
            if "therapeutic_group" in df.columns:
                df["therapeutic_group"] = df["therapeutic_group"].str.strip().str.upper()

            # Calcular márgenes si no existen
            if "margin_amount" not in df.columns:
                if "sale_price" in df.columns and "purchase_price" in df.columns:
                    df["margin_amount"] = (df["sale_price"] - df["purchase_price"]).round(2)
                    # Evitar división por cero
                    df["margin_percentage"] = df.apply(
                        lambda row: (
                            round((row["margin_amount"] / row["sale_price"] * 100), 2) if row["sale_price"] > 0 else 0
                        ),
                        axis=1,
                    )

            # Procesar datetime si viene en formato combinado
            # CRÍTICO: dayfirst=True para formato español DD/MM/YYYY
            if "sale_datetime" in df.columns:
                df["sale_datetime"] = pd.to_datetime(df["sale_datetime"], errors="coerce", dayfirst=True)
                # Extraer fecha y hora por separado
                df["sale_date"] = df["sale_datetime"].dt.date
                df["sale_time"] = df["sale_datetime"].dt.time

            # Limpiar valores numéricos (Farmanager puede usar coma como separador decimal)
            numeric_columns = [
                "quantity",
                "sale_price",
                "purchase_price",
                "total_amount",
                "discount_amount",
                "margin_amount",
                "unit_price",
            ]
            for col in numeric_columns:
                if col in df.columns:
                    # Reemplazar comas por puntos para decimales
                    df[col] = df[col].astype(str).str.replace(",", ".")
                    df[col] = pd.to_numeric(df[col], errors="coerce")
                    # Valores negativos en cantidad pueden ser devoluciones
                    if col == "quantity":
                        df["is_return"] = df[col] < 0
                        df[col] = df[col].abs()

            # Procesar tipos de cliente si existe
            if "client_type" in df.columns:
                client_mapping = {
                    "PARTICULAR": "regular",
                    "MUTUA": "insurance",
                    "RESIDENCIA": "residence",
                    "HOSPITAL": "hospital",
                }
                df["client_type"] = df["client_type"].astype(str).str.upper()
                df["client_type"] = df["client_type"].map(lambda x: client_mapping.get(x, "other"))

            # Eliminar filas completamente vacías
            df = df.dropna(how="all")

            # Detectar devoluciones o cancelaciones por null en cliente
            if "client_name" in df.columns:
                df["is_cancelled"] = df["client_name"].astype(str).str.lower().isin(["null", "none", ""])

            # Agregar información específica de Farmanager
            df["erp_type"] = "farmanager"

            return df

        except Exception as e:
            # Si hay un error durante el procesamiento, lanzar excepción específica
            if "quantity" in str(e) or "sale_price" in str(e) or "total_amount" in str(e):
                raise InvalidDataTypeError(
                    file_name="archivo_farmanager",
                    parser_type="Farmanager",
                    column=str(e).split("'")[1] if "'" in str(e) else "columna_numérica",
                    expected_type="numeric",
                )
            else:
                raise ProcessingError(
                    process="limpieza_datos_farmanager",
                    step="clean_data",
                    reason=f"Error procesando datos de Farmanager: {str(e)}",
                )
