﻿# backend/app/parsers/base_parser.py
import logging
import re
import unicodedata
from abc import ABC, abstractmethod
from pathlib import Path
from typing import Dict, Optional, Tuple

import pandas as pd

from ..utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)


class BaseParser(ABC):
    """
    Clase base para todos los parsers de ERPs farmacéuticos
    """

    # Columnas requeridas en el formato normalizado
    REQUIRED_COLUMNS = ["sale_date", "product_name", "quantity", "total_amount"]

    # Columnas de producto (se requiere al menos una)
    PRODUCT_ID_COLUMNS = ["ean13", "codigo_nacional"]

    # Columnas opcionales pero útiles
    OPTIONAL_COLUMNS = [
        "product_ean",
        "laboratory",
        "unit_price",
        "purchase_price",
        "sale_price",
        "discount_amount",
        "margin_amount",
        "supplier",
        "category",
        "is_generic",
        "employee_code",
        "client_type",
    ]

    def __init__(self, pharmacy_id: str, upload_id: str):
        self.pharmacy_id = pharmacy_id
        self.upload_id = upload_id
        self.errors = []
        self.warnings = []
        self.manual_selection = False  # Flag para bypass de detect_format()

    @staticmethod
    def normalize_column_name(column_name: str) -> str:
        """
        Normaliza nombres de columnas para comparación insensible a encoding/locale.

        Resuelve Issue: Archivos CSV con encoding cp1252/latin1 leídos en locale UTF-8
        producen caracteres corruptos (Artículo → Art�culo) que fallan la detección.

        Esta normalización:
        1. Descompone acentos con NFKD (Artículo → A-r-t-i-́-c-u-l-o)
        2. Remueve caracteres combinantes (acentos) y no-alfanuméricos (incluyendo �)
        3. Mantiene SOLO letras y números
        4. Convierte a mayúsculas

        Args:
            column_name: Nombre de columna original (puede tener encoding corrupto)

        Returns:
            Nombre normalizado: solo letras/números, sin acentos, mayúsculas

        Examples:
            >>> BaseParser.normalize_column_name("Artículo")
            'ARTICULO'
            >>> BaseParser.normalize_column_name("Art�culo")  # encoding corrupto
            'ARTICULO'
            >>> BaseParser.normalize_column_name("Descripción")
            'DESCRIPCION'
            >>> BaseParser.normalize_column_name("P.V.P.")
            'PVP'
        """
        # Paso 1: Normalizar NFKD para descomponer acentos
        # Ejemplo: "í" (U+00ED) → "i" (U+0069) + "´" (U+0301 combining acute)
        try:
            nfkd_form = unicodedata.normalize('NFKD', column_name)
        except Exception:
            nfkd_form = column_name

        # Paso 2: Remover acentos/combinantes y mantener SOLO alfanuméricos
        # c.isalnum() = True solo para letras (A-Z, a-z) y dígitos (0-9)
        # Esto ELIMINA automáticamente: �, espacios, puntos, /, -, etc.
        cleaned_chars = [c for c in nfkd_form if not unicodedata.combining(c) and c.isalnum()]

        # Paso 3: Juntar caracteres
        temp_result = "".join(cleaned_chars)

        # Paso 4: Convertir a mayúsculas y eliminar cualquier residuo no alfanumérico
        # (redundante después de isalnum(), pero garantiza limpieza total)
        final_result = re.sub(r'[^A-Z0-9]', '', temp_result.upper())

        return final_result

    @staticmethod
    def column_matches_indicator(
        col_normalized: str,
        indicator_normalized: str,
        min_length: int = 3,
        fuzzy_threshold: float = 0.85
    ) -> bool:
        """
        Matching robusto de columna normalizada contra indicator esperado.

        Usa estrategia multi-nivel para manejar casos edge de encoding corrupto:
        1. Match exacto (más confiable)
        2. Substring bidireccional (maneja caracteres faltantes)
        3. Fuzzy matching (fallback para casos extremos)

        Args:
            col_normalized: Nombre de columna normalizado (ej: "ARTICULO")
            indicator_normalized: Indicator esperado normalizado (ej: "ARTICULO")
            min_length: Longitud mínima para considerar match válido (default: 3)
            fuzzy_threshold: Ratio mínimo de similitud para fuzzy match (default: 0.85)

        Returns:
            True si hay match por cualquiera de las 3 estrategias, False si no

        Examples:
            >>> BaseParser.column_matches_indicator("ARTICULO", "ARTICULO")
            True  # Match exacto
            >>> BaseParser.column_matches_indicator("ARTCULO", "ARTICULO")
            True  # Substring (falta 'I')
            >>> BaseParser.column_matches_indicator("ARTCLO", "ARTICULO")
            True  # Fuzzy (86% similar)
            >>> BaseParser.column_matches_indicator("FECHA", "ARTICULO")
            False  # Sin match
        """
        # Validación: longitudes mínimas
        if len(col_normalized) < min_length or len(indicator_normalized) < min_length:
            return False

        # Estrategia 1: Match exacto (caso ideal)
        if col_normalized == indicator_normalized:
            return True

        # Estrategia 2: Substring bidireccional (maneja 1-2 caracteres faltantes)
        # Ejemplo: "ARTICULO" in "ARTCULO" or "ARTCULO" in "ARTICULO" → True
        if indicator_normalized in col_normalized or col_normalized in indicator_normalized:
            return True

        # Estrategia 3: Fuzzy matching (fallback para 3+ caracteres corruptos)
        # SequenceMatcher ratio: 1.0 = idéntico, 0.0 = completamente diferente
        # Usamos threshold=0.85 para permitir ~15% de diferencia
        try:
            from difflib import SequenceMatcher
            ratio = SequenceMatcher(None, col_normalized, indicator_normalized).ratio()
            if ratio >= fuzzy_threshold:
                return True
        except Exception:
            # Si falla fuzzy matching, no bloqueamos el flujo
            pass

        return False

    @abstractmethod
    def detect_format(self, df: pd.DataFrame) -> bool:
        """
        Detecta si el DataFrame corresponde a este formato de ERP
        Returns: True si el formato es correcto, False si no
        """
        pass

    @abstractmethod
    def map_columns(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Mapea las columnas del formato específico al formato normalizado
        """
        pass

    @abstractmethod
    def clean_data(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Limpia y prepara los datos específicos del formato
        """
        pass

    def parse_file(self, file_path: str, encoding: str = "utf-8") -> Tuple[pd.DataFrame, Dict]:
        """
        Pipeline completo de parseo
        Returns: (DataFrame normalizado, diccionario de estadísticas)
        """
        stats = {
            "total_rows": 0,
            "processed_rows": 0,
            "error_rows": 0,
            "warnings": [],
            "errors": [],
        }

        try:
            # 1. Leer archivo
            df = self.read_file(file_path, encoding)
            stats["total_rows"] = len(df)

            # 2. Validar formato (SOLO si NO es selección manual del usuario)
            # CRITICAL FIX: Cuando el usuario selecciona manualmente el parser en frontend,
            # confiamos en su elección y NO validamos con detect_format()
            # Esto resuelve el problema de archivos Farmanager mal clasificados como Farmatic
            if not self.manual_selection:
                logger.info("[PARSE_FILE] Validando formato con detect_format() (auto-detection)")
                if not self.detect_format(df):
                    raise ValueError("El archivo no corresponde al formato esperado")
            else:
                logger.warning("[PARSE_FILE] BYPASS detect_format() - selección manual del usuario")
                logger.warning(f"[PARSE_FILE] Parser: {self.__class__.__name__}, confiando en elección del usuario")

            # 3. Mapear columnas
            df = self.map_columns(df)

            # 4. Limpiar datos
            df = self.clean_data(df)

            # 5. Procesar códigos de producto (EAN13 y código nacional)
            df = self.process_product_codes(df)

            # 6. Validar datos requeridos
            df = self.validate_required_fields(df)

            # 7. Agregar metadatos
            df = self.add_metadata(df)

            # 8. Filtrar filas válidas
            valid_df = df[df["is_valid"]].copy()
            stats["processed_rows"] = len(valid_df)
            stats["error_rows"] = len(df) - len(valid_df)

            # Agregar warnings y errors
            stats["warnings"] = self.warnings
            stats["errors"] = self.errors

            return valid_df, stats

        except Exception as e:
            logger.error(f"Error parseando archivo: {str(e)}")
            stats["errors"].append(str(e))
            return pd.DataFrame(), stats

    def read_file(self, file_path: str, encoding: str = "utf-8") -> pd.DataFrame:
        """
        Lee el archivo según su extensión
        """
        path = Path(file_path)
        extension = path.suffix.lower()

        try:
            if extension == ".csv":
                # Intentar diferentes separadores comunes
                for sep in [";", ",", "\t", "|"]:
                    try:
                        df = pd.read_csv(file_path, encoding=encoding, sep=sep)
                        if len(df.columns) > 1:  # Verificar que se separó correctamente
                            return df
                    except (
                        pd.errors.ParserError,
                        pd.errors.EmptyDataError,
                        UnicodeDecodeError,
                        MemoryError,
                        IOError,
                        OSError,
                    ) as e:
                        logger.debug(f"Failed to read CSV with separator '{sep}' and encoding '{encoding}': {e}")
                        continue

                # Si llegamos aquí, no se pudo leer con ningún separador
                raise ValueError(f"No se pudo leer el archivo CSV con encoding {encoding}")

            elif extension in [".xlsx", ".xls"]:
                df = pd.read_excel(file_path)
                return df

            else:
                raise ValueError(f"Formato de archivo no soportado: {extension}")

        except Exception as e:
            # Issue #412: Cambiado de ERROR a WARNING - el sistema puede reintentar con otro encoding
            logger.warning(f"[ENCODING] Intento de lectura falló: {str(e)}")
            raise

    def validate_required_fields(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Valida que los campos requeridos estén presentes y sean válidos
        """
        df["is_valid"] = True

        # Log columnas disponibles para debugging
        logger.debug(f"Columnas disponibles en DataFrame: {list(df.columns)}")
        logger.debug(f"Total filas a validar: {len(df)}")

        # Validar columnas básicas requeridas
        for col in self.REQUIRED_COLUMNS:
            if col not in df.columns:
                self.errors.append(f"Columna requerida no encontrada: {col}")
                df["is_valid"] = False
            else:
                # Marcar como inválidas las filas con valores nulos en campos requeridos
                null_count = df[col].isna().sum()
                if null_count > 0:
                    logger.debug(f"Columna '{col}': {null_count} filas con valores nulos")
                df.loc[df[col].isna(), "is_valid"] = False

        # Validar que al menos una columna de producto esté presente
        product_id_columns_present = [col for col in self.PRODUCT_ID_COLUMNS if col in df.columns]

        # Log detallado de validación de columnas de producto
        # Ayuda a diagnosticar errores de "DataFrame vacío después de parseo"
        logger.debug(f"PRODUCT_ID_COLUMNS requeridas: {self.PRODUCT_ID_COLUMNS}")
        logger.debug(f"PRODUCT_ID_COLUMNS encontradas: {product_id_columns_present}")

        if not product_id_columns_present:
            logger.warning(
                f"❌ Ninguna columna de producto encontrada. Requeridas: {self.PRODUCT_ID_COLUMNS}, "
                f"Disponibles: {list(df.columns)}"
            )
            self.errors.append(
                f"Se requiere al menos una columna de identificación de producto: {', '.join(self.PRODUCT_ID_COLUMNS)}"
            )
            df["is_valid"] = False
        else:
            logger.debug(f"✅ Columnas de producto válidas: {product_id_columns_present}")

            # Marcar como inválidas las filas que no tengan ningún identificador de producto válido
            has_product_id = pd.Series(False, index=df.index)
            for col in product_id_columns_present:
                non_null_count = df[col].notna().sum()
                logger.debug(f"Columna '{col}': {non_null_count} filas con valores no-nulos")
                has_product_id |= df[col].notna()

            invalid_product_id_count = (~has_product_id).sum()
            if invalid_product_id_count > 0:
                logger.debug(f"Marcando {invalid_product_id_count} filas como inválidas (sin código de producto)")
            df.loc[~has_product_id, "is_valid"] = False

        # Validaciones específicas
        if "sale_date" in df.columns:
            # Convertir a datetime si no lo es
            # CRÍTICO: dayfirst=True para formato español DD/MM/YYYY
            # Sin esto, "01/10/2024" se interpreta como 10 de enero (MM/DD/YYYY americano)
            # en lugar de 1 de octubre (DD/MM/YYYY español)
            try:
                df["sale_date"] = pd.to_datetime(df["sale_date"], errors="coerce", dayfirst=True)
                df.loc[df["sale_date"].isna(), "is_valid"] = False
            except (TypeError, ValueError, KeyError) as e:
                logger.warning(f"Error parsing dates: {e}")
                self.warnings.append("Algunas fechas no pudieron ser parseadas")

        # Validar cantidades y montos numéricos
        numeric_fields = ["quantity", "total_amount", "unit_price", "margin_amount"]
        for field in numeric_fields:
            if field in df.columns:
                df[field] = pd.to_numeric(df[field], errors="coerce")

        return df

    def extract_national_code_from_ean(self, ean_code: str) -> Optional[str]:
        """
        Extrae código nacional de EAN si es español

        Args:
            ean_code: Código EAN completo

        Returns:
            Código nacional (6 dígitos) o None si no es válido
        """
        if not ean_code or not isinstance(ean_code, str):
            return None

        ean_clean = ean_code.strip()
        if len(ean_clean) >= 12 and ean_clean.startswith("847000"):
            return ean_clean[6:12]  # Caracteres 7-12 (índices 6-11)
        return None

    def process_product_codes(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Procesa códigos de producto para extraer EAN13 y código nacional

        Args:
            df: DataFrame con datos de ventas

        Returns:
            DataFrame con campos ean13 y codigo_nacional poblados
        """
        # Asegurar que las columnas existen
        if "ean13" not in df.columns:
            df["ean13"] = None
        if "codigo_nacional" not in df.columns:
            df["codigo_nacional"] = None

        # Si tenemos product_ean, copiarlo a ean13
        if "product_ean" in df.columns and df["product_ean"].notna().any():
            df["ean13"] = df["product_ean"].fillna(df["ean13"])

        # Si tenemos product_code y parece un EAN, copiarlo a ean13
        if "product_code" in df.columns:
            # Detectar códigos que parecen EAN (13 dígitos numéricos)
            product_code_str = df["product_code"].astype(str)
            ean_mask = (product_code_str.str.len() == 13) & (product_code_str.str.isdigit())
            df.loc[ean_mask & df["ean13"].isna(), "ean13"] = df.loc[ean_mask, "product_code"]

        # Extraer código nacional de EAN13 para códigos españoles
        if df["ean13"].notna().any():
            df["codigo_nacional"] = df["ean13"].apply(
                lambda ean: (self.extract_national_code_from_ean(ean) if pd.notna(ean) else None)
            )

        # NUEVO: Farmatic usa códigos nacionales (6-7 dígitos), NO EAN13
        # Si product_code existe pero NO se mapeó a ean13 (porque no es EAN),
        # copiarlo directamente a codigo_nacional
        # Fix para error "DataFrame vacío después de parseo" cuando archivos Farmatic
        # tienen solo códigos nacionales sin EAN13 (formato esperado)
        if "product_code" in df.columns:
            # Máscara: tiene product_code pero no tiene codigo_nacional todavía
            mask_needs_cn = df["product_code"].notna() & df["codigo_nacional"].isna()
            if mask_needs_cn.any():
                df.loc[mask_needs_cn, "codigo_nacional"] = df.loc[mask_needs_cn, "product_code"]
                logger.debug(
                    f"Mapeado {mask_needs_cn.sum()} códigos de product_code → codigo_nacional (códigos NO-EAN)"
                )

        return df

    def add_metadata(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Agrega metadatos necesarios para el sistema
        """
        df["pharmacy_id"] = self.pharmacy_id
        df["upload_id"] = self.upload_id
        df["created_at"] = utc_now()

        # Extraer componentes de fecha si existen
        if "sale_date" in df.columns:
            df["year"] = df["sale_date"].dt.year
            df["month"] = df["sale_date"].dt.month
            df["day"] = df["sale_date"].dt.day

        return df

    def normalize_product_name(self, name: str) -> str:
        """
        Normaliza nombres de productos (quitar espacios extra, mayúsculas, etc.)

        CAPA 1: Usa normalize_text() para prevenir corruption de encoding UTF-8
        en cloud providers (Render, AWS, Azure).
        """
        if pd.isna(name):
            return ""

        # Aplicar normalización defensiva UTF-8
        from app.utils.text_normalization import normalize_text
        normalized = normalize_text(str(name), max_length=500, context="product_name_parser")

        return normalized.upper() if normalized else ""

    def detect_generic(self, product_name: str, laboratory: str = None) -> bool:
        """
        Detecta si un producto es genérico basándose en el nombre y laboratorio
        """
        if pd.isna(product_name):
            return False

        name_upper = str(product_name).upper()
        generic_indicators = ["EFG", "GENERICO", "GENERIC"]

        # Verificar indicadores en el nombre
        for indicator in generic_indicators:
            if indicator in name_upper:
                return True

        # Verificar laboratorios de genéricos conocidos
        if laboratory:
            lab_upper = str(laboratory).upper()
            generic_labs = ["CINFA", "NORMON", "STADA", "KERN", "SANDOZ", "TEVA"]
            for lab in generic_labs:
                if lab in lab_upper:
                    return True

        return False
