# backend/app/models/user.py
import uuid

from sqlalchemy import Boolean, Column, DateTime, ForeignKey, String
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship

from app.core.subscription_limits import ROLE_PERMISSIONS  # Issue #399: New permission system
from app.database import Base
from app.utils.datetime_utils import utc_now


class User(Base):
    """Modelo para usuarios del sistema"""

    __tablename__ = "users"
    __table_args__ = {"extend_existing": True}

    # Primary key
    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)

    # Foreign keys
    pharmacy_id = Column(
        UUID(as_uuid=True),
        ForeignKey("pharmacies.id", ondelete="CASCADE"),
        nullable=False,  # OBLIGATORIO: Cada usuario debe tener una farmacia
        unique=True,  # ✅ RESTAURADO: Relación 1:1 (una farmacia = un usuario)
        index=True,  # Índice para performance
    )

    # Issue #348: RBAC (Role-Based Access Control)
    role_id = Column(
        UUID(as_uuid=True),
        ForeignKey("roles.id", ondelete="RESTRICT"),
        nullable=False,
        index=True,
        server_default="33333333-3333-3333-3333-333333333333"  # Default to 'user' role
    )

    # Issue #348: Subscription management
    subscription_plan = Column(
        String(50),
        nullable=False,
        default="free",
        index=True
    )

    # Issue #444: Subscription expiration dates
    subscription_start = Column(DateTime(timezone=True), nullable=True)
    subscription_end = Column(DateTime(timezone=True), nullable=True, index=True)

    # Issue #348: Soft delete (GDPR Article 17)
    deleted_at = Column(DateTime(timezone=True), nullable=True)

    # Información de autenticación
    email = Column(String(255), unique=True, nullable=False, index=True)
    username = Column(String(100), unique=True, nullable=False, index=True)
    hashed_password = Column(String(255), nullable=True)  # Nullable for OAuth users

    # OAuth fields
    oauth_provider = Column(String(50), nullable=True)  # 'google', 'microsoft', etc.
    oauth_provider_id = Column(String(255), nullable=True)  # User ID from OAuth provider

    # Información personal
    full_name = Column(String(255))
    phone = Column(String(20))
    dni_nie = Column(String(20), nullable=True, unique=True)  # DNI/NIE español para cumplimiento

    # Roles y permisos
    role = Column(String(50), default="user")  # 'admin', 'user' only (simplified from 4 roles)
    is_active = Column(Boolean, default=True)
    is_superuser = Column(Boolean, default=False)
    is_verified = Column(Boolean, default=False)

    # Timestamps
    created_at = Column(DateTime, default=utc_now)
    updated_at = Column(DateTime, default=utc_now, onupdate=utc_now)
    last_login = Column(DateTime)
    email_verified_at = Column(DateTime)

    # Configuración de notificaciones
    notify_uploads = Column(Boolean, default=True)
    notify_errors = Column(Boolean, default=True)
    notify_analysis = Column(Boolean, default=True)

    # Relaciones con otras tablas
    pharmacy = relationship("Pharmacy", back_populates="user")  # ✅ Singular (relación 1:1)
    uploads = relationship("FileUpload", back_populates="user", cascade="all, delete-orphan")

    # Issue #348: RBAC relationship
    role_obj = relationship("Role", back_populates="users")

    def __repr__(self):
        return f"<User {self.email}>"

    def to_dict(self):
        """Convierte el modelo a diccionario para APIs"""
        return {
            "id": str(self.id),
            "pharmacy_id": str(self.pharmacy_id) if self.pharmacy_id else None,
            "email": self.email,
            "username": self.username,
            "full_name": self.full_name,
            "phone": self.phone,
            "role": self.role,
            "role_id": str(self.role_id) if self.role_id else None,  # Issue #348
            "subscription_plan": self.subscription_plan,  # Issue #348
            "subscription_start": self.subscription_start.isoformat() if self.subscription_start else None,  # Issue #444
            "subscription_end": self.subscription_end.isoformat() if self.subscription_end else None,  # Issue #444
            "is_active": self.is_active,
            "is_verified": self.is_verified,
            "created_at": self.created_at.isoformat() if self.created_at else None,
            "last_login": self.last_login.isoformat() if self.last_login else None,
            "deleted_at": self.deleted_at.isoformat() if self.deleted_at else None,  # Issue #348
        }

    def has_permission(self, permission: str) -> bool:
        """
        Verifica si el usuario tiene un permiso específico.

        Issue #348: Prioridad de verificación:
        1. Superuser → TRUE
        2. Soft deleted → FALSE
        3. Inactive → FALSE
        4. Role-based → Check role_obj.has_permission()
        5. Fallback → Legacy role field

        Args:
            permission: Permission constant from app.core.permissions

        Returns:
            True if user has the permission, False otherwise
        """
        # Superusers have all permissions
        if self.is_superuser:
            return True

        # Issue #348: Soft deleted users have no permissions
        if self.deleted_at:
            return False

        if not self.is_active:
            return False

        # Issue #348: Check role_obj first (RBAC)
        if self.role_obj:
            return self.role_obj.has_permission(permission)

        # Fallback to legacy role field (for backwards compatibility)
        # Issue #399: Use subscription_limits ROLE_PERMISSIONS (includes VIEW_SYSTEM_STATS)
        user_permissions = ROLE_PERMISSIONS.get(self.role, ())
        return permission in user_permissions
