"""
Modelos de Tracking para Modulos de Control
============================================

Estos modelos representan el estado LOCAL de kaiFarma, separado de los datos del ERP.
El ERP es la fuente de verdad para los datos crudos, pero el tracking de acciones
(contactado, marcado para devolucion, etc.) vive aqui.

Principio: El ERP es READ-ONLY. Estos modelos son READ-WRITE en SQLite local.
"""

from datetime import datetime
from typing import Optional

from sqlalchemy import Column, DateTime, Float, Integer, String, Text

from app.database import Base


class CustomerOrderTracking(Base):
    """
    Tracking de encargos de clientes.

    Combina:
    - ID del encargo en el ERP (inmutable, PK)
    - Estado local de nuestra maquina de estados
    - Timestamps de acciones del farmaceutico
    - Cache de datos para UI (evitar round-trips al ERP)

    Estados posibles:
    - NEW: Detectado por primera vez en el ERP
    - ALERTED: Mostrado en el dashboard (pero ignorado por el usuario)
    - CONTACTED: El farmaceutico pulso "Ya he llamado"
    - TO_RETURN: Marcado para devolucion al proveedor
    - RESOLVED: Desaparecio del ERP (recogido o devuelto)
    """

    __tablename__ = "tracking_encargos"

    # Primary Key vinculada al ID del ERP (ej: "ENC-17399-45123")
    erp_order_id = Column(String(100), primary_key=True)

    # Datos cacheados para performance y ordenacion
    cached_pvl = Column(Float, default=0.0, comment="Valor coste inmovilizado (EUR)")
    cached_units = Column(Integer, default=1, comment="Unidades encargadas")
    customer_name = Column(String(200), default="Desconocido", comment="Snapshot del nombre cliente")
    customer_phone = Column(String(50), nullable=True, comment="Telefono para llamar")
    product_description = Column(String(500), default="", comment="Descripcion del producto")
    product_cn = Column(String(20), nullable=True, comment="Codigo Nacional")

    # Maquina de Estados Local
    status = Column(String(20), default="NEW")  # NEW, ALERTED, CONTACTED, TO_RETURN, RESOLVED

    # Timestamps
    first_detected_at = Column(DateTime, default=datetime.utcnow)
    last_seen_in_erp = Column(
        DateTime,
        default=datetime.utcnow,
        comment="Ultima vez que vimos este encargo en el ERP (para reconciliacion)"
    )
    alerted_at = Column(DateTime, nullable=True)
    contacted_at = Column(DateTime, nullable=True)
    marked_return_at = Column(DateTime, nullable=True)
    resolved_at = Column(DateTime, nullable=True)

    # Notas del farmaceutico
    notes = Column(Text, nullable=True)

    @property
    def days_pending(self) -> int:
        """Dias naturales desde que lo detectamos por primera vez"""
        if self.first_detected_at is None:
            return 0
        return (datetime.utcnow() - self.first_detected_at).days

    @property
    def severity(self) -> str:
        """
        Semaforo de urgencia para el UI.

        Retorna:
            - "success": Resuelto (verde)
            - "info": Reciente o marcado para devolucion (gris/azul)
            - "warning": 5-9 dias pendiente (amarillo)
            - "urgent": 10-14 dias pendiente (naranja/rojo claro)
            - "critical": 15+ dias pendiente (rojo oscuro)
        """
        if self.status == "RESOLVED":
            return "success"
        if self.status == "TO_RETURN":
            return "info"

        days = self.days_pending
        if days >= 15:
            return "critical"  # Devolver YA
        if days >= 10:
            return "urgent"    # Llamar YA
        if days >= 5:
            return "warning"   # Atento
        return "info"          # Reciente, sin urgencia

    @property
    def is_spam(self) -> bool:
        """
        True si ya molestamos al farmaceutico/cliente recientemente.

        Evita:
        - Mostrar alerta repetidamente si ya contactamos hace <2 dias
        - Saturar el dashboard con el mismo encargo

        Returns:
            True si debemos silenciar este encargo temporalmente
        """
        if self.status == "CONTACTED" and self.contacted_at:
            days_since_contact = (datetime.utcnow() - self.contacted_at).days
            return days_since_contact < 2
        return False

    @property
    def economic_value(self) -> float:
        """Valor economico total inmovilizado"""
        return (self.cached_pvl or 0.0) * (self.cached_units or 1)

    def mark_as_alerted(self) -> None:
        """Transicion: NEW -> ALERTED"""
        if self.status == "NEW":
            self.status = "ALERTED"
            self.alerted_at = datetime.utcnow()

    def mark_as_contacted(self, notes: Optional[str] = None) -> None:
        """Transicion: * -> CONTACTED"""
        self.status = "CONTACTED"
        self.contacted_at = datetime.utcnow()
        if notes:
            self.notes = notes

    def mark_for_return(self, notes: Optional[str] = None) -> None:
        """Transicion: * -> TO_RETURN"""
        self.status = "TO_RETURN"
        self.marked_return_at = datetime.utcnow()
        if notes:
            self.notes = notes

    def mark_as_resolved(self) -> None:
        """Transicion: * -> RESOLVED (normalmente por reconciliacion automatica)"""
        self.status = "RESOLVED"
        self.resolved_at = datetime.utcnow()

    def update_from_erp(
        self,
        pvl: float,
        units: int,
        customer_name: str,
        customer_phone: Optional[str] = None,
        product_description: str = "",
        product_cn: Optional[str] = None,
    ) -> None:
        """
        Actualiza cache con datos frescos del ERP.
        Llamar en cada sync para mantener datos actualizados.
        """
        self.cached_pvl = pvl
        self.cached_units = units
        self.customer_name = customer_name
        self.customer_phone = customer_phone
        self.product_description = product_description
        self.product_cn = product_cn
        self.last_seen_in_erp = datetime.utcnow()


class OrderProposalAnalysisCache(Base):
    """
    Cache de analisis de propuestas de pedido (Modulo 2: Interceptor).

    Guarda el resultado del ultimo analisis para:
    - Mostrar historial
    - Calcular metricas de ahorro acumulado
    - Evitar re-analizar si no hay cambios
    """

    __tablename__ = "proposal_analysis_cache"

    id = Column(Integer, primary_key=True, autoincrement=True)

    # Identificacion
    proposal_id = Column(String(100), index=True)  # ID de la propuesta en el ERP
    supplier_code = Column(String(50), default="")
    supplier_name = Column(String(200), default="")

    # Resultado del analisis
    analyzed_at = Column(DateTime, default=datetime.utcnow)
    total_lines = Column(Integer, default=0)
    lines_ok = Column(Integer, default=0)
    lines_warning = Column(Integer, default=0)
    lines_blocker = Column(Integer, default=0)
    lines_opportunity = Column(Integer, default=0)
    potential_savings = Column(Float, default=0.0, comment="Ahorro potencial en EUR")

    # Acciones tomadas
    was_reviewed = Column(Integer, default=0)  # SQLite doesn't have boolean, use 0/1
    reviewed_at = Column(DateTime, nullable=True)

    @property
    def has_issues(self) -> bool:
        """True si hay lineas con problemas"""
        return (self.lines_warning or 0) > 0 or (self.lines_blocker or 0) > 0


class ReceiptAuditRecord(Base):
    """
    Registro de auditorias de albaranes (Modulo 3: Auditor Post-Recepcion).

    Guarda el resultado de cada auditoria automatica para:
    - Historial de anomalias detectadas
    - Calcular impacto economico acumulado
    - Seguimiento de acciones tomadas
    """

    __tablename__ = "receipt_audit_records"

    id = Column(Integer, primary_key=True, autoincrement=True)

    # Identificacion del albaran
    receipt_id = Column(String(100), index=True)  # ID del albaran en el ERP
    receipt_date = Column(DateTime)
    supplier_code = Column(String(50), default="")
    supplier_name = Column(String(200), default="")
    delivery_note_number = Column(String(100), default="")

    # Resultado de la auditoria
    audited_at = Column(DateTime, default=datetime.utcnow)
    status = Column(String(20), default="pending")  # pending, passed, issues_found, reviewed, actioned
    total_lines = Column(Integer, default=0)
    lines_with_issues = Column(Integer, default=0)
    total_economic_impact = Column(Float, default=0.0, comment="Impacto economico en EUR")

    # Detalle de issues (JSON serializado)
    issues_json = Column(Text, nullable=True, comment="Lista de issues en JSON")

    # Acciones tomadas
    action_taken = Column(String(50), nullable=True)  # accepted, claimed, returned
    actioned_at = Column(DateTime, nullable=True)
    actioned_by = Column(String(100), nullable=True)

    @property
    def has_issues(self) -> bool:
        """True si se detectaron anomalias"""
        return (self.lines_with_issues or 0) > 0
