"""
Modelo para tracking del estado del sistema y progreso de inicialización.
Solución unificada para Issues #12 y #14.
"""

import enum
import uuid

from sqlalchemy import Column, DateTime
from sqlalchemy import Enum as SQLEnum
from sqlalchemy import Integer, Text
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.sql import func

from app.models.base import Base


class SystemComponent(str, enum.Enum):
    """Componentes del sistema que pueden tener estado"""

    CATALOG = "catalog"
    NOMENCLATOR = "nomenclator"
    CIMA = "cima"
    HOMOGENEOUS_GROUPS = "homogeneous_groups"
    DATABASE = "database"
    REDIS = "redis"
    BACKEND = "backend"
    FRONTEND = "frontend"


class SystemStatusEnum(str, enum.Enum):
    """Estados posibles de los componentes del sistema"""

    NOT_INITIALIZED = "not_initialized"
    INITIALIZING = "initializing"
    READY = "ready"
    ERROR = "error"
    WARNING = "warning"
    UPDATING = "updating"


class SystemStatus(Base):
    """
    Modelo para tracking del estado del sistema.
    Usado para mostrar transparencia en la inicialización y estado operacional.
    """

    __tablename__ = "system_status"
    __table_args__ = {"extend_existing": True}

    id = Column(UUID(as_uuid=True), primary_key=True, default=uuid.uuid4)
    component = Column(SQLEnum(SystemComponent), nullable=False, unique=True, index=True)
    status = Column(
        SQLEnum(SystemStatusEnum),
        nullable=False,
        default=SystemStatusEnum.NOT_INITIALIZED,
    )
    progress = Column(Integer, default=0)  # 0-100 porcentaje
    message = Column(Text, nullable=True)  # Mensaje descriptivo actual
    details = Column(Text, nullable=True)  # Detalles adicionales o error

    # Métricas específicas del componente
    total_items = Column(Integer, nullable=True)  # Total de items a procesar
    processed_items = Column(Integer, nullable=True)  # Items procesados
    error_count = Column(Integer, default=0)  # Contador de errores

    # Tiempos
    started_at = Column(DateTime(timezone=True), nullable=True)
    estimated_completion = Column(DateTime(timezone=True), nullable=True)
    last_success_at = Column(DateTime(timezone=True), nullable=True)
    last_error_at = Column(DateTime(timezone=True), nullable=True)

    # Checkpoint para sincronización resiliente
    checkpoint_data = Column(Text, nullable=True)  # JSON con última página procesada, timestamp, etc.
    checkpoint_page = Column(Integer, nullable=True, default=0)  # Última página procesada exitosamente
    checkpoint_timestamp = Column(DateTime(timezone=True), nullable=True)  # Cuándo se guardó el checkpoint

    # Tracking de actualizaciones
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    updated_at = Column(DateTime(timezone=True), onupdate=func.now())

    def to_dict(self):
        """Convertir a diccionario para API responses"""
        return {
            "id": str(self.id),
            "component": self.component.value if self.component else None,
            "status": self.status.value if self.status else None,
            "progress": self.progress,
            "message": self.message,
            "details": self.details,
            "total_items": self.total_items,
            "processed_items": self.processed_items,
            "error_count": self.error_count,
            "started_at": self.started_at.isoformat() if self.started_at else None,
            "estimated_completion": (self.estimated_completion.isoformat() if self.estimated_completion else None),
            "last_success_at": (self.last_success_at.isoformat() if self.last_success_at else None),
            "last_error_at": (self.last_error_at.isoformat() if self.last_error_at else None),
            "checkpoint_page": self.checkpoint_page,
            "checkpoint_timestamp": (self.checkpoint_timestamp.isoformat() if self.checkpoint_timestamp else None),
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
        }

    def __repr__(self):
        return f"<SystemStatus(component={self.component}, status={self.status}, progress={self.progress}%)>"
